<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Model;

use ActiveRecord\Base;
use Application\Core\Entity\TranslationInterface;
use Application\Core\Model\Manager\SiteManager;
use Pongho\Core\Kernel;

/**
 * Modello per le traduzioni.
 *
 * @property string $key
 * @property int    $language_site_id
 * @property string $application
 * @property string $value
 */
class Translation extends Base implements TranslationInterface
{
    /**
     * Nome della tabella.
     *
     * @var string
     */
    public static $table_name = 'translations';

    /**
     * Chiave primaria.
     *
     * @var array
     */
    public static $primary_key = ['key', 'language_site_id'];

    /**
     * Relazioni 'belongs_to'.
     *
     * @var array
     */
    public static $belongs_to = [
        ['language_site', 'model' => LanguageSite::class],
    ];

    /**
     * findAllBySite
     *
     * @param int $site_id
     * @return Translation[]
     */
    public static function findAllBySite($site_id)
    {
        return static::all([
            'select'     => '`from`.*',
            'joins'      => 'LEFT JOIN ' . LanguageSite::tableName() . ' AS ls ON `from`.language_site_id = ls.id',
            'conditions' => ['ls.site_id = ?', $site_id]
        ]);
    }

    /**
     * findAllByKeyAndSite
     *
     * @param string $key
     * @param int    $site_id
     * @return Translation[]
     */
    public static function findAllByKeyAndSite($key, $site_id)
    {
        return static::all([
            'select'     => '`from`.*',
            'joins'      => 'LEFT JOIN ' . LanguageSite::tableName() . ' AS ls ON `from`.language_site_id = ls.id',
            'conditions' => ['`from`.key = :key AND ls.site_id = :site', 'key' => $key, 'site' => $site_id]
        ]);
    }

    /**
     * Restituisce la prima traduzione che trova nel sito per la chiave indicata
     *
     * @param string $key
     * @param int    $site_id
     * @return Translation
     */
    public static function findIfKeyInSite($key, $site_id)
    {
        return static::first([
            'select'     => '`from`.*',
            'joins'      => 'LEFT JOIN ' . LanguageSite::tableName() . ' AS ls ON `from`.language_site_id = ls.id',
            'conditions' => ['`from`.key = :key AND ls.site_id = :site', 'key' => $key, 'site' => $site_id]
        ]);
    }

    /**
     * Restituisce la traduzione per
     *
     * @param string     $key
     * @param string|int $language_id_or_culture
     * @param int        $site_id
     * @return Translation
     */
    public static function findByKeyAndSiteAndLanguage($key, $site_id, $language_id_or_culture)
    {
        $options = [
            'select'     => '`from`.*',
            'joins'      => 'LEFT JOIN ' . LanguageSite::tableName() . ' AS ls ON `from`.language_site_id = ls.id',
            'conditions' => [
                '`from`.key = :key AND ls.site_id = :site',
                'key'  => $key,
                'site' => $site_id,
            ]
        ];

        if (is_numeric($language_id_or_culture)) {
            $options = self::addCondition($options, ['ls.language_id = :lng', 'lng' => $language_id_or_culture]);
        } else {
            $options['joins'] .= ' INNER JOIN ' . Language::tableName() . ' AS l ON l.id = ls.language_id';
            $options = self::addCondition($options, ['l.culture = :lng', 'lng' => $language_id_or_culture]);
        }

        return static::first($options);
    }

    /**
     * findAllByLanguageSite
     *
     * @param int $language_site_id
     * @return Translation[]
     */
    public static function findAllByLanguageSite($language_site_id)
    {
        return static::all([
            'conditions' => ['language_site_id = ?', $language_site_id]
        ]);
    }

    /**
     * @param int $site_id
     * @return Translation[]
     */
    public static function findKeysBySite($site_id)
    {
        $sql = 'SELECT ls.id, t.key, t.application, ls.language_id'
            . ' FROM (SELECT * FROM ' . static::tableName() . ' AS t INNER JOIN ' . LanguageSite::tableName() . ' AS ls ON t.language_site_id = ls.id WHERE ls.site_id = ' . $site_id . ') AS t'
            . ' CROSS JOIN ' . LanguageSite::tableName() . ' AS ls'
            . ' WHERE ls.site_id = ' . $site_id
            . ' GROUP BY ls.id, t.key';

        return static::findBySql($sql);
    }

    /**
     * @param int    $site_id
     * @param string $application
     * @return Translation[]
     */
    public static function findAllBySiteAndApplication($site_id, $application)
    {
        $key_field = static::connection()->quoteFieldName('key');

        return static::all([
                'select'     => '`from`.*',
                'joins'      => 'LEFT JOIN ' . LanguageSite::tableName() . ' AS ls ON `from`.language_site_id = ls.id',
                'conditions' => [
                    'ls.site_id = :site AND `from`.application = :application',
                    'site'        => $site_id,
                    'application' => $application
                ],
                'order'      => $key_field,
            ]);
    }

    /**
     * allByKeysAndSiteAndApplication
     *
     * @param array  $keys
     * @param int    $site_id
     * @param string $application
     * @return Translation[]
     */
    public static function allByKeysAndSiteAndApplication(array $keys, $site_id, $application)
    {
        $keys = array_map(
            function ($value) {
                return "'{$value}'";
            },
            $keys
        );

        return static::all([
                'select'     => '`from`.*',
                'joins'      => 'LEFT JOIN ' . LanguageSite::tableName() . ' AS ls ON `from`.language_site_id = ls.id',
                'conditions' => [
                    '`from`.key IN (' . implode(',',
                        $keys) . ') AND ls.site_id = :site AND `from`.application = :application',
                    'site'        => $site_id,
                    'application' => $application
                ]
            ]);
    }

    /**
     * search
     *
     * @param string $query
     * @param int    $site_id
     * @return Translation[]
     */
    public static function search($query, $site_id)
    {
        return static::all([
                'select'     => '`from`.*',
                'joins'      => 'LEFT JOIN ' . LanguageSite::tableName() . ' AS ls ON `from`.language_site_id = ls.id'
                    . ' LEFT JOIN ' . static::tableName() . ' AS t ON `from`.key = t.key'
                    . ' LEFT JOIN ' . LanguageSite::tableName() . ' AS lsj ON t.language_site_id = lsj.id',
                'conditions' => [
                    '(pongho_like(t.key, :query) OR pongho_like(t.value, :query)) AND ls.site_id = :site AND lsj.site_id = :site',
                    'query' => '%' . $query . '%',
                    'site'  => $site_id
                ]

            ]);
    }

    /**
     * Aggiunge una traduzione per tutte le lingue
     *
     * <code>
     * ::addTranslation('key', 'value')                         Aggiunge una traduzione per le lingue di default di tutti i siti per l'applicazione 'theme'
     * ::addTranslation('key', 'value', 'admin', 1)             Aggiunge una traduzione per la coppia sito/lingua corrispondente a 1 nell'applicazione 'admin'
     * ::addTranslation('key', 'value', 'admin', null, 1)       Aggiunge una traduzione per la lingua 1 di tutti i siti
     * ::addTranslation('key', 'value', 'admin', null, 'it_IT') Aggiunge una traduzione per la lingua 'it_IT' di tutti i siti
     * ::addTranslation('key', 'value', 'admin', null, 1, 1)    Aggiunge una traduzione per la lingua 1 del sito 1
     * ::addTranslation('key', 'value', 'admin', null, null, 1) Aggiunge una traduzione per tutte le lingue del sito 1
     *
     * Specificando $language_site_id non avranno effetto $language_id_or_culture e/o $site_id, ad esempio
     *
     * ::addTranslation('key', 'value', 'admin', 1, null, 1) è considerato uguale a ::addTranslation('key', 'value', 'admin', 1)
     * </code>
     *
     * @param string $key
     * @param string $value
     * @param string $application
     * @param null   $language_site_id
     * @param null   $language_id_or_culture
     * @param null   $site_id
     * @throws \DomainException
     */
    public static function addTranslation($key, $value, $application = 'theme', $language_site_id = null, $language_id_or_culture = null, $site_id = null)
    {
        $options = [];

        if ($language_site_id !== null) {
            $options = LanguageSite::addCondition($options, ['id = :ls', 'ls' => $language_site_id]);
        } else {
            if ($language_id_or_culture !== null) {

                if (is_numeric($language_id_or_culture)) {
                    $options = LanguageSite::addCondition(
                        $options,
                        ['language_id = :lng', 'lng' => $language_id_or_culture]
                    );
                } else {
                    $options['select'] = '`from`.*';
                    $options['joins'] = 'INNER JOIN ' . Language::tableName() . ' AS l ON l.id = `from`.language_id';
                    $options = LanguageSite::addCondition(
                        $options,
                        ['l.culture = :lng', 'lng' => $language_id_or_culture]
                    );
                }
            }

            if ($site_id !== null) {
                $options = LanguageSite::addCondition($options, ['site_id = :s', 's' => $site_id]);
            }
        }

        if (empty($options)) {
            /** @var SiteManager $site_manager */
            $site_manager = Kernel::instance()->getContainer()->get('site_manager');

            // Aggiunge la traduzione nella lingua di default di ciascun sito
            foreach ($site_manager->findAll() as $site) {
                $options = LanguageSite::addCondition($options, [
                    'site_id = :s AND language_id = :l',
                    's' => $site->getId(),
                    'l' => $site->getDefaultLanguageId(),
                ]);

                /** @var LanguageSite $language_site */
                foreach (LanguageSite::all($options) as $language_site) {
                    $translation = Translation::create([
                        'key'              => $key,
                        'language_site_id' => $language_site->id,
                        'application'      => $application,
                        'value'            => $value,
                    ]);

                    if ($translation->isNewRecord()) {
                        throw new \DomainException('The translation "' . $key . '" could not be created. Another translation with the same key may exist');
                    }
                }
            }
        } else {
            // Aggiunge la traduzione secondo i parametri specificati
            /** @var LanguageSite $language_site */
            foreach (LanguageSite::all($options) as $language_site) {
                $translation = Translation::create([
                    'key'              => $key,
                    'language_site_id' => $language_site->id,
                    'application'      => $application,
                    'value'            => $value,
                ]);

                if ($translation->isNewRecord()) {
                    throw new \DomainException('The translation "' . $key . '" could not be created. Another translation with the same key may exist');
                }
            }
        }
    }

    /**
     * Elimina una traduzione in tutte le lingue. Se viene specificato il parametro $site_id elimina la traduzione per tutte le lingue del sito specificato
     *
     * @param string $key
     * @param int    $site_id
     * @throws \RuntimeException
     */
    public static function deleteTranslation($key, $site_id = null)
    {
        $options = [
            'conditions' => [
                '`key` = :key',
                'key' => $key,
            ],
        ];

        if ($site_id !== null) {
            $options['select'] = '`from`.*';
            $options['joins'] = 'INNER JOIN ' . LanguageSite::tableName() . ' AS ls ON ls.id = `from`.language_site_id';
            $options = Translation::addCondition($options, ['ls.site_id = :site', 'site' => $site_id]);
        }

        foreach (Translation::all($options) as $translation) {
            if (!$translation->delete()) {
                throw new \RuntimeException('The translation "' . $key . '" could not be deleted');
            }
        }
    }

    /**
     * Modifica la chiave di una traduzione in tutte le lingue
     *
     * @param string $old_key
     * @param string $new_key
     * @param int    $site_id
     * @throws \RuntimeException
     * @throws \DomainException
     */
    public static function changeKey($old_key, $new_key, $site_id = null)
    {
        $options = [
            'conditions' => [
                '`key` = :key',
                'key' => $old_key,
            ],
        ];

        if ($site_id !== null) {
            $options['select'] = '`from`.*';
            $options['joins'] = 'INNER JOIN ' . LanguageSite::tableName() . ' AS ls ON ls.id = `from`.language_site_id';
            $options = Translation::addCondition($options, ['ls.site_id = :site', 'site' => $site_id]);
        }

        foreach (Translation::all($options) as $translation) {
            // ENGGHHHHHH: Non sembra possibile modificare una chiave primaria
            // Creo una nuova traduzione con gli stessi identici dati della vecchia traduzione, tranne la key che è quella nuova
            $attributes = $translation->attributes();
            $attributes['key'] = $new_key;

            $new_translation = Translation::create($attributes);

            if ($new_translation->isNewRecord()) {
                throw new \DomainException('The key "' . $old_key . '" cannot be changed because another "' . $new_key . '" already exists');
            }

            // Elimino la vecchia traduzione
            if (!$translation->delete()) {
                throw new \RuntimeException('The old translation "' . $old_key . '" could not be deleted');
            }
        }
    }

    /**
     * Modifica l'applicazione di una traduzione per tutte le lingue
     *
     * @param string $key
     * @param string $application
     * @param int    $site_id
     * @throws \RuntimeException
     */
    public static function changeApplication($key, $application, $site_id = null)
    {
        $options = [
            'conditions' => [
                '`key` = :key',
                'key' => $key,
            ],
        ];

        if ($site_id !== null) {
            $options['select'] = '`from`.*';
            $options['joins'] = 'INNER JOIN ' . LanguageSite::tableName() . ' AS ls ON ls.id = `from`.language_site_id';
            $options = Translation::addCondition($options, ['ls.site_id = :site', 'site' => $site_id]);
        }

        /** @var Translation $translation */
        foreach (Translation::all($options) as $translation) {
            $translation->application = $application;

            if (!$translation->save()) {
                throw new \RuntimeException('The application for the translation "' . $translation->key . '" could not be changed');
            }
        }
    }

    /**
     * Modifica una traduzione
     *
     * <code>
     * ::editTranslation('key', 'value translated', 1)                  Modifica la traduzione per la coppia lingua/sito corrispondente a 1 (es italiano del sito 1)
     * ::editTranslation('key', 'value translated', null, 1)            Modifica la traduzione per la lingua 1 di tutti i siti (es italiano)
     * ::editTranslation('key', 'value translated', null, 'it_IT')      Modifica la traduzione per la lingua 'it_IT' di tutti i siti
     * ::editTranslation('key', 'value translated', null, 1, 1)         Modifica la traduzione per la lingua 1 del sito 1 (equivale al primo caso)
     * ::editTranslation('key', 'value translated')                     Utilizzo errato, è necessario specificare la lingua in uno dei due modi consentiti
     * ::editTranslation('key', 'value translated', null, null)         Utilizzo errato, è necessario specificare la lingua in uno dei due modi consentiti
     * </code>
     *
     * @param string     $key
     * @param string     $value
     * @param int        $language_site_id
     * @param int|string $language_id_or_culture
     * @param int        $site_id
     * @throws \RuntimeException
     * @throws \BadMethodCallException
     */
    public static function editTranslation($key, $value, $language_site_id = null, $language_id_or_culture = null, $site_id = null)
    {
        if ($language_site_id !== null) {
            $options = [
                'conditions' => ['`key` = :key AND language_site_id = :ls', 'key' => $key, 'ls' => $language_site_id]
            ];
        } else {
            if ($language_id_or_culture === null) {
                throw new \BadMethodCallException('The language of the translation you need to edit is required');
            }

            $options = [
                'select'     => '`from`.*',
                'joins'      => 'INNER JOIN ' . LanguageSite::tableName() . ' AS ls ON ls.id = `from`.language_site_id',
                'conditions' => ['`key` = :key', 'key' => $key],
            ];

            if (is_numeric($language_id_or_culture)) {
                $options = Translation::addCondition(
                    $options,
                    ['language_id = :lng', 'lng' => $language_id_or_culture]
                );
            } else {
                $options['joins'] .= ' INNER JOIN ' . Language::tableName() . ' AS l ON l.id = ls.language_id';
                $options = Translation::addCondition(
                    $options,
                    ['l.culture = :lng', 'lng' => $language_id_or_culture]
                );
            }

            if ($site_id !== null) {
                $options = Translation::addCondition($options, ['ls.site_id = :s', 's' => $site_id]);
            }
        }

        // Aggiunge la traduzione secondo i parametri specificati
        /** @var Translation $translation */
        foreach (Translation::all($options) as $translation) {
            $translation->value = $value;

            if (!$translation->save()) {
                throw new \RuntimeException('The translation "' . $translation->key . '" could not be updated');
            }
        }
    }

    /**
     * Crea o modifica una traduzione
     *
     * Se non viene specificato il sito, viene eseguita per tutti i siti
     *
     * @param string $key
     * @param string $value
     * @param string $application
     * @param string $language_culture
     * @param int    $site_id
     */
    public static function putTranslation($key, $value, $application, $language_culture, $site_id = null)
    {
        if ($site_id === null) {
            /** @var SiteManager $site_manager */
            $site_manager = Kernel::instance()->getContainer()->get('site_manager');

            foreach ($site_manager->findAll() as $site) {
                static::translationCreateOrEdit($key, $value, $application, $language_culture, $site->getId());
            }
        } else {
            static::translationCreateOrEdit($key, $value, $application, $language_culture, $site_id);
        }
    }

    /**
     * Metodo di supporto per putTranslation()
     *
     * @param string $key
     * @param string $value
     * @param string $application
     * @param string $language_culture
     * @param int    $site_id
     */
    protected static function translationCreateOrEdit($key, $value, $application, $language_culture, $site_id)
    {
        if (Translation::findByKeyAndSiteAndLanguage($key, $site_id, $language_culture)) {
            static::editTranslation($key, $value, null, $language_culture, $site_id);
        } else {
            static::addTranslation($key, $value, $application, null, $language_culture, $site_id);
        }
    }
}
