<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Model;

use ActiveRecord\Base;
use Application\Core\Entity\PermitInterface;
use Pongho\Core\Kernel;

/**
 * Modello per i permessi.
 *
 * @property int    $id
 * @property string $key
 * @property bool   $is_enabled
 */
class Permit extends Base implements PermitInterface
{
    /**
     * Nome della tabella.
     *
     * @var string
     */
    public static $table_name = 'permits';

    /**
     * Relazioni 'has_and_belongs_to_many'.
     *
     * @var array
     */
    public static $has_and_belongs_to_many = [
        ['roles']
    ];

    /**
     * Restituisce il modello del permesso in base alla chiave.
     *
     * @param string $key
     * @return Permit
     */
    public static function findByKey($key)
    {
        return static::first([
            'conditions' => ['`key` = ?', $key]
        ]);
    }

    /**
     * Restituisce i modelli dei permessi in base ad un prefisso.
     *
     * @param string $prefix
     * @return Permit[]
     */
    public static function findByPrefix($prefix)
    {
        return static::all([
            'conditions' => ['pongho_like(`key`, ?)', $prefix . '_%)']
        ]);
    }

    /**
     * @todo
     */
    public static function findBySuffix($suffix)
    {
    }

    /**
     * Cerca un permesso o lo crea.
     *
     * Il parametro $is_enabled viene utilizzato solo per la creazione.
     *
     * @param string $key
     * @param bool   $is_enabled
     * @return Permit
     */
    public static function findOrCreate($key, $is_enabled = true)
    {
        $key = (string) $key;
        $is_enabled = (bool) $is_enabled;

        $permit = Permit::findByKey($key);

        if (!$permit) {
            $permit = Permit::create([
                'key'        => $key,
                'is_enabled' => $is_enabled,
            ]);
        }

        return $permit;
    }

    /**
     * Abilita una lista di permessi in base alla loro chiave.
     *
     * Nel caso il permesso non esista questo viene creato.
     *
     * @param array $keys
     */
    public static function enableKeys(array $keys)
    {
        foreach ($keys as $key) {
            if (($permit = static::findByKey($key)) === null) {
                static::create([
                    'key'        => $key,
                    'is_enabled' => true,
                ]);
            } else {
                $permit->is_enabled = true;
                $permit->save(true);
            }
        }
    }

    /**
     * Disabilita una lista di permessi in base alla loro chiave.
     *
     * Nel caso il permesso non esista questo viene creato.
     *
     * @param array $keys
     */
    public static function disableKeys(array $keys)
    {
        foreach ($keys as $key) {
            if (($permit = static::findByKey($key)) !== null) {
                $permit->is_enabled = false;
                $permit->save(true);
            }
        }
    }

    /**
     * Indica se un permesso è abilitato o meno.
     *
     * @param string $key
     * @return bool
     */
    public static function isEnabled($key)
    {
        $permit = static::findByKey($key);

        if ($permit === null) {
            return false;
        }

        return $permit->is_enabled;
    }

    /**
     * @deprecated
     */
    public function name()
    {
        /** @var \Application\Core\Localization $lang */
        $lang = Kernel::instance()->getContainer()->get('language');

        return $lang->get('permit_' . $this->key);
    }

    /**
     * {@inheritdoc}
     */
    public function delete($in_recursion = false)
    {
        if ($this->deleteRolesRelations() && parent::delete()) {
            return true;
        }

        return false;
    }

    /**
     * Elimina le associazioni con i ruoli.
     *
     * @return bool
     */
    public function deleteRolesRelations()
    {
        $permit_roles = PermitRole::all([
            'conditions' => ['permit_id = :permit', 'permit' => $this->id]
        ]);

        foreach ($permit_roles as $permit_role) {
            if (!$permit_role->delete()) {
                return false;
            }
        }

        return true;
    }
}
