<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Model;

use ActiveRecord\Base;
use Application\Admin\Form\TranslatableInterface;
use Application\Core\Entity\NotificationTemplateInterface;

/**
 * NotificationTemplate
 *
 * @property int $id
 * @property int $site_id
 * @property string $notification
 * @property string $sender_to_admin
 * @property string $recipient_to_admin
 * @property string $reply_to_admin
 * @property string $sender_to_user
 * @property string $recipient_to_user
 * @property string $reply_to_user
 * @property string $subject_to_admin
 * @property string $content_to_admin
 *
 * Ereditati tramite join
 * @property string $subject_to_user
 * @property string $content_to_user
 */
class NotificationTemplate extends Base implements NotificationTemplateInterface, TranslatableInterface
{
    /**
     * Nome della tabella.
     *
     * @var string
     */
    public static $table_name = 'notifications_templates';

    /**
     * @var NotificationTemplateTranslation[]
     */
    protected $translations;

    /**
     * {@inheritdoc}
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * {@inheritdoc}
     */
    public function getNotificationName()
    {
        return $this->notification;
    }

    /**
     * {@inheritdoc}
     */
    public function getSenderToAdmin()
    {
        return $this->sender_to_admin;
    }

    /**
     * {@inheritdoc}
     */
    public function getRecipientToAdmin()
    {
        return $this->recipient_to_admin;
    }

    /**
     * {@inheritdoc}
     */
    public function getReplyToAdmin()
    {
        return $this->reply_to_admin;
    }

    /**
     * {@inheritdoc}
     */
    public function getSubjectToAdmin()
    {
        return $this->subject_to_admin;
    }

    /**
     * {@inheritdoc}
     */
    public function getContentToAdmin()
    {
        return $this->content_to_admin;
    }

    /**
     * {@inheritdoc}
     */
    public function getSenderToUser()
    {
        return $this->sender_to_user;
    }

    /**
     * {@inheritdoc}
     */
    public function getRecipientToUser()
    {
        return $this->recipient_to_user;
    }

    /**
     * {@inheritdoc}
     */
    public function getReplyToUser()
    {
        return $this->reply_to_user;
    }

    /**
     * {@inheritdoc}
     */
    public function getSubjectToUser($language_id)
    {
        return $this->getTranslation($language_id)->subject_to_user;
    }

    /**
     * {@inheritdoc}
     */
    public function getContentToUser($language_id)
    {
        return $this->getTranslation($language_id)->content_to_user;
    }

    /**
     * {@inheritdoc}
     */
    public function getTranslations($only_enabled = true)
    {
        if ($this->translations === null) {
            $options = [
                'conditions' => [
                    'template_id = :template',
                    'template' => $this->getId(),
                ],
            ];

            $this->translations = [];

            /** @var NotificationTemplateTranslation $translation */
            foreach (NotificationTemplateTranslation::all($options) as $translation) {
                $this->translations[$translation->language_id] = $translation;
            }
        }

        return $this->translations;
    }

    /**
     * {@inheritdoc}
     */
    public function getTranslation($language_id)
    {
        if ($this->hasTranslation($language_id)) {
            return $this->translations[$language_id];
        }

        return null;
    }

    /**
     * Indica se è presente la traduzione in una determinata lingua
     *
     * @param int $language_id
     * @return bool
     */
    public function hasTranslation($language_id)
    {
        $this->getTranslations();

        return array_key_exists($language_id, $this->translations);
    }

    /**
     * {@inheritdoc}
     */
    public function setTranslation($language_id, array $attributes)
    {
        if ($this->isNewRecord()) {
            $this->save();
        }

        $attributes['language_id'] = $language_id;
        $attributes['template_id'] = $this->getId();

        if ($translation = $this->getTranslation($language_id)) {
            $translation->updateAttributes($attributes);
        } else {
            $translation = new NotificationTemplateTranslation($attributes);
        }

        $this->translations[$language_id] = $translation;

        if ($translation->save()) {
            // Invalido la cache
            $this->translations = null;

            return true;
        }

        return false;
    }

    /**
     * {@inheritdoc}
     */
    public function getEntityType()
    {
        return 'template';
    }

    /**
     * {@inheritdoc}
     */
    public function prepareTranslation($language_id, array $attributes)
    {
        $this->translations[$language_id] = new NotificationTemplateTranslation($attributes);
    }

    /**
     * {@inheritdoc}
     */
    public function prepareTranslations(array $translations)
    {
        foreach ($translations as $language_id => $attributes) {
            $this->translations[$language_id] = new NotificationTemplateTranslation($attributes);
        }
    }

    /**
     * {@inheritdoc}
     */
    public function delete()
    {
        return $this->deleteTranslations() && parent::delete();
    }

    /**
     * @return bool
     */
    protected function deleteTranslations()
    {
        foreach ($this->getTranslations() as $translation) {
            if (!$translation->delete()) {
                return false;
            }
        }

        return true;
    }
}
