<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Model\Manager;

use Application\Core\Entity\SiteInterface;
use Application\Core\Model\Site;

/**
 * Class SiteManager
 */
class SiteManager extends AbstractCoreManager implements SiteManagerInterface
{
    /**
     * @var string
     */
    static $model = 'Site';

    /**
     * {@inheritdoc}
     */
    public function findById($site_id)
    {
        $this->prepare();

        $model = $this->getCache()->get(static::$model, $site_id);

        if (!$model) {
            throw new \InvalidArgumentException(sprintf('The site "%s" is undefined', $site_id));
        }

        return $model;
    }

    /**
     * {@inheritdoc}
     */
    public function findAll()
    {
        $this->prepare();

        return $this->getCache()->all(static::$model);
    }

    /**
     * {@inheritdoc}
     */
    public function getMainSite()
    {
        return $this->findById(1);
    }

    /**
     * {@inheritdoc}
     */
    public function create($name, $domain, $default_language, $images_url, $uploads_url, $themes_url, $content_path = '/content', $images_path = '/content/images', $uploads_path = '/content/uploads', $themes_path = '/content/themes', array $settings = array())
    {
        if (is_numeric($default_language)) {
            $default_language_id = intval($default_language);
        } else {
            $default_language_id = $default_language->getId();
        }

        $model = new Site();

        $model->name = $name;
        $model->domain = $domain;
        $model->default_language_id = $default_language_id;
        $model->content_path = $content_path;
        $model->images_path = $images_path;
        $model->images_url = $images_url;
        $model->uploads_path = $uploads_path;
        $model->uploads_url = $uploads_url;
        $model->themes_path = $themes_path;
        $model->themes_url = $themes_url;
        $model->setDefaultOptions($settings);

        return $model;
    }

    /**
     * @param SiteInterface|Site $site
     * {@inheritdoc}
     */
    public function save(SiteInterface $site)
    {
        if (!$site->save()) {
            throw new \Exception();
        }

        $this->getCache()->clear();

        return $this;
    }

    /**
     * Precarica i modelli per la cache interna
     */
    protected function prepare()
    {
        if (!$this->getCache()->isEmpty(static::$model)) {
            return;
        }

        /** @var Site $model */
        foreach (Site::all() as $model) {
            $this->getCache()->set(static::$model, $model->getId(), $model);
        }
    }
}
