<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Model\Manager;

use Application\Core\Entity\ModuleSiteInterface;
use Application\Core\Model\ModuleSite;

/**
 * Class ModuleSiteManager
 */
class ModuleSiteManager extends AbstractCoreManager implements ModuleSiteManagerInterface
{
    /**
     * @var string
     */
    static $model = 'ModuleSite';

    /**
     * {@inheritdoc}
     */
    public function findAll()
    {
        $this->prepare();

        return $this->getCache()->all(static::$model);
    }

    /**
     * {@inheritdoc}
     */
    public function findById($module_site_id)
    {
        $this->prepare();

        $model = $this->getCache()->get(static::$model, $module_site_id);

        return $model;
    }

    /**
     * {@inheritdoc}
     */
    public function findAllBySite($site_id)
    {
        return array_filter(
            $this->findAll(),
            function(ModuleSiteInterface $model) use ($site_id) {
                if ($model->getSiteId() !== $site_id) {
                    return false;
                }

                return true;
            }
        );
    }

    /**
     * {@inheritdoc}
     */
    public function findAllByModule($module_id)
    {
        return array_filter(
            $this->findAll(),
            function(ModuleSiteInterface $model) use ($module_id) {
                if ($model->getModuleId() !== $module_id) {
                    return false;
                }

                return true;
            }
        );
    }

    /**
     * {@inheritdoc}
     */
    public function findBySiteAndModule($site_id, $module_id)
    {
        $model = array_filter(
            $this->findAll(),
            function(ModuleSiteInterface $model) use ($site_id, $module_id) {
                if ($model->getSiteId() !== $site_id) {
                    return false;
                }

                if ($model->getModuleId() !== $module_id) {
                    return false;
                }

                return true;
            }
        );

        if (empty($model)) {
            return null;
        }

        return reset($model);
    }

    /**
     * {@inheritdoc}
     */
    public function create($module, $site, $is_enabled, array $settings = array())
    {
        if (is_numeric($module)) {
            $module_id = intval($module);
        } else {
            $module_id = $module->getId();
        }

        if (is_numeric($site)) {
            $site_id = intval($site);
        } else {
            $site_id = $site->getId();
        }

        $model = new ModuleSite();

        $model->module_id = $module_id;
        $model->site_id = $site_id;
        $model->is_enabled = $is_enabled;
        $model->setDefaultOptions($settings);

        return $model;
    }

    /**
     * @param ModuleSiteInterface|ModuleSite $module_site
     * {@inheritdoc}
     */
    public function save(ModuleSiteInterface $module_site)
    {
        if (!$module_site->save()) {
            throw new \Exception();
        }

        $this->getCache()->clear();

        return $this;
    }

    /**
     * Precarica i modelli per la cache interna
     */
    protected function prepare()
    {
        if (!$this->getCache()->isEmpty(static::$model)) {
            return;
        }

        /** @var ModuleSiteInterface $model */
        foreach (ModuleSite::all() as $model) {
            $this->getCache()->set(static::$model, $model->getId(), $model);
        }
    }
}
