<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Model;

use ActiveRecord\Base;
use Application\Core\Entity\LanguageSiteInterface;
use Application\Core\Entity\ModuleInterface;
use Application\Core\Model\Manager\LanguageManager;
use Application\Core\Model\Manager\LanguageModuleManager;
use Application\Core\Model\Manager\SiteManager;
use Pongho\Core\Kernel;

/**
 * Modello per le relazioni tra lingue e siti.
 *
 * @property int      $id
 * @property int      $language_id
 * @property int      $site_id
 * @property string   $path
 * @property bool     $is_enabled
 * @property Language $language
 * @property Site     $site
 */
class LanguageSite extends Base implements LanguageSiteInterface
{
    /**
     * Nome della tabella.
     *
     * @var string
     */
    public static $table_name = 'languages_sites';

    /**
     * @var array
     */
    public static $belongs_to = [
        ['language', 'model' => Language::class],
        ['site', 'model' => Site::class],
    ];

    /**
     * @return array
     */
    public static function getDefaultModelOptions()
    {
        return [
            'select'     => 'l.*, `from`.*',
            'joins'      => 'INNER JOIN ' . Language::tableName() . ' AS l ON l.id = `from`.language_id',
        ];
    }

    /**
     * Ricerca in base a `language_id` e `site_id`.
     *
     * @param int $language_id
     * @param int $site_id
     * @return LanguageSite
     */
    public static function findByLanguageAndSite($language_id, $site_id)
    {
        $options = self::addCondition(
            static::getDefaultModelOptions(),
            [
                '`from`.language_id = :language AND `from`.site_id = :site',
                'language' => $language_id,
                'site'     => $site_id
            ]
        );

        return self::first($options);
    }

    /**
     * Ricerca tutti.
     *
     * @return LanguageSite[]
     */
    public static function findAll()
    {
        return self::all(static::getDefaultModelOptions());
    }

    /**
     * Ricerca tutti in base a `site_id`.
     *
     * @param int $site_id
     * @return LanguageSite[]
     */
    public static function findAllBySite($site_id)
    {
        $options = self::addCondition(static::getDefaultModelOptions(), [
            '`from`.site_id = :site',
            'site' => $site_id,
        ]);

        return self::all($options);
    }

    /**
     * Ricerca in base a `language_id`, `site_id` e `module_id`.
     *
     * È necessario anche `$module_id` per conoscere il nome ed il percorso del modulo.
     *
     * @param int $language_id
     * @param int $site_id
     * @param int $module_id
     * @return LanguageSite
     *
     * @deprecated
     */
    public static function findById($language_id, $site_id, $module_id)
    {
        $options = static::getDefaultModelOptions();
        $options['select'] = 'l.*, lm.*, `from`.*';
        $options['joins'] .= ' INNER JOIN ' . LanguageModule::tableName() . ' AS lm ON lm.language_site_id = `from`.id';

        $options = self::addCondition($options, [
            '`from`.language_id = :language AND `from`.site_id = :site AND lm.module_id = :module',
            'language' => $language_id,
            'site'     => $site_id,
            'module'   => $module_id
        ]);

        return self::first($options);
    }

    /**
     * {@inheritdoc}
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * {@inheritdoc}
     */
    public function getPath()
    {
        return $this->path;
    }

    /**
     * {@inheritdoc}
     */
    public function getLanguageId()
    {
        return $this->language_id;
    }

    /**
     * {@inheritdoc}
     */
    public function getSiteId()
    {
        return $this->site_id;
    }

    /**
     * {@inheritdoc}
     */
    public function getLanguage()
    {
        /** @var LanguageManager $language_manager */
        $language_manager = Kernel::instance()->getContainer()->get('language_manager');

        return $language_manager->findById($this->getLanguageId());
    }

    /**
     * {@inheritdoc}
     */
    public function getSite()
    {
        /** @var SiteManager $language_manager */
        $site_manager = Kernel::instance()->getContainer()->get('site_manager');

        return $site_manager->findById($this->getSiteId());
    }

    /**
     * {@inheritdoc}
     */
    public function enable()
    {
        $this->is_enabled = true;

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function disable()
    {
        $this->is_enabled = false;

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function isEnabled()
    {
        return $this->is_enabled;
    }

    /**
     * Restituisce le traduzioni.
     *
     * @return Translation[]
     */
    public function getTranslations()
    {
        return Translation::findAllByLanguageSite($this->id);
    }

    /**
     * @deprecated Usare getLanguage()->getCulture()
     */
    public function getCulture()
    {
        if (!$this->attributePresent('culture')) {
            throw new \LogicException("Impossibile determinare la cultura perché manca l'argomento culture: hai aggiunto la JOIN con la tabella delle lingue?.");
        }

        return $this->culture;
    }

    /**
     * @deprecated Usare getLanguage()->getIso()
     */
    public function getIso()
    {
        if (!$this->attributePresent('iso')) {
            throw new \LogicException("Impossibile determinare il codice iso perché manca l'argomento iso: hai aggiunto la JOIN con la tabella delle lingue?.");
        }

        return $this->iso;
    }

    /**
     * @deprecated Usare getLanguage()->getName()
     */
    public function getName()
    {
        if (!$this->attributePresent('name')) {
            throw new \LogicException("Impossibile determinare il nome perché manca l'argomento name: hai aggiunto la JOIN con la tabella delle lingue?.");
        }

        return $this->name;
    }

    /**
     * @deprecated Usare getLanguage()->getLocale()
     */
    public function getLocale()
    {
        if (!$this->attributePresent('locale')) {
            throw new \LogicException("Impossibile determinare il locale perché manca l'argomento locale: hai aggiunto la JOIN con la tabella delle lingue?.");
        }

        return $this->locale;
    }

    /**
     * @deprecated Usare getLanguage()->getTimezone()
     */
    public function getTimezone()
    {
        if (!$this->attributePresent('timezone')) {
            throw new \LogicException("Impossibile determinare la timezone perché manca l'argomento timezone: hai aggiunto la JOIN con la tabella delle lingue?.");
        }

        return $this->timezone;
    }

    /**
     * {@inheritdoc}
     */
    public function getLanguageModule($module)
    {
        if (is_numeric($module)) {
            $module_id = intval($module);
        } elseif ($module instanceof ModuleInterface) {
            $module_id = $module->getId();
        } else {
            throw new \InvalidArgumentException(
                sprintf(
                    'Module must be an integer or an instance of ModuleInterface. "%s" given instead.',
                    var_to_string($module)
                )
            );
        }

        /** @var LanguageModuleManager $language_module_manager */
        $language_module_manager = Kernel::instance()->getContainer()->get('language_module_manager');

        return $language_module_manager->findByLanguageSiteAndModule($this->id, $module_id);
    }

    /**
     * {@inheritdoc}
     */
    public function getLanguageModules()
    {
        /** @var LanguageModuleManager $language_module_manager */
        $language_module_manager = Kernel::instance()->getContainer()->get('language_module_manager');

        return $language_module_manager->findAllByLanguageSite($this->id);
    }
}
