<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Model;

use ActiveRecord\Base;
use Application\Core\Entity\GroupInterface;

/**
 * Modello per i gruppi.
 *
 * @property int    $id
 * @property string $name
 * @property string $description
 */
class Group extends Base implements GroupInterface
{
    /**
     * Nome della tabella.
     *
     * @var string
     */
    public static $table_name = 'groups';

    /**
     * @var GroupUser[]
     */
    protected $users;

    /**
     * {@inheritdoc}
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * {@inheritdoc}
     */
    public function getName()
    {
        return $this->name;
    }

    /**
     * {@inheritdoc}
     */
    public function getDescription()
    {
        return $this->description;
    }

    /**
     * Restituisce le relazioni del gruppo con gli utenti.
     *
     * @return GroupUser[]
     */
    public function getUsers()
    {
        if ($this->users === null) {
            $this->users = GroupUser::all(['conditions' => ['group_id = :group', 'group' => $this->id]]);
        }

        return $this->users;
    }

    /**
     * Controlla se l'utente è già presente nel gruppo.
     *
     * @param int $user_id
     * @return bool
     */
    public function hasUser($user_id)
    {
        return (($group_user = GroupUser::find($this->id, intval($user_id))) !== null);
    }

    /**
     * Aggiunge un utente al gruppo.
     *
     * @param int $user_id
     */
    public function addUser($user_id)
    {
        if (!$this->hasUser($user_id)) {
            GroupUser::create(['group_id' => $this->id, 'user_id' => $user_id]);
        }
    }

    /**
     * Restituisce un array di `Account::id`.
     *
     * @param mixed $accounts Account, array di Account, array di id
     * @return array
     */
    protected function parseUsers($accounts)
    {
        if (!is_array($accounts)) {
            $accounts = [$accounts];
        }

        foreach ($accounts as &$account) {
            if ($account instanceof Account) {
                $account = $account->id;
            }
        }

        return $accounts;
    }

    /**
     * Aggiunge più utenti al gruppo.
     *
     * @param mixed $accounts Account, array di Account, array di id
     */
    public function addUsers($accounts)
    {
        foreach ($this->parseUsers($accounts) as $user_id) {
            $this->addUser($user_id);
        }
    }

    /**
     * Rimuove la relazione con l’utente.
     *
     * @param int $user_id
     */
    public function delUser($user_id)
    {
        if (($group_user = GroupUser::find($this->id, intval($user_id))) !== null) {
            $group_user->delete();
        }
    }

    /**
     * Rimuove le relazioni di più utenti.
     *
     * @param mixed $accounts Account, array di Account, array di id
     */
    public function delUsers($accounts)
    {
        foreach ($this->parseUsers($accounts) as $user_id) {
            $this->delUser($user_id);
        }
    }

    /**
     * Sostituisce gli utenti di un gruppo.
     *
     * @param mixed $accounts Account, array di Account, array di id
     */
    public function setUsers($accounts)
    {
        if ($this->emptyGroup()) {
            $this->addUsers($accounts);
        }
    }

    /**
     * Svuota il gruppo.
     *
     * @return bool
     */
    public function emptyGroup()
    {
        $options = ['conditions' => ['group_id = ?', $this->id]];

        foreach (GroupUser::all($options) as $group) {
            if (!$group->delete()) {
                return false;
            }
        }

        return true;
    }

    /**
     * {@inheritdoc}
     */
    public function delete()
    {
        if ($this->emptyGroup() && parent::delete()) {
            return true;
        }

        return false;
    }
}
