<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Model;

use ActiveRecord\Base;
use Application\Core\Entity\ApplicationInterface;

/**
 * Application.
 *
 * @property integer           $id
 * @property string            $app_name
 * @property integer           $load_order
 * @property integer           $total_installations
 * @property ApplicationSite[] $installations
 */
class Application extends Base implements ApplicationInterface
{
    /**
     * Nome della tabella.
     *
     * @var string
     */
    public static $table_name = 'apps';

    /**
     * Relazioni 'has_many'.
     *
     * @var array
     */
    public static $has_many = [
        ['installations', 'model' => ApplicationSite::class, 'foreign_key' => 'app_id'],
    ];

    /**
     * Relazioni 'count_has_many'.
     *
     * @var array
     */
    public static $count_has_many = [
        ['total_installations', 'model' => ApplicationSite::class, 'foreign_key' => 'app_id'],
    ];

    /**
     * @param int $site_id
     * @return Application[]
     */
    public static function findAllEnabled($site_id)
    {
        return self::all([
            'select'     => '`from`.*',
            'joins'      => 'INNER JOIN ' . ApplicationSite::tableName() . ' AS appsite ON `from`.id = appsite.app_id',
            'conditions' => [
                'appsite.site_id = :site',
                'site' => $site_id,
            ],
            'order'      => 'app_name',
        ]);
    }

    /**
     * @param string $name
     * @return Base
     */
    public static function findByName($name)
    {
        return self::first([
            'conditions' => ['app_name = ?', $name],
        ]);
    }

    /**
     * {@inheritdoc}
     */
    public function delete()
    {
        foreach ($this->installations as $app_site) {
            if (!$app_site->delete()) {
                return false;
            }
        }

        return parent::delete();
    }

    /**
     * {@inheritdoc}
     */
    public function getName()
    {
        return $this->app_name;
    }

    /**
     * {@inheritdoc}
     */
    public function getVersion()
    {
        // Creo il nome del kernel
        /** @var \Pongho\Core\Plugin $kernel_class */
        $kernel_class = "Application\\{$this->app_name}\\Kernel";

        if (!class_exists($kernel_class)) {
            return 'Kernel not exists';
        }

        return $kernel_class::VERSION;
    }
}
