<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Model;

use ActiveRecord\Base;
use Application\Core\I18n\Address\AddressInterface;
use Pongho\Core\Kernel;

/**
 * Modello per gli indirizzi.
 *
 * @property int      $id
 * @property int      $user_id
 * @property string   $name
 * @property string   $surname
 * @property string   $address1
 * @property string   $address2
 * @property string   $city
 * @property string   $postcode
 * @property int      $province_id
 * @property int      $country_id
 * @property string   $telephone
 * @property Account  $user
 * @property Province $province
 * @property Country  $country
 *
 * @property string   $province_name
 */
class Address extends Base implements AddressInterface
{
    /**
     * Nome della tabella.
     *
     * @var string
     */
    public static $table_name = 'addresses';

    /**
     * Relazioni 'belongs_to'.
     *
     * @var array
     */
    public static $belongs_to = [
        ['user', 'model' => Account::class],
        ['province', 'model' => Province::class],
        ['country', 'model' => Country::class],
    ];

    /**
     * Elenco degli attributi che compongono un indirizzo.
     *
     * @var array
     *
     * @deprecated
     */
    public static $ATTRIBUTES = [
        'name',
        'surname',
        'address1',
        'address2',
        'city',
        'postcode',
        'province_id',
        'country_id',
        'telephone'
    ];

    /**
     * __toString()
     *
     * Permette di stampare l’indirizzo completo con un semplice `echo`.
     *
     * <code>
     *   echo Address::find(123);
     * </code>
     *
     * @return string
     */
    public function __toString()
    {
        return $this->render();
    }

    /**
     * {@inheritdoc}
     */
    public function getFirstName()
    {
        return $this->name;
    }

    /**
     * {@inheritdoc}
     */
    public function getLastName()
    {
        return $this->surname;
    }

    /**
     * {@inheritdoc}
     */
    public function getCompany()
    {
        return '';
    }

    /**
     * {@inheritdoc}
     */
    public function getStreet()
    {
        return $this->address1;
    }

    /**
     * {@inheritdoc}
     */
    public function getAdditionalInfo()
    {
        return $this->address2;
    }

    /**
     * {@inheritdoc}
     */
    public function getCity()
    {
        return $this->city;
    }

    /**
     * {@inheritdoc}
     */
    public function getPostCode()
    {
        return $this->postcode;
    }

    /**
     * {@inheritdoc}
     */
    public function getProvince()
    {
        return $this->province;
    }

    /**
     * Restituisce il nome della provincia.
     *
     * Questo metodo è necessario per risolvere un problema di retrocompatibilità.
     *
     * @return string
     */
    public function getProvinceName()
    {
        if ($this->province_id === null) {
            return $this->province_name;
        }

        return $this->province->name;
    }

    /**
     * {@inheritdoc}
     */
    public function getCountry()
    {
        return $this->country;
    }

    /**
     * {@inheritdoc}
     */
    public function getTelephone()
    {
        return $this->telephone;
    }

    /**
     * Restituisce l’indirizzo nel formato indicato dal paese di appartenenza.
     *
     * @return string
     */
    public function render()
    {
        /** @var \Application\Core\I18n\Address\AddressRendererFactory $factory */
        $factory = Kernel::instance()->getContainer()->get('address_renderer_factory');
        $countryCode = $this->country ? $this->country->code : 'IT';

        return $factory->create($countryCode)->render($this);
    }

    /**
     * {@inheritdoc}
     */
    public function validate()
    {
        /** @var \Application\Core\I18n\Address\AddressValidatorFactory $factory */
        $factory = Kernel::instance()->getContainer()->get('address_validator_factory');
        $countryCode = $this->country ? $this->country->code : 'IT';

        $errors = [];
        $factory->create($countryCode)->validate($this, $errors);

        foreach ($errors as $field => $error_message) {
            if ($field === 'province') {
                $field = 'province_id';
            } elseif ($field === 'country') {
                $field = 'country_id';
            }

            $this->errors->add($field, $error_message);
        }

        return $this->errors->is_empty();
    }
}
