<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Model;

use ActiveRecord\Base;
use Application\Core\Entity\AccountInterface;
use Application\Core\Entity\CountryInterface;
use Application\Core\Entity\ProvinceInterface;
use Application\Core\Entity\UserDetailsInterface;
use Application\Core\I18n\Address\Address as I18nAddress;
use Application\Core\Utilities\Validator;
use Application\Showcase\Model\Order;
use Pongho\Core\Kernel;
use Pongho\EventDispatcher\Event;
use Pongho\Utilities\DateTime;

/**
 * Modello per gli account.
 *
 * @property int       $id
 * @property int       $language_id
 * @property int       $role_id
 * @property string    $username
 * @property string    $email
 * @property string    $password
 * @property int       $avatar_id
 * @property string    $description
 * @property string    $activation_key
 * @property string    $reset_password_key
 * @property string    $fullname
 * @property bool      $is_active
 * @property bool      $is_founder
 * @property bool      $is_business
 * @property DateTime  $created_at
 * @property DateTime  $updated_at
 * @property DateTime  $last_login_at
 * @property DateTime  $expire_at
 * @property string    $name
 * @property string    $surname
 * @property string    $company
 * @property string    $partita_iva
 * @property string    $codice_fiscale
 * @property string    $address
 * @property string    $address2
 * @property string    $city
 * @property string    $postcode
 * @property int       $province_id
 * @property int       $country_id
 * @property string    $telephone
 * @property string    $fax
 * @property string    $website
 * @property string    $googleplus
 * @property bool      $newsletter
 * @property string    $newsletter_activation_key
 * @property string    $url
 * @property string    $old_url
 * @property int       $shipping_address_id
 *
 * @property int       $total_orders
 *
 * @property Language  $language
 * @property Role      $role
 * @property Province  $province
 * @property Country   $country
 * @property Group[]   $groups
 * @property Address[] $addresses
 * @property Address   $shipping_address
 */
class Account extends Base implements AccountInterface, UserDetailsInterface
{
    /**
     * Identificativo dell’utente anonimo.
     */
    const ANONYMOUS = 1;

    /**
     * Nome della tabella.
     *
     * @var string
     */
    public static $table_name = 'users';

    /**
     * Relazioni 'belongs_to'.
     *
     * @var array
     */
    public static $belongs_to = [
        ['language', 'model' => Language::class],
        ['role', 'model' => Role::class],
        ['province', 'model' => Province::class],
        ['country', 'model' => Country::class],
        ['shipping_address', 'model' => Address::class],
    ];

    /**
     * Relazioni 'has_and_belongs_to_many'.
     *
     * @var array
     */
    public static $has_and_belongs_to_many = [
        ['groups', 'model' => Group::class],
    ];

    /**
     * Relazioni 'has_many'.
     *
     * @var array
     */
    public static $has_many = [
        ['addresses', 'model' => Address::class, 'foreign_key' => 'user_id'],
    ];

    /**
     * Relazioni 'count_has_many'.
     *
     * @var array
     */
    public static $count_has_many = [
        ['total_orders', 'model' => Order::class, 'foreign_key' => 'customer_id'],
    ];

    /**
     * Validazioni 'validates_presence_of'.
     *
     * @var array
     */
    public static $validates_presence_of = [
        ['username', 'message' => 'required_username'],
        ['email', 'message' => 'required_email'],
        ['password', 'message' => 'required_password', 'on' => 'create'],
    ];

    /**
     * Validazioni 'validates_uniqueness_of'.
     *
     * @var array
     */
    public static $validates_uniqueness_of = [
        ['url', 'message' => 'invalid_url_uniqueness'],
    ];

    /**
     * Validazioni 'validates_format_of'.
     *
     * @var array
     */
    public static $validates_format_of = [
        ['email', 'message' => 'invalid_email_format', 'with' => REGEXP_VALIDATE_EMAIL],
    ];

    /**
     * Validazioni 'validates_by_methods_of'.
     *
     * @var array
     */
    public static $validates_by_methods_of = [
        ['username', 'method' => 'validate_username_format', 'message' => 'invalid_username_format'],
        ['username', 'method' => 'validate_uniqueness_of_username', 'message' => 'invalid_username_uniqueness'],
        ['email', 'method' => 'validate_uniqueness_of_email', 'message' => 'invalid_email_uniqueness'],
        ['partita_iva', 'method' => 'validate_partita_iva', 'message' => 'invalid_partita_iva_format'],
        ['codice_fiscale', 'method' => 'validate_codice_fiscale', 'message' => 'invalid_codice_fiscale_format'],
    ];

    /**
     * Callback 'after_validation_on_create'.
     *
     * @var array
     */
    public static $after_validation_on_create = ['validate_privacy', 'validate_confirm_password_on_create'];

    /**
     * Callback 'after_validation_on_update'.
     *
     * @var array
     */
    public static $after_validation_on_update = ['validate_confirm_password_on_update'];

    /**
     * Callback 'after_validation'.
     *
     * @var array
     */
    public static $after_validation = ['after_validation'];

    /**
     * Callback 'before_create'.
     *
     * @var array
     */
    public static $before_create = ['generateUrl'];

    /**
     * Callback 'after_create'.
     *
     * @var array
     */
    public static $after_create = ['addSelfPortfolioRelation'];

    /**
     * Callback 'before_destroy'.
     *
     * @var array
     */
    public static $before_destroy = ['delGroups', 'delete_newsletter_recipients'];

    /**
     * Lista dei gruppi associati all'utente.
     *
     * @var Group[]
     */
    protected $groups;

    /**
     * @var bool
     */
    protected $validate_partita_iva = true;

    /**
     * @var bool
     */
    protected $validate_codice_fiscale = true;

    /**
     * Cache interna - Avatar.
     *
     * @var File
     */
    private $avatar;

    /**
     * Elenco dei genitori diretti, usato per cache interna.
     *
     * @var Account[]
     */
    protected static $parents;

    /**
     * Cerca un account in base all’indirizzo email.
     *
     * @param string $email
     * @return Account
     */
    public static function findByEmail($email)
    {
        if (!$email) {
            return null;
        }

        $email = strtolower($email);

        return static::first([
            'conditions' => ['email = ?', $email],
        ]);
    }

    /**
     * Cerca un account in base allo username.
     *
     * @param string $username
     * @return Account
     */
    public static function findByUsername($username)
    {
        if (!$username) {
            return null;
        }

        if (preg_match(REGEXP_VALIDATE_EMAIL, $username)) {
            $username = strtolower($username);
        }

        return static::first([
            'conditions' => ['username = ?', $username],
        ]);
    }

    /**
     * Cerca un account in base alla chiave di reset della password.
     *
     * @param string $key
     * @return Account
     */
    public static function findByResetPasswordKey($key)
    {
        if (!$key) {
            return null;
        }

        return static::first([
            'conditions' => ['reset_password_key = ?', $key],
        ]);
    }

    /**
     * {@inheritdoc}
     */
    public function __set($name, $value)
    {
        // Caso speciale per la password.
        if ($name === 'password' || $name === 'confirm_password') {
            $value = clean_password($value);

            if ($name === 'password' && $value === '') {
                return;
            }
        }

        if ($name === 'email') {
            $value = strtolower($value);
        }

        if ($name === 'username' && preg_match(REGEXP_VALIDATE_EMAIL, $value)) {
            $value = strtolower($value);
        }

        parent::__set($name, $value);
    }

    /**
     * Controlla che il checkbox sulla privacy sia spuntato.
     *
     * @internal
     */
    public function validate_privacy()
    {
        if ($this->attributePresent('privacy') && $this->attributes['privacy']) {
            return;
        }

        $this->errors->add('privacy', 'required_privacy');
    }

    /**
     * Verifica la correttezza della conferma della password nel caso di nuovo account.
     *
     * È previsto che il campo `confirm_password` non sia presente, ma sia obbligatorio se presente. Questa ulteriore
     * verifica va fatta dal controller controllando i dati inviati dalla richiesta HTTP.
     *
     * @internal
     */
    public function validate_confirm_password_on_create()
    {
        $this->validate_confirm_password();
    }

    /**
     * Verifica la correttezza della conferma della password in caso di modifica dell’account.
     *
     * A differenza della creazione di un nuovo account, la modifica prevede l'obbligo della conferma della password se
     * la password è stata modificata. Questa differenza perché in fase di creazione si suppone che il controller invii
     * una notifica via e-mail con i dati inseriti dall’utente, password compresa. Ma tale notifica non viene inviata
     * in caso di modifica.
     *
     * @internal
     */
    public function validate_confirm_password_on_update()
    {
        if ($this->attributeEdited('password') && $this->password != '') {
            if ($this->attributePresent('confirm_password')) {
                $this->validate_confirm_password();
            } else {
                $this->errors->add('confirm_password', 'required_confirm_password');
            }
        }
    }

    /**
     * Verifica la correttezza della conferma della password se presente.
     *
     * @internal
     */
    protected function validate_confirm_password()
    {
        if ($this->attributePresent('confirm_password')) {
            if (empty($this->confirm_password)) {
                $this->errors->add('confirm_password', 'required_confirm_password');
            } elseif ($this->password != $this->confirm_password) {
                $this->errors->add('confirm_password', 'password_mismatch');
            }
        }
    }

    /**
     * Verifica che lo username non sia uguale ad una e-mail diversa da quella dell’account.
     *
     * @param string $username
     * @return bool
     *
     * @internal
     */
    public function validate_username_format($username)
    {
        $username = strtolower($username);

        if (preg_match(REGEXP_VALIDATE_EMAIL, $username)) {
            return $this->email === $username;
        }

        return true;
    }

    /**
     * Verifica l’unicità dello username e-mail.
     *
     * @param string $username
     * @return bool
     *
     * @internal
     */
    public function validate_uniqueness_of_username($username)
    {
        $username = strtolower($username);

        if (preg_match(REGEXP_VALIDATE_EMAIL, $username)) {
            return true;
        }

        $conditions = ['username = :username', 'username' => $username];

        if (!$this->isNewRecord()) {
            $primary_keys = $this->primaryKey();

            $primary_key_conditions = [];
            foreach ($primary_keys as $pk_name) {
                $primary_key_conditions[] = "$pk_name = :$pk_name";
                $conditions[$pk_name] = $this->$pk_name;
            }

            $conditions[0] .= ' AND NOT ( ' . implode(' AND ', $primary_key_conditions) . ' )';
        }

        if (static::count(['conditions' => $conditions])) {
            return false;
        }

        return true;
    }

    /**
     * Verifica l’unicità dell’indirizzo e-mail.
     *
     * @param string $email
     * @return bool
     *
     * @internal
     */
    public function validate_uniqueness_of_email($email)
    {
        $email = strtolower($email);
        $conditions = ['email = :email', 'email' => $email];

        if (!$this->isNewRecord()) {
            $primary_keys = $this->primaryKey();

            $primary_key_conditions = [];
            foreach ($primary_keys as $pk_name) {
                $primary_key_conditions[] = "$pk_name = :$pk_name";
                $conditions[$pk_name] = $this->$pk_name;
            }

            $conditions[0] .= ' AND NOT ( ' . implode(' AND ', $primary_key_conditions) . ' )';
        }

        if (static::count(['conditions' => $conditions])) {
            return false;
        }

        return true;
    }

    /**
     * Permette di abilitare o disabilitare il controllo della partita iva.
     *
     * @param bool $enabled
     */
    public function enableValidationOfPartitaIva($enabled = true)
    {
        $this->validate_partita_iva = (bool) $enabled;
    }

    /**
     * Permette di abilitare o disabilitare il controllo del codice fiscale.
     *
     * @param bool $enabled
     */
    public function enableValidationOfCodiceFiscale($enabled = true)
    {
        $this->validate_codice_fiscale = (bool) $enabled;
    }

    /**
     * Verifica la validità della partita iva. Il campo non è comunque obbligatorio.
     *
     * @param string $partita_iva
     * @return bool
     *
     * @internal
     */
    public function validate_partita_iva($partita_iva)
    {
        if ($this->validate_partita_iva) {
            return Validator::partita_iva($partita_iva);
        }

        return true;
    }

    /**
     * Verifica la validità del codice fiscale. Il campo non è comunque obbligatorio.
     *
     * @param string $codice_fiscale
     * @return bool
     *
     * @internal
     */
    public function validate_codice_fiscale($codice_fiscale)
    {
        if ($this->validate_codice_fiscale) {
            return Validator::codice_fiscale($codice_fiscale);
        }

        return true;
    }

    /**
     * Imposta il campo fullname.
     *
     * @internal
     */
    public function after_validation()
    {
        $this->fullname = $this->name();
    }

    /**
     * Restituisce i gruppi ai quali fa parte l'utente.
     *
     * @return Group[]
     */
    public function getGroups()
    {
        if ($this->groups === null) {
            $this->groups = [];

            $groups = GroupUser::all([
                'joins'      => 'LEFT JOIN ' . Group::tableName() . ' AS g ON g.id = `from`.group_id',
                'conditions' => ['`from`.user_id = ?', $this->id]
            ]);

            /** @var Group $group */
            foreach ($groups as $group) {
                $this->groups[$group->id] = $group;
            }
        }

        return $this->groups;
    }

    /**
     * Imposta i gruppi dell'utente.
     *
     * @param array $groups
     * @return $this
     */
    public function setGroups(array $groups)
    {
        // preferisco lavorare con gli ID dei gruppi
        if (reset($groups) instanceof Group) {
            foreach ($groups as &$group) {
                $group = $group->id;
            }
        }

        // preferisco operare eseguendo una transazione
        $account = $this;

        self::transaction(
            function () use ($account, $groups) {
                // rimuovo tutti i permessi
                $account->delGroups();

                foreach ($groups as $group_id) {
                    GroupUser::create([
                        'group_id' => $group_id,
                        'user_id'  => $account->id,
                    ]);
                }
            }
        );

        $this->groups = null;

        return $this;
    }

    /**
     * Elimina le associazioni con i gruppi.
     *
     * @return $this
     */
    public function delGroups()
    {
        $groups = GroupUser::all([
            'conditions' => ['user_id = ?', $this->id]
        ]);

        foreach ($groups as $group) {
            $group->delete();
        }

        return $this;
    }

    /**
     * Verifica se l’utente appartiene ad un gruppo.
     *
     * È possibile specificare anche una lista di gruppi passando come argomento un array. In questo caso, il metodo
     * restituisce true solo se l'account fa parte di tutti i gruppi passati.
     *
     * Il gruppo può essere sia istanza di Group che l'ID del gruppo.
     *
     * @param mixed $groups Può essere l’ID del gruppo o un array di ID. Può essere anche un modello gruppo, o un array di modelli.
     * @return bool
     */
    public function inGroup($groups)
    {
        if (!is_array($groups)) {
            $groups = [$groups];
        }

        foreach ($groups as &$group) {
            if ($group instanceof Group) {
                $group = $group->id;
            }
        }

        if (count($groups) == 1) {
            return array_key_exists(reset($groups), $this->getGroups());
        } else {
            $result = true;

            foreach ($groups as $group) {
                if (!array_key_exists($group, $this->getGroups())) {
                    $result = false;
                }
            }

            return $result;
        }
    }

    /**
     * Associa un gruppo all’utente.
     *
     * @param int|Group $group
     * @return $this
     */
    public function addGroup($group)
    {
        if ($group instanceof Group) {
            $group = $group->id;
        }

        $rel = GroupUser::find([$group, $this->id]);

        if ($rel === null) {
            GroupUser::create([
                'group_id' => $group,
                'user_id'  => $this->id,
            ]);
        }

        return $this;
    }

    /**
     * Rimuove un gruppo dall’utente.
     *
     * @param int|Group|GroupUser $group
     * @return $this
     */
    public function delGroup($group)
    {
        if ($group instanceof GroupUser) {
            $group->delete();
        } else {
            if ($group instanceof Group) {
                $group = $group->id;
            }

            $rel = GroupUser::find([$group, $this->id]);

            if ($rel) {
                $rel->delete();
            }
        }

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function getParents()
    {
        if (self::$parents === null) {
            $options = [
                'select'     => '`from`.*',
                'joins'      => 'LEFT JOIN ' . AccountRelation::tableName() . ' AS r ON r.parent_id = `from`.id',
                'conditions' => ['child_id = ?', $this->id],
            ];

            self::$parents = static::all($options);
        }

        return self::$parents;
    }

    /**
     * {@inheritdoc}
     */
    public function setParents(array $parents)
    {
        $old_parents = [];
        foreach ($this->getParents() as $old_parent) {
            $old_parents[$old_parent->id] = $old_parent;
        }

        foreach ($parents as $parent_id_or_parent_instance) {
            if ($parent_id_or_parent_instance instanceof static) {
                $parent_id = $parent_id_or_parent_instance->id;
            } else {
                $parent_id = $parent_id_or_parent_instance;
            }

            if (isset($old_parents[$parent_id])) {
                unset($old_parents[$parent_id]);
            } else {
                $this->setParent($parent_id);
            }
        }

        foreach ($old_parents as $old_parent) {
            if (!$this->delParent($old_parent)) {
                return false;
            }
        }

        self::$parents = null;

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function setParent($parent_id_or_parent_instance)
    {
        if ($parent_id_or_parent_instance instanceof static) {
            $parent_id = $parent_id_or_parent_instance->id;
        } else {
            $parent_id = $parent_id_or_parent_instance;
        }

        if ($parent_id === $this->id) {
            throw new \LogicException('The user cannot be a parent or a child of himself');
        }

        if (!$this->hasParent($parent_id)) {
            // Creo una relazione padre-figlio
            /** @var AccountRelation $relation */
            $relation = AccountRelation::create([
                'parent_id' => $parent_id,
                'child_id'  => $this->id,
            ]);

            // Creo le relazioni antenato-figlio. Gli antenati del figlio sono il padre e gli antenati del padre.
            $ancestors_options = [
                'conditions' => ['descendant_id = ?', $parent_id],
            ];

            /** @var AccountPortfolio $ancestor_relation */
            foreach (AccountPortfolio::all($ancestors_options) as $ancestor_relation) {
                $attributes = [
                    'relation_id'   => $relation->id,
                    'ancestor_id'   => $ancestor_relation->ancestor_id,
                    'descendant_id' => $this->id,
                    'depth'         => $ancestor_relation->depth + 1,
                ];

                AccountPortfolio::create($attributes);
            }
        }

        self::$parents = null;

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function delParent($parent_id_or_instance)
    {
        if ($parent_id_or_instance instanceof static) {
            $parent_id = $parent_id_or_instance->id;
        } else {
            $parent_id = $parent_id_or_instance;
        }

        /** @var AccountRelation $relation */
        $relation = AccountRelation::first([
            'conditions' => ['parent_id = ? AND child_id = ?', $parent_id, $this->id],
        ]);

        if ($relation !== null) {
            $ancestors_options = [
                'conditions' => ['relation_id = ?', $relation->id],
            ];

            /** @var AccountPortfolio $ancestor_relation */
            foreach (AccountPortfolio::all($ancestors_options) as $ancestor_relation) {
                if (!$ancestor_relation->delete()) {
                    return false;
                }
            }

            return $relation->delete();
        }

        self::$parents = null;

        return true;
    }

    /**
     * {@inheritdoc}
     */
    public function hasParent($parent_id)
    {
        $options = [
            'conditions' => ['parent_id = ? AND child_id = ?', $parent_id, $this->id],
        ];

        return AccountRelation::find($options) !== null;
    }

    /**
     * {@inheritdoc}
     */
    public function hasAncestor($ancestor_id)
    {
        $options = [
            'conditions' => [
                'ancestor_id = :ancestor AND descendant_id = :descendant',
                'ancestor'   => $ancestor_id,
                'descendant' => $this->id,
            ],
        ];

        return AccountPortfolio::find($options) !== null;
    }

    /**
     * {@inheritdoc}
     */
    public function getDescendants()
    {
        $options = [
            'joins' => 'INNER JOIN ' . AccountPortfolio::tableName() . ' AS p ON p.descendant_id = `from`.id',
            'conditions' => [
                'ancestor_id = :ancestor AND descendant_id != :me',
                'ancestor' => $this->id,
                'me'       => $this->id,
            ],
            'collection' => true,
        ];

        return Account::all($options);
    }

    /**
     * {@inheritdoc}
     */
    public function countDescendants()
    {
        $options = [
            'conditions' => [
                'ancestor_id = :ancestor AND descendant_id != :me',
                'ancestor' => $this->id,
                'me'       => $this->id,
            ],
        ];

        return AccountPortfolio::count($options);
    }

    /**
     * {@inheritdoc}
     */
    public function hasDescendant($descendant_id)
    {
        $options = [
            'conditions' => [
                'ancestor_id = :ancestor AND descendant_id = :descendant',
                'ancestor'   => $this->id,
                'descendant' => $descendant_id,
            ],
        ];

        return AccountPortfolio::find($options) !== null;
    }

    /**
     * {@inheritdoc}
     */
    public function generateUrl()
    {
        $this->url = (new DateTime())->format('U') . (string) random(8, '0123456789');
    }

    /**
     * {@inheritdoc}
     */
    public function getUrl()
    {
        return $this->url;
    }

    /**
     * Aggiunge la relazione a se stesso nella tabella del portfolio.
     *
     * @internal
     */
    public function addSelfPortfolioRelation()
    {
        $attributes = [
            'relation_id'   => null,
            'ancestor_id'   => $this->id,
            'descendant_id' => $this->id,
            'depth'         => 0,
        ];

        AccountPortfolio::create($attributes);
    }

    /**
     * Elimina le relazioni dell’utente e sposta tutti i suoi figli sotto il controllo di tutti i suoi padri.
     *
     * @return bool
     *
     * @internal
     */
    public function deletePortfolioRelations()
    {
        // Ricavo l’elenco degli ID dei genitori
        $parents_options = [
            'conditions' => ['child_id = ?', $this->id],
        ];

        $parents = [];
        /** @var AccountRelation $relation */
        foreach (AccountRelation::all($parents_options) as $relation) {
            $parents[] = $relation->parent_id;
        }

        // Per ogni Account figlio, associo i nuovi padri
        $children_options = [
            'select'     => '`from`.*',
            'joins'      => 'LEFT JOIN ' . AccountRelation::tableName() . ' AS ar ON ar.child_id = `from`.id',
            'conditions' => ['ar.parent_id = ?', $this->id],
        ];

        /** @var Account $children */
        foreach (Account::all($children_options) as $children) {
            foreach ($parents as $parent_id) {
                $children->setParent($parent_id);
            }
        }

        // A questo punto, elimino tutte le associazioni dell’utente
        $portfolio_options = [
            'conditions' => ['ancestor_id = :account OR descendant_id = :account', 'account' => $this->id],
        ];

        /** @var AccountPortfolio $portfolio */
        foreach (AccountPortfolio::all($portfolio_options) as $portfolio) {
            $portfolio->delete();
        }

        $relations_options = [
            'conditions' => ['parent_id = :account OR child_id = :account', 'account' => $this->id],
        ];

        /** @var AccountRelation $relation */
        foreach (AccountRelation::all($relations_options) as $relation) {
            $relation->delete();
        }

        return true;
    }

    /**
     * @todo
     * @internal
     */
    public function delete_newsletter_recipients()
    {
//		$rows = NewsletterSend::all(array(
//			'conditions'	=> array('recipient_id = ?', $this->id)
//		));
//
//		foreach ( $rows as $row )
//		{
//			$row->recipient_id = null;
//			$row->save();
//		}
    }

    /**
     * {@inheritdoc}
     */
    public function isActive()
    {
        return $this->is_active;
    }

    /**
     * {@inheritdoc}
     */
    public function isAnonymous()
    {
        return $this->id == self::ANONYMOUS;
    }

    /**
     * {@inheritdoc}
     */
    public function isAdmin()
    {
        return $this->role_id == Role::ADMIN;
    }

    /**
     * {@inheritdoc}
     */
    public function isFounder()
    {
        return (bool) $this->is_founder;
    }

    /**
     * {@inheritdoc}
     */
    public function isBusiness()
    {
        return (bool) $this->is_business;
    }

    /**
     * {@inheritdoc}
     */
    public function hasPermit($key)
    {
        // Il founder ha automaticamente qualsiasi permesso
        if ($this->isFounder()) {
            return true;
        }

        if ($role = $this->role) {
            return $role->hasPermit($key);
        }

        return false;
    }

    /**
     * {@inheritdoc}
     */
    public function name($invert = false)
    {
        $event = new Event($this, $this->getEventName('name_function'), ['invert' => $invert]);

        if ($name = $this->dispatcher->notifyUntil($event)->getReturnValue()) {
            return $name;
        }

        if ($invert) {
            $name = trim("$this->surname $this->name");
        } else {
            $name = trim("$this->name $this->surname");
        }

        return empty($name) ? $this->username : $name;
    }

    /**
     * {@inheritdoc}
     */
    public function getActivationKey()
    {
        return $this->activation_key;
    }

    /**
     * {@inheritdoc}
     */
    public function resetActivationKey()
    {
        $this->activation_key = random(32);

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function delete()
    {
        if (!$this->isDeletable()) {
            return false;
        }

        try {
            $event = new Event($this, $this->getEventName('before_delete'));
            $this->dispatcher->notify($event);
        } catch (\Exception $e) {
            return false;
        }

        if (!$this->deleteGroupsRelations()) {
            return false;
        }

        if (!$this->deleteAvatar()) {
            return false;
        }

        if (!$this->deletePortfolioRelations()) {
            return false;
        }

        if (!parent::delete()) {
            return false;
        }

        try {
            $event = new Event($this, $this->getEventName('after_delete'));
            $this->dispatcher->notify($event);
        } catch (\Exception $e) {
            return false;
        }

        return true;
    }

    /**
     * {@inheritdoc}
     */
    public function isDeletable()
    {
        $event = new Event($this, $this->getEventName('is_deletable'), ['is_deletable' => true]);
        $this->dispatcher->notify($event);

        return $event->getParameter('is_deletable');
    }

    /**
     * Reimposta l'autore di nodi e commenti ad anonimo prima di cancellare l'utente che li ha creati.
     *
     * @return bool
     *
     * @deprecated
     */
    public function updateCmsRelations()
    {
        /* @var Manager\ApplicationManager $app_manager */
        $app_manager = Kernel::instance()->getContainer()->get('application_manager');

        if ($app_manager->isInstalled('Cms')) {
            $options = ['conditions' => ['author_id = :user', 'user' => $this->id]];

            /** @var \ActiveRecord\Base $node_model_class */
            $node_model_class = 'Application\\Cms\\Model\\Node';

            foreach ($node_model_class::all($options) as $node) {
                $node->author_id = 1;
                if (!$node->save()) {
                    return false;
                }
            }

            /** @var \ActiveRecord\Base $comment_model_class */
            $comment_model_class = 'Application\\Cms\\Model\\Comment';

            foreach ($comment_model_class::all($options) as $comment) {
                $comment->author_id = 1;
                $comment->author_name = $this->name();
                $comment->author_email = $this->email;

                if (!$comment->save()) {
                    return false;
                }
            }
        }

        return true;
    }

    /**
     * Rimuove l’utente dai gruppi.
     *
     * @return bool
     */
    public function deleteGroupsRelations()
    {
        $options = ['conditions' => ['user_id = :user', 'user' => $this->id]];

        foreach (GroupUser::all($options) as $group) {
            if (!$group->delete()) {
                return false;
            }
        }

        return true;
    }

    /**
     * Rimuovo i reminder dell’utente
     *
     * @return bool
     *
     * @deprecated
     */
    public function deleteReminders()
    {
        /* @var Manager\ApplicationManager $app_manager */
        $app_manager = Kernel::instance()->getContainer()->get('application_manager');

        if ($app_manager->isInstalled('Reminder')) {
            $options = ['conditions' => ['user_id = :user', 'user' => $this->id]];

            /** @var \ActiveRecord\Base $reminder_model_class */
            $reminder_model_class = 'Application\\Reminder\\Model\\Reminder';

            foreach ($reminder_model_class::all($options) as $reminder) {
                if (!$reminder->delete()) {
                    return false;
                }
            }
        }

        return true;
    }

    /**
     * {@inheritdoc}
     */
    public function getAvatar()
    {
        if ($this->avatar === null && $this->avatar_id) {
            $this->avatar = File::find($this->avatar_id);
        }

        return $this->avatar;
    }

    /**
     * {@inheritdoc}
     */
    public function getAvatarId()
    {
        return $this->avatar_id;
    }

    /**
     * {@inheritdoc}
     */
    public function setAvatar($avatar)
    {
        if ($avatar instanceof File) {
            $avatar_id = $avatar->getId();
        } elseif (is_numeric($avatar)) {
            $avatar_id = (int) $avatar;
        } elseif ($avatar === null) {
            $avatar_id = null;
        } else {
            throw new \InvalidArgumentException('Expected an integer or an instance of File.');
        }

        $this->avatar_id = $avatar_id;
        $this->avatar = null;

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function deleteAvatar()
    {
        if ($this->avatar_id) {
            $image = File::find($this->avatar_id);

            $this->avatar_id = null;
            $this->save(false);

            return $image->delete();
        }

        return true;
    }

    /**
     * Alias di getAvatar().
     *
     * @see Account::getAvatar() Metodo di riferimento.
     *
     * @return File
     */
    public function avatar()
    {
        return $this->getAvatar();
    }

    /**
     * Restituisce l’indirizzo nel formato indicato dal paese di appartenenza.
     *
     * @return string
     */
    public function renderAddress()
    {
        $address = new I18nAddress(
            $this->name,
            $this->surname,
            $this->company,
            $this->address,
            $this->address2,
            $this->postcode,
            $this->city,
            $this->province,
            $this->country,
            $this->telephone
        );

        /** @var \Application\Core\I18n\Address\AddressRendererFactory $factory */
        $factory = Kernel::instance()->getContainer()->get('address_renderer_factory');
        $countryCode = $this->country ? $this->country->code : 'IT';

        return $factory->create($countryCode)->render($address);
    }

    /**
     * {@inheritdoc}
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * {@inheritdoc}
     */
    public function getLanguage()
    {
        return $this->language;
    }

    /**
     * @return int
     */
    public function getLanguageId()
    {
        return $this->language_id;
    }

    /**
     * @param int|Language $language
     * @return $this
     */
    public function setLanguage($language)
    {
        if ($language instanceof Language) {
            $language_id = $language->id;
        } elseif (is_numeric($language)) {
            $language_id = (int) $language;
        } else {
            throw new \InvalidArgumentException('Expected an integer or an instance of Language.');
        }

        $this->language_id = $language_id;
        unset($this->language);

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function getRole()
    {
        return $this->role;
    }

    /**
     * {@inheritdoc}
     */
    public function getRoleId()
    {
        return $this->role_id;
    }

    /**
     * {@inheritdoc}
     */
    public function setRole($role)
    {
        if ($role instanceof Role) {
            $role_id = $role->id;
        } elseif (is_numeric($role)) {
            $role_id = (int) $role;
        } else {
            throw new \InvalidArgumentException('Expected an integer or an instance of Role.');
        }

        $this->role_id = $role_id;
        unset($this->role);

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function getUsername()
    {
        return $this->username;
    }

    /**
     * {@inheritdoc}
     */
    public function setUsername($username)
    {
        $this->username = $username;

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function getEmail()
    {
        return $this->email;
    }

    /**
     * {@inheritdoc}
     */
    public function setEmail($email)
    {
        $this->email = strtolower($email);

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function getPassword()
    {
        return $this->password;
    }

    /**
     * {@inheritdoc}
     */
    public function setPassword($password)
    {
        $this->password = $password;

        return $this;
    }

    /**
     * @param string $password
     * @return $this
     */
    public function setConfirmPassword($password)
    {
        $this->confirm_password = $password;

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function getSubscribeDate()
    {
        return $this->created_at;
    }

    /**
     * {@inheritdoc}
     */
    public function getCreationDate()
    {
        return $this->created_at;
    }

    /**
     * {@inheritdoc}
     */
    public function getLastUpdateDate()
    {
        return $this->updated_at;
    }

    /**
     * {@inheritdoc}
     */
    public function getLastLoginDate()
    {
        return $this->last_login_at;
    }

    /**
     * @return DateTime
     */
    public function getExpirationDate()
    {
        return $this->expire_at;
    }

    /**
     * @param DateTime $expiration_date
     * @return $this
     */
    public function setExpirationDate(DateTime $expiration_date)
    {
        $this->expire_at = $expiration_date;

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function activate()
    {
        $this->is_active = true;

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function deactivate()
    {
        $this->is_active = false;

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function getUserId()
    {
        return $this->id;
    }

    /**
     * {@inheritdoc}
     */
    public function getAccount()
    {
        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function getUserDetails()
    {
        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function getName()
    {
        return $this->name;
    }

    /**
     * {@inheritdoc}
     */
    public function setName($name)
    {
        $this->name = $name;

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function getSurname()
    {
        return $this->surname;
    }

    /**
     * {@inheritdoc}
     */
    public function setSurname($surname)
    {
        $this->surname = $surname;

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function getDescription()
    {
        return $this->description;
    }

    /**
     * {@inheritdoc}
     */
    public function setDescription($description)
    {
        $this->description = $description;

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function getCompany()
    {
        return $this->company;
    }

    /**
     * {@inheritdoc}
     */
    public function setCompany($company)
    {
        $this->company = $company;

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function getTaxCode()
    {
        return $this->codice_fiscale;
    }

    /**
     * {@inheritdoc}
     */
    public function setTaxCode($tax_code)
    {
        $this->codice_fiscale = $tax_code;

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function getVatNumber()
    {
        return $this->partita_iva;
    }

    /**
     * {@inheritdoc}
     */
    public function setVatNumber($vat_number)
    {
        $this->partita_iva = $vat_number;

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function getAddress()
    {
        return $this->address;
    }

    /**
     * {@inheritdoc}
     */
    public function setAddress($address)
    {
        $this->address = $address;

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function getAddress2()
    {
        return $this->address2;
    }

    /**
     * {@inheritdoc}
     */
    public function setAddress2($address2)
    {
        $this->address2 = $address2;

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function getCity()
    {
        return $this->city;
    }

    /**
     * {@inheritdoc}
     */
    public function setCity($city)
    {
        $this->city = $city;

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function getPostCode()
    {
        return $this->postcode;
    }

    /**
     * {@inheritdoc}
     */
    public function setPostcode($postcode)
    {
        $this->postcode = $postcode;

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function getProvince()
    {
        return $this->province;
    }

    /**
     * {@inheritdoc}
     */
    public function setProvince($province)
    {
        if (is_numeric($province)) {
            $this->province_id = $province;
            $this->province = null;
        } elseif ($province instanceof ProvinceInterface) {
            $this->province_id = $province->getId();
            $this->province = $province;
        } else {
            throw new \InvalidArgumentException(
                sprintf(
                    'The province must be an integer or an instance of ProvinceInterface. "%s" given instead.',
                    var_to_string($province)
                )
            );
        }

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function getProvinceId()
    {
        return $this->province_id;
    }

    /**
     * {@inheritdoc}
     */
    public function setProvinceId($province_id)
    {
        if ($province_id) {
            $this->province_id = $province_id;
            $this->province = null;
        }

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function getCountry()
    {
        return $this->country;
    }

    /**
     * {@inheritdoc}
     */
    public function setCountry($country)
    {
        if (is_numeric($country)) {
            $this->country_id = $country;
            $this->country = null;
        } elseif ($country instanceof CountryInterface) {
            $this->country_id = $country->getId();
            $this->country = $country;
        } else {
            throw new \InvalidArgumentException(
                sprintf(
                    'The country must be an integer or an instance of CountryInterface. "%s" given instead.',
                    var_to_string($country)
                )
            );
        }

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function getCountryId()
    {
        return $this->country_id;
    }

    /**
     * {@inheritdoc}
     */
    public function setCountryId($country_id)
    {
        $this->country_id = $country_id;
        $this->country = null;

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function getTelephone()
    {
        return $this->telephone;
    }

    /**
     * {@inheritdoc}
     */
    public function setTelephone($telephone)
    {
        $this->telephone = $telephone;

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function getMobile()
    {
        throw new \LogicException('Not implemented');
    }

    /**
     * {@inheritdoc}
     */
    public function setMobile($mobile)
    {
        throw new \LogicException('Not implemented');
    }

    /**
     * {@inheritdoc}
     */
    public function getFax()
    {
        return $this->fax;
    }

    /**
     * {@inheritdoc}
     */
    public function setFax($fax)
    {
        $this->fax = $fax;

        return $this;
    }

    /**
     * {@inheritdoc}
     */
    public function getWebsite()
    {
        return $this->website;
    }

    /**
     * {@inheritdoc}
     */
    public function setWebsite($website)
    {
        $this->website = $website;

        return $this;
    }

    /**
     * @param bool $privacy
     * @return $this
     */
    public function setPrivacy($privacy)
    {
        $this->privacy = (bool) $privacy;

        return $this;
    }
}
