<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author  Daniele De Nobili
 * @package Application\Core
 */

namespace Application\Core;

use Application\Core\Model\LanguageSite;
use Pongho\Core\LocalizationInterface;
use Pongho\Utilities\Inflector;

/**
 * Gestore della localizzazione.
 */
class Localization implements LocalizationInterface
{
    /**
     * Modello della lingua in uso.
     *
     * @var \Application\Core\Model\LanguageSite
     */
    protected $language;

    /**
     * Elenco delle traduzioni disponibili.
     *
     * @var array
     */
    protected $translations;

    /**
     * Costruttore.
     *
     * @param \Application\Core\Model\LanguageSite $language
     */
    public function __construct(LanguageSite $language)
    {
        $this->setLanguage($language);
    }

    /**
     * Restituisce la lingua.
     *
     * @return \Application\Core\Model\LanguageSite
     *
     * @deprecated
     */
    public function getLanguage()
    {
        return $this->language;
    }

    /**
     * Restituisce l’ID della lingua.
     *
     * @return integer
     *
     * @deprecated
     */
    public function getLanguageId()
    {
        return $this->language->language_id;
    }

    /**
     * Restituisce il percorso base della lingua.
     *
     * @return string
     *
     * @deprecated
     */
    public function getLanguagePath()
    {
        return $this->language->path;
    }

    /**
     * Permette di impostare la lingua.
     *
     * @param \Application\Core\Model\LanguageSite $language
     *
     * @deprecated
     */
    public function setLanguage(LanguageSite $language)
    {
        $this->language = $language;
        $this->translations = null;

        setlocale(LC_TIME, $language->locale);
        date_default_timezone_set($language->timezone);
    }

    /**
     * Restituisce la traduzione di una voce della lingua.
     *
     * @param string $key
     * @return string
     *
     * @deprecated
     */
    public function __get($key)
    {
        return $this->get($key);
    }

    /**
     * Restituisce la traduzione di una voce della lingua al pari di __get().
     * Se è indicato anche il parametro $count, la funzione restituisce la traduzione in base al numero indicato.
     * Ad esempio:
     *
     * <code>
     *   get('row', 1) // Restituisce “Una riga”
     *   get('row', 0) // Restituisce “Nessuna riga”
     *   get('row', 10) // Restituisce “10 righe”
     *   get('row') // Restituisce “riga”
     * </code>
     *
     * @param string  $key
     * @param integer $count
     * @return string
     */
    public function get($key, $count = null)
    {
        $this->prepareTranslations();

        if (is_int($count)) {
            switch ($count) {
                case 0:
                    $key .= '_zero';
                    break;

                case 1:
                    $key .= '_one';
                    break;

                default:
                    $key .= '_more';

                    return array_key_exists($key, $this->translations) ? sprintf($this->translations[$key], $count) : sprintf(Inflector::humanize($key), $count);
            }
        }

        return array_key_exists($key, $this->translations) ? $this->translations[$key] : Inflector::humanize($key);
    }

    /**
     * Restituisce un array con tutte le traduzioni
     *
     * @return array
     */
    public function all()
    {
        $this->prepareTranslations();

        return $this->translations;
    }

    /**
     * Prepara le traduzioni
     */
    protected function prepareTranslations()
    {
        if ($this->translations === null) {
            $this->translations = array();

            foreach ($this->language->getTranslations() as $translation) {
                $this->translations[$translation->key] = $translation->value;
            }
        }
    }
}
