<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Installer\Helper;

use Doctrine\DBAL\Connection;
use Doctrine\DBAL\DBALException;

/**
 * TranslationsHelper
 */
class TranslationsHelper
{
    /**
     * @var Connection
     */
    protected $connection;

    /**
     * @var array
     */
    protected $languages_sites;

    /**
     * @param Connection $connection
     */
    public function __construct(Connection $connection)
    {
        $this->connection = $connection;
    }

    /**
     * Installa le traduzioni usando una stringa in formato JSON.
     *
     * @param string $json
     * @return $this
     */
    public function installFromJson($json)
    {
        $key_field_name = $this->connection->getDatabasePlatform()->quoteIdentifier('key');

        $stm = "INSERT INTO pongho_translations (language_site_id, application, {$key_field_name}, value) VALUES (?, ?, ?, ?)";
        $sth = $this->connection->prepare($stm);

        foreach (json_decode($json, true) as $translation) {
            $sth->bindValue(1, $translation['language_site_id'], \PDO::PARAM_INT);
            $sth->bindValue(2, $translation['application'], \PDO::PARAM_STR);
            $sth->bindValue(3, $translation['key'], \PDO::PARAM_STR);
            $sth->bindValue(4, $translation['value'], \PDO::PARAM_STR);

            try {
                $sth->execute();
            } catch (DBALException $e) {
                // La traduzione esiste già, ignoro l'errore e proseguo

                continue;
            }
        }

        return $this;
    }

    /**
     * Installa le traduzioni caricandole da un file JSON.
     *
     * @param string $json_file
     * @return $this
     */
    public function loadFromJson($json_file)
    {
        return $this->installFromJson(file_get_contents($json_file));
    }

    /**
     * @param int $language_site_id
     * @return $this
     */
    public function setLanguageSite($language_site_id)
    {
        trigger_error(sprintf('Il metodo %s non è testato.', __METHOD__), E_USER_NOTICE);

        $this->languages_sites = [$language_site_id];

        return $this;
    }

    /**
     * @param int $language_id
     * @param int $site_id
     * @return $this
     */
    public function setLanguageAndSite($language_id, $site_id)
    {
        trigger_error(sprintf('Il metodo %s non è testato.', __METHOD__), E_USER_NOTICE);

        $this->setLanguagesSites(function () use ($language_id, $site_id) {
            $sth = $this->connection->prepare(
                'SELECT id FROM pongho_languages_sites WHERE language_id = :language AND site_id = :site'
            );

            $sth->execute([
                'language' => $language_id,
                'site'     => $site_id,
            ]);

            return $sth;
        });

        return $this;
    }

    /**
     * @param string $culture
     * @return $this
     */
    public function setCulture($culture)
    {
        $this->setLanguagesSites(function () use ($culture) {
            $sth = $this->connection->prepare(
                'SELECT ls.id FROM pongho_languages_sites AS ls INNER JOIN pongho_languages AS l ON l.id = ls.language_id WHERE l.culture = :culture'
            );

            $sth->execute([
                'culture' => $culture,
            ]);

            return $sth;
        });

        return $this;
    }

    /**
     * @param int $language_id
     * @return $this
     */
    public function setLanguage($language_id)
    {
        trigger_error(sprintf('Il metodo %s non è testato.', __METHOD__), E_USER_NOTICE);

        $this->setLanguagesSites(function () use ($language_id) {
            $sth = $this->connection->prepare(
                'SELECT id FROM pongho_languages_sites WHERE language_id = :language'
            );

            $sth->execute([
                'language' => $language_id,
            ]);

            return $sth;
        });

        return $this;
    }

    /**
     * @param int $site_id
     * @return $this
     */
    public function setSite($site_id)
    {
        trigger_error(sprintf('Il metodo %s non è testato.', __METHOD__), E_USER_NOTICE);

        $this->setLanguagesSites(function () use ($site_id) {
            $sth = $this->connection->prepare(
                'SELECT id FROM pongho_languages_sites WHERE site_id = :site'
            );

            $sth->execute([
                'site' => $site_id,
            ]);

            return $sth;
        });

        return $this;
    }

    /**
     * @param string $key
     * @param string $value
     * @param string $application
     * @return $this
     */
    public function addOrChangeTranslation($key, $value, $application)
    {
        trigger_error(sprintf('Il metodo %s non è testato.', __METHOD__), E_USER_NOTICE);

        if ($this->exists($key)) {
            $this->changeTranslation($key, $value);
        } else {
            $this->addTranslation($key, $value, $application);
        }

        return $this;
    }

    /**
     * @param string $key
     * @param string $value
     * @param string $application
     * @return $this
     */
    public function addTranslation($key, $value, $application)
    {
        $this->exec(function ($language_site_id) use ($key, $value, $application) {
            $this->insertTranslation($language_site_id, $key, $value, $application);
        });

        return $this;
    }

    /**
     * @param string $key
     * @param string $value
     * @return $this
     */
    public function changeTranslation($key, $value)
    {
        $key_field_name = $this->connection->getDatabasePlatform()->quoteIdentifier('key');

        $this->exec(function ($language_site_id) use ($key, $value, $key_field_name) {
            $this->connection->update(
                'pongho_translations',
                [
                    'value'            => $value,
                ],
                [
                    $key_field_name    => $key,
                    'language_site_id' => $language_site_id,
                ],
                [
                    $key_field_name    => \PDO::PARAM_STR,
                    'language_site_id' => \PDO::PARAM_INT,
                    'application'      => \PDO::PARAM_STR,
                    'value'            => \PDO::PARAM_STR,
                ]
            );
        });

        return $this;
    }

    /**
     * @param string $old_key
     * @param string $new_key
     * @return $this
     */
    public function changeKey($old_key, $new_key)
    {
        $key_field_name = $this->connection->getDatabasePlatform()->quoteIdentifier('key');

        $this->connection->update(
            'pongho_translations',
            [
                $key_field_name => $new_key,
            ],
            [
                $key_field_name => $old_key,
            ]
        );

        return $this;
    }

    /**
     * @param string $old_key
     * @param string $new_key
     * @param string $value
     * @param string $application
     * @return $this
     */
    public function changeKeyOrAdd($old_key, $new_key, $value, $application)
    {
        trigger_error(sprintf('Il metodo %s non è testato.', __METHOD__), E_USER_NOTICE);

        $key_field_name = $this->connection->getDatabasePlatform()->quoteIdentifier('key');

        $this->exec(function ($language_site_id) use ($old_key, $new_key, $value, $application, $key_field_name) {
            $old = $this->connection->fetchAssoc(
                "SELECT * FROM pongho_translations WHERE {$key_field_name} = :old_key",
                [
                    'old_key' => $old_key,
                ]
            );

            if ($old) {
                $this->connection->update(
                    'pongho_translations',
                    [
                        $key_field_name => $new_key,
                    ],
                    [
                        $key_field_name    => $old_key,
                        'language_site_id' => $language_site_id,
                    ]
                );
            } else {
                $this->insertTranslation($language_site_id, $new_key, $value, $application);
            }
        });

        return $this;
    }

    /**
     * @param string $old_key
     * @param string $new_key
     * @param string $value
     * @param string $application
     * @return $this
     */
    public function copyOrAdd($old_key, $new_key, $value, $application)
    {
        $key_field_name = $this->connection->getDatabasePlatform()->quoteIdentifier('key');

        $this->exec(function ($language_site_id) use ($old_key, $new_key, $value, $application, $key_field_name) {
            $old = $this->connection->fetchAssoc(
                "SELECT * FROM pongho_translations WHERE {$key_field_name} = :old_key AND language_site_id = :ls",
                [
                    'old_key' => $old_key,
                    'ls' => $language_site_id,
                ]
            );

            if ($old) {
                $value = $old['value'];
            }

            $this->insertTranslation($language_site_id, $new_key, $value, $application);
        });

        return $this;
    }

    /**
     * @param string $key
     * @param string $application
     * @return $this
     */
    public function changeApplication($key, $application)
    {
        trigger_error(sprintf('Il metodo %s non è ancora stato sviluppato.', __METHOD__), E_USER_ERROR);

        return $this;
    }

    /**
     * @param string $key
     * @return bool
     */
    public function exists($key)
    {
        trigger_error(sprintf('Il metodo %s non è ancora stato sviluppato.', __METHOD__), E_USER_ERROR);
    }

    /**
     * @param string $key
     * @return $this
     */
    public function removeTranslation($key)
    {
        $key_field_name = $this->connection->getDatabasePlatform()->quoteIdentifier('key');

        $this->connection->delete(
            'pongho_translations',
            [
                $key_field_name => $key,
            ],
            [
                $key_field_name => \PDO::PARAM_STR,
            ]
        );

        return $this;
    }

    /**
     * @param int $language_site_id
     * @param $key
     * @param $application
     * @param $value
     */
    protected function insertTranslation($language_site_id, $key, $value, $application)
    {
        $key_field_name = $this->connection->getDatabasePlatform()->quoteIdentifier('key');

        try {
            $this->connection->insert(
                'pongho_translations',
                [
                    $key_field_name    => $key,
                    'language_site_id' => $language_site_id,
                    'application'      => $application,
                    'value'            => $value,
                ],
                [
                    $key_field_name    => \PDO::PARAM_STR,
                    'language_site_id' => \PDO::PARAM_INT,
                    'application'      => \PDO::PARAM_STR,
                    'value'            => \PDO::PARAM_STR,
                ]
            );
        } catch (DBALException $e) {
            // La traduzione esisteva già
        }
    }

    /**
     * @param callable $callable
     */
    protected function exec(callable $callable)
    {
        foreach ($this->languages_sites as $language_site_id) {
            call_user_func($callable, $language_site_id);
        }
    }

    /**
     * @param callable $callable
     */
    protected function setLanguagesSites(callable $callable)
    {
        $this->languages_sites = [];

        $sth = call_user_func($callable);

        while ($id = $sth->fetchColumn()) {
            $this->languages_sites[] = $id;
        }
    }
}
