<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Installer\Helper;

use Doctrine\DBAL\Connection;

/**
 * PermissionsHelper
 */
class PermissionsHelper
{
    /**
     * @var Connection
     */
    protected $connection;

    /**
     * @param Connection $connection
     */
    public function __construct(Connection $connection)
    {
        $this->connection = $connection;
    }

    /**
     * @param string $name
     * @param bool   $is_enabled
     */
    public function addPermission($name, $is_enabled = true)
    {
        $this->addPermissions([$name], $is_enabled);
    }

    /**
     * @param array $keys
     * @param bool  $is_enabled
     */
    public function addPermissions(array $keys, $is_enabled = true)
    {
        $key_field_name = $this->connection->quoteIdentifier('key');

        $placeholders = substr(str_repeat('?, ', count($keys)), 0, -2);
        $stm = "SELECT * FROM pongho_permits WHERE {$key_field_name} IN ({$placeholders})";

        $sth = $this->connection->prepare($stm);
        $sth->execute($keys);

        $permissions = [];
        while ($row = $sth->fetch()) {
            $permissions[$row['key']] = $row;
        }

        foreach ($keys as $key) {
            if (!isset($permissions[$key])) {
                $this->connection->insert(
                    'pongho_permits',
                    [
                        $key_field_name => $key,
                        'is_enabled'    => $is_enabled,
                    ],
                    [
                        $key_field_name => \PDO::PARAM_STR,
                        'is_enabled'    => \PDO::PARAM_BOOL,
                    ]
                );
            } elseif ($permissions[$key]['is_enabled'] !== $is_enabled) {
                $this->connection->update(
                    'pongho_permits',
                    [
                        'is_enabled' => $is_enabled,
                    ],
                    [
                        'id' => $row['id'],
                    ],
                    [
                        'id'         => \PDO::PARAM_INT,
                        'is_enabled' => \PDO::PARAM_BOOL,
                    ]
                );
            }
        }
    }

    /**
     * @param string $oldKey
     * @param string $newKey
     */
    public function renamePermission($oldKey, $newKey)
    {
        $key_field_name = $this->connection->quoteIdentifier('key');

        $newId = $this->findPermitId($newKey);
        $oldId = $this->findPermitId($oldKey);

        if ($newId && $oldId) {
            // Seleziono i ruoli interessati dai due permessi e li riassocio solo al nuovo.
            $sth = $this->connection->prepare(
                'SELECT role_id FROM pongho_permits_roles WHERE permit_id IN (:new, :old) GROUP BY role_id'
            );

            $sth->execute([
                'new' => $newId,
                'old' => $oldId,
            ]);

            $roles = [];
            while ($role = $sth->fetchColumn()) {
                $roles[] = $role;
            }

            if ($roles) {
                $this->connection->exec(
                    "DELETE FROM pongho_permits_roles WHERE permit_id IN ($newId, $oldId) AND role_id IN (" . implode(', ', $roles) . ')'
                );

                foreach ($roles as $role) {
                    $this->connection->insert(
                        'pongho_permits_roles',
                        [
                            'permit_id' => $newId,
                            'role_id'   => $role,
                        ],
                        [
                            'permit_id' => \PDO::PARAM_INT,
                            'role_id'   => \PDO::PARAM_INT,
                        ]
                    );
                }
            }
        } else {
            $this->connection->update(
                'pongho_permits',
                [
                    $key_field_name => $newKey,
                ],
                [
                    $key_field_name => $oldKey,
                ],
                [
                    $key_field_name => \PDO::PARAM_STR,
                ]
            );
        }
    }

    /**
     * @param array $keys
     */
    public function renamePermissions(array $keys)
    {
        foreach ($keys as $oldKey => $newKey) {
            $this->renamePermission($oldKey, $newKey);
        }
    }

    /**
     * @param string $key
     */
    public function removePermission($key)
    {
        $id = $this->findPermitId($key);

        if (!$id) {
            return;
        }

        $this->connection->delete('pongho_permits_roles', ['permit_id' => $id]);
        $this->connection->delete('pongho_permits', ['id' => $id]);
    }

    /**
     * @param array $keys
     */
    public function removePermissions(array $keys)
    {
        foreach ($keys as $key) {
            $this->removePermission($key);
        }
    }

    /**
     * @param string $key
     * @return int
     */
    private function findPermitId($key)
    {
        $key_field_name = $this->connection->quoteIdentifier('key');

        return (int) $this->connection->fetchColumn(
            'SELECT id FROM pongho_permits WHERE ' . $key_field_name . ' = :key',
            [
                'key' => $key
            ]
        );
    }
}
