<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\DependencyInjection;

use Application\Core\Mailer\MailerHelper;
use Pongho\DependencyInjection\Container;
use Pongho\DependencyInjection\ServiceProviderInterface;

/**
 * Class MailerServiceProvider
 */
class MailerServiceProvider implements ServiceProviderInterface
{
    /**
     * {@inheritdoc}
     */
    public function register(Container $container)
    {
        $container->share('mailer', array($this, 'getMailerService'));
        $container->share('mailer_transport', array($this, 'getMailerTransportService'));
        $container->share('mailer_helper', array($this, 'getMailerHelperService'));
    }

    /**
     * Restituisce il gestore delle email.
     *
     * @param Container $container
     * @return \Application\Core\Utilities\Mailer
     *
     * @internal
     */
    public function getMailerService(Container $container)
    {
        $mailer_class = $container->getParameter('mailer.mailer_class');

        /** @var \Application\Core\Utilities\Mailer $mailer */
        $mailer = new $mailer_class($container->get('site'), $container->get('language'));
        $mailer->setTransport($container->get('mailer_transport'));

        return $mailer;
    }

    /**
     * Restituisce il gestore delle email.
     *
     * @param Container $container
     * @return \Swift_Transport
     *
     * @internal
     */
    public function getMailerTransportService(Container $container)
    {
        if ($container->hasParameter('mailer.transport')) {
            $transport_callable = $container->getParameter('mailer.transport');

            if (!is_callable($transport_callable)) {
                throw new \LogicException('Parameter "mailer.transport" must be callable!');
            }

            /** @var \Swift_Transport $transport */
            $transport = call_user_func($transport_callable);
        } else {
            $class = $container->getParameter('mailer.transport_class');
            $params = $container->getParameter('mailer.transport_params');

            /** @var \Swift_Transport $transport */
            $transport = call_user_func_array(array($class, 'newInstance'), $params);
        }

        return $transport;
    }

    /**
     * @param Container $container
     * @return MailerHelper
     *
     * @internal
     */
    public function getMailerHelperService(Container $container)
    {
        return new MailerHelper(
            $container->get('site'),
            $container->get('translator')
        );
    }
}
