<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\DependencyInjection;

use Application\Core\Console\Command\ChangeDomainCommand;
use Application\Core\Export\DataExporter\UsersExporter;
use Application\Core\Export\Serializer\CsvSerializer;
use Application\Core\I18n\Locale\IntlLocale;
use Application\Core\I18n\Locale\it_IT\Locale;
use Application\Core\I18n\LocaleRegistry;
use Application\Core\Utilities\AccountInfoRenderer;
use Application\Core\Utilities\AdminUserNotificationHelper;
use Application\Core\Utilities\ChangeDomainHandlerFactory;
use Application\Core\Utilities\UserActivationLinkBuilder;
use Pongho\DependencyInjection\Container;
use Pongho\DependencyInjection\ServiceProviderInterface;

/**
 * CoreServiceProvider
 */
class CoreServiceProvider implements ServiceProviderInterface
{
    /**
     * {@inheritdoc}
     */
    public function register(Container $container)
    {
        $container->share('user_activation_link_builder', array($this, 'getUserActivationLinkBuilder'));
        $container->share('admin_user_notification_helper', array($this, 'getAdminUserNotificationHelper'));
        $container->share('account_info_renderer', array($this, 'getAccountInfoRendererService'));

        //
        // I18n
        //

        $container->share('i18n.locale_registry', function () {
            $registry = new LocaleRegistry();

            // fallback it_IT
            $registry->registerDefault(function () {
                return new Locale();
            });

            return $registry;
        });

        //
        // Admin
        //

        $container->share('admin.export_serializer', function () {
            return new CsvSerializer();
        });

        $container->share('admin.users.exporter', function (Container $container) {
            return new UsersExporter(
                $container->get('translator'),
                $container->get('connection')
            );
        });

        //
        // Console
        //

        $container->share('core.change_domain_handler', function (Container $container) {
            return (new ChangeDomainHandlerFactory($container->get('connection')))->create();
        });

        $container->extend(
            'console_commands',
            function (\ArrayObject $commands, Container $container) {
                $commands->append(new ChangeDomainCommand($container->get('core.change_domain_handler')));
            }
        );
    }

    /**
     * @return UserActivationLinkBuilder
     *
     * @internal
     */
    public function getUserActivationLinkBuilder()
    {
        return new UserActivationLinkBuilder();
    }

    /**
     * @param Container $container
     * @return AdminUserNotificationHelper
     *
     * @internal
     */
    public function getAdminUserNotificationHelper(Container $container)
    {
        return new AdminUserNotificationHelper(
            $container->get('notification_sender'),
            $container->get('notification_collection'),
            $container->get('user_activation_link_builder')
        );
    }

    /**
     * @param Container $container
     * @return AccountInfoRenderer
     *
     * @internal
     */
    public function getAccountInfoRendererService(Container $container)
    {
        return new AccountInfoRenderer($container->get('translator'));
    }
}
