<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Cron;

use Pongho\DependencyInjection\Container;
use Psr\Log\LoggerInterface;

/**
 * TaskRunner.
 */
class TaskRunner
{
    /**
     * @var TaskRepository
     */
    protected $repository;

    /**
     * @var Container
     */
    protected $container;

    /**
     * @var LoggerInterface
     */
    protected $logger;

    /**
     * @param TaskRepository  $repository
     * @param Container       $container
     * @param LoggerInterface $logger
     */
    public function __construct(TaskRepository $repository, Container $container, LoggerInterface $logger)
    {
        $this->repository = $repository;
        $this->container = $container;
        $this->logger = $logger;
    }

    /**
     * Esegue il prossimo task in coda e lo salva.
     *
     * @return bool
     */
    public function run()
    {
        $task = $this->repository->get();

        if (!$task) {
            return false;
        }

        $this->logger->info('Run the task...', ['task' => $task]);

        if ($this->handleTask($task)) {
            $this->logger->info('The task has been completed.', ['task' => $task]);
        } else {
            $this->logger->info('Tried to run the task, but has failed.', ['task' => $task]);
        }

        $this->repository->save($task);

        $this->logger->info('The task has benn saved.', ['task' => $task]);

        return true;
    }

    /**
     * Esegue il task passato.
     *
     * @param Task $task
     * @return bool
     */
    protected function handleTask(Task $task)
    {
        if ($task->getAttempts() > 5) {
            $this->logger->warning('This task has exceeded the maximum number of attempts.', ['task' => $task]);

            return false;
        }

        $task->attempt();

        $service_name = $task->getServiceName();

        if (!$this->container->has($service_name)) {
            $this->logger->critical(sprintf('The service "%s" does not exists.', $service_name), ['task' => $task]);

            return false;
        }

        $service = $this->container->get($service_name);

        if (!is_callable($service)) {
            $this->logger->critical(sprintf('The service "%s" is not a callable.', $service_name), ['task' => $task]);

            return false;
        }

        $args = $task->getData();

        try {
            if ($service($args)) {
                $task->executed();

                return true;
            } else {
                $this->logger->warning(
                    sprintf('The service "%s" must be return "true" or thrown an exception.', $service_name),
                    [
                        'task' => $task,
                    ]
                );
            }
        } catch (\Exception $e) {
            $this->logger->emergency(
                sprintf('The task "%s" throws an exception: %s', $service_name, $e->getMessage()),
                [
                    'exception' => $e,
                    'task' => $task,
                ]
            );
        }

        return false;
    }
}
