<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Cron;

use Doctrine\DBAL\Connection;

/**
 * TaskRepository
 */
class TaskRepository
{
    /**
     * @var Connection
     */
    protected $connection;

    /**
     * @param Connection $connection
     */
    public function __construct(Connection $connection)
    {
        $this->connection = $connection;
    }

    /**
     * @param string $service_name
     * @param array  $data
     * @return bool
     */
    public function create($service_name, array $data)
    {
        return (bool) $this->connection->insert(
            'pongho_cron_tasks',
            [
                'service_name' => $service_name,
                'data'         => json_encode($data),
                'created_at'   => new \DateTime(),
            ],
            [
                'service_name' => \PDO::PARAM_STR,
                'data'         => \PDO::PARAM_STR,
                'created_at'   => 'datetime',
            ]
        );
    }

    /**
     * @return Task
     */
    public function get()
    {
        $sql = <<<SQL
SELECT *
  FROM pongho_cron_tasks
 WHERE executed_at IS NULL
   AND attempts < 5
   AND (last_attempt_at IS NULL OR last_attempt_at < :attempt_date)
 ORDER BY created_at
SQL;

        $sth = $this->connection->prepare($sql);

        $sth->setFetchMode(\PDO::FETCH_CLASS, 'Application\\Core\\Cron\\Task');
        $sth->bindValue('attempt_date', new \DateTime('10 minutes ago'), 'datetime');
        $sth->execute();

        $task = $sth->fetch();

        if (!$task) {
            return null;
        }

        return $task;
    }

    /**
     * @param Task $task
     */
    public function save(Task $task)
    {
        $r = new \ReflectionObject($task);

        $pk_fields = ['id'];
        $untouchable_fields = ['id', 'service_name', 'data', 'created_at'];

        $pk = [];
        $properties = [];

        foreach ($r->getProperties() as $property) {
            $property->setAccessible(true);
            $name = $property->getName();

            if (in_array($name, $pk_fields)) {
                $pk[$name] = $property->getValue($task);
            }

            if (!in_array($name, $untouchable_fields)) {
                $properties[$name] = $property->getValue($task);
            }
        }

        $this->connection->update(
            'pongho_cron_tasks',
            $properties,
            $pk,
            [
                'id'              => \PDO::PARAM_INT,
                'attempts'        => \PDO::PARAM_INT,
                'last_attempt_at' => 'datetime',
                'executed_at'     => 'datetime',
            ]
        );
    }
}
