<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Controller;

use Application\Core\CommandBus\Command\User\Login\Exception\LoginExceptionInterface;
use Application\Core\CommandBus\Command\User\Login\LoginCommand;
use Application\Core\Controller;
use Application\Core\Entity\AccountInterface;
use Application\Core\Model\Account;
use Application\Core\Model\File;
use Application\Core\Model\Group;
use Application\Core\Model\Role;
use Pongho\FileUploader\FileUploader;
use Pongho\FileUploader\FileUploaderException;
use Pongho\Form\Generator\HtmlGenerator;
use Pongho\Form\Utilities\FormBuilder;
use Pongho\Form\Utilities\FormConfig;
use Pongho\Http\Exception\HttpNotFoundException;
use Pongho\Http\Exception\HttpUnauthorizedException;
use Pongho\Http\JsonResponse;

/**
 * Controller per la gestione degli utenti.
 */
class UserController extends Controller
{
    /**
     * @var array
     */
    protected $panels;

    /**
     * Azione "login" per l’accesso al sito.
     *
     * @return \Pongho\Http\Response
     */
    public function loginAction()
    {
        if ($this->getHelper()->getUser()->isLogged()) {
            return $this->getHelper()->redirectResponse('/');
        }

        $request = $this->getRequest();
        $session = $this->getSession();

        $errors = array();

        $redirect = null;
        if ($request->query->has('redirect')) {
            $redirect = $request->query->get('redirect');
        } elseif ($session->has('redirect')) {
            $redirect = $session->get('redirect');
            $session->del('redirect');
        }

        if ($request->getMethod() === 'POST') {
            $username_or_email = isset($_POST['username_or_email']) ? $_POST['username_or_email'] : '';
            $password = isset($_POST['password']) ? clean_password($_POST['password']) : '';
            $autologin = isset($_POST['autologin']) && $_POST['autologin'] ? true : false;

            $loginCommand = new LoginCommand($username_or_email, $password, $autologin);

            try {
                $this->getCommandBus()->handle($loginCommand);

                if ($redirect === null) {
                    $redirect = '/';
                }

                return $this->getHelper()->redirectResponse(
                    $this->getHelper()->filter($this, 'user.controller.login.filter_redirect', $redirect)
                );
            } catch (LoginExceptionInterface $e) {
                if ($request->isAjax()) {
                    return $this->getHelper()->displayJsonError(
                        $this->getHelper()->getLocalization()->get('login_error'),
                        array(
                            'errors' => $errors,
                        )
                    );
                }
            }
        } else {
            $loginCommand = new LoginCommand('', '');
        }

        $login_url = rtrim($this->getLanguagePath(), '/') . '/user/login/';

        if ($redirect) {
            $login_url .= '?' . http_build_query(array('redirect' => $redirect));
        }

        // vista
        $this->getHelper()->getTheme()
            ->setTemplate('user/login.php')
            ->assignVars(
                array(
                    'login_url'         => url($login_url),
                    'errors'            => $errors,
                    'username_or_email' => htmlspecialchars($loginCommand->getUsername()),
                )
            );

        // Intestazioni
        $this->setMetaTags(
            $this->getHelper()->getLocalization()->get('login_title'),
            $this->getHelper()->getLocalization()->get('login_meta_description')
        );

        // Evento di chiusura
        $this->getHelper()->notify($this, 'user.controller.login.after_set_view');

        return null;
    }

    /**
     * Azione "logout".
     *
     * @return \Pongho\Http\RedirectResponse
     */
    public function logoutAction()
    {
        $this->getHelper()->getUser()->logout();

        return $this->getHelper()->redirectResponse($this->getLanguagePath());
    }

    /**
     * Azione "subscribe" per l’iscrizione al sito.
     *
     * @return \Pongho\Http\Response
     */
    public function subscribeAction()
    {
        if ($this->getHelper()->getUser()->isLogged()) {
            return $this->getHelper()->redirectResponse('/');
        }

        /** @var \Psr\Log\LoggerInterface $logger */
        $logger = $this->getContainer()->get('logger');

        $site = $this->getHelper()->getSite();
        $lang = $this->getHelper()->getLocalization();

        // Impostazioni iniziali
        $default = array(
            'language_id' => $this->getLanguageId(),
            'role_id'     => Role::USER_LOGGED,
            'is_active'   => null,
            'is_founder'  => false,
            'newsletter'  => true,
            'privacy'     => false,
        );

        /** @var Account $account */
        $account = new Account($default);

        if ($this->getRequest()->getMethod() === 'POST') {
            // Account
            $account->email = isset($_POST['email']) ? htmlspecialchars(trim($_POST['email'])) : '';
            $account->username = isset($_POST['username']) ? htmlspecialchars(trim($_POST['username'])) : $account->email;

            // Password
            $password = isset($_POST['password']) ? $_POST['password'] : random(8);
            $account->password = $password;

            // Privacy
            $account->privacy = isset($_POST['privacy']) ? ($_POST['privacy'] ? true : false) : false;

            if ($this->getRequest()->isAjax()) {
                $account->privacy = true;
            }

            // Campi personali
            $account->name = isset($_POST['name']) ? htmlspecialchars(trim($_POST['name'])) : '';
            $account->surname = isset($_POST['surname']) ? htmlspecialchars(trim($_POST['surname'])) : '';
            $account->company = isset($_POST['company']) ? htmlspecialchars(trim($_POST['company'])) : '';
            $account->partita_iva = isset($_POST['partita_iva']) ? htmlspecialchars(trim($_POST['partita_iva'])) : '';
            $account->codice_fiscale = isset($_POST['codice_fiscale']) ? htmlspecialchars(trim($_POST['codice_fiscale'])) : '';

            // Indirizzo
            $account->address = isset($_POST['address']) ? htmlspecialchars(trim($_POST['address'])) : '';
            $account->address2 = isset($_POST['address2']) ? htmlspecialchars(trim($_POST['address2'])) : '';
            $account->city = isset($_POST['city']) ? htmlspecialchars(trim($_POST['city'])) : '';
            $account->postcode = isset($_POST['postcode']) ? htmlspecialchars(trim($_POST['postcode'])) : '';
            $account->province_id = isset($_POST['province']) ? intval($_POST['province']) : null;
            $account->country_id = isset($_POST['country']) ? intval($_POST['country']) : null;

            // Contatti
            $account->telephone = isset($_POST['telephone']) ? htmlspecialchars(trim($_POST['telephone'])) : '';
            $account->fax = isset($_POST['fax']) ? htmlspecialchars(trim($_POST['fax'])) : '';
            $account->website = isset($_POST['website']) ? htmlspecialchars(trim($_POST['website'])) : '';

            // Newsletter
            $account->newsletter = isset($_POST['newsletter']) && $_POST['newsletter'] ? true : false;

            // Modalità di registrazione con attivazione semplice o via e-mail
            switch ($site->getOption('subscription_mode')) {
                case 'simple':
                    $account->activate();
                    break;

                case 'email':
                    $account->resetActivationKey();
                    break;
            }

            // Eventi
            $account = $this->getHelper()->filter($this, 'user.controller.subscribe.filter_account', $account);
            $this->getHelper()->notify($this, 'user.controller.subscribe.before_save', array('account' => $account));

            // Attributi da loggare, rimuovo alcuni dati riservati quali password e chiavi di attivazione
            $attrs = $account->attributes();
            unset($attrs['password']);
            unset($attrs['activation_key']);

            $logger->info(
                '[USER] Subscribe details',
                array(
                    'subscription_mode' => $site->getOption('subscription_mode'),
                    'details'           => $attrs,
                )
            );

            if ($account->save()) {
                $logger->info(
                    '[USER] Subscribe OK',
                    array(
                        'account' => array(
                            'id'       => $account->getId(),
                            'email'    => $account->getEmail(),
                            'username' => $account->getUsername(),
                        ),
                    )
                );

                // Gruppi
                $groups = isset($_POST['groups']) ? $_POST['groups'] : array();
                foreach ($groups as $group_id) {
                    /** @var $group \Application\Core\Model\Group */
                    if (($group = Group::find($group_id)) !== null) {
                        $group->addUser($account->id);

                        $logger->debug(
                            '[USER] Adding account to group',
                            array(
                                'account' => $account->getId(),
                                'group'   => $group_id,
                            )
                        );
                    }
                }

                // Evento
                $this->getHelper()->notify($this, 'user.controller.subscribe.after_save', array('account' => $account));

                if ($account->is_active) {
                    $this->getHelper()->notify($this, 'account_activation', array('account' => $account));
                }

                /** @var \Application\Core\Notification\NotificationCollection $collection */
                $collection = $this->getContainer()->get('notification_collection');

                switch ($site->getOption('subscription_mode')) {
                    case 'simple':
                        /** @var \Application\Core\Notification\UserSubscribeSimpleNotification $notification */
                        $notification = $collection->get('core.user_subscribe_simple');
                        break;

                    case 'admin':
                        /** @var \Application\Core\Notification\UserSubscribeAdminNotification $notification */
                        $notification = $collection->get('core.user_subscribe_admin');
                        break;

                    default: // email
                        /** @var \Application\Core\Notification\UserSubscribeEmailNotification $notification */
                        $notification = $collection->get('core.user_subscribe_email');

                        $notification->setAccountActivationLink(
                            absolute_url(
                                'user/activation/?' . http_build_query([
                                    'u'   => $account->getId(),
                                    'key' => $account->getActivationKey(),
                                ])
                            )
                        );
                }

                /**
                 * @var \Application\Core\Notification\NotificationSender        $sender
                 * @var \Application\Core\Notification\UserSubscribeNotification $notification
                 */

                $notification->setAccount($account);
                $notification->setPassword(htmlspecialchars($password));
                $notification->setSettingsLink(pongho_url("/users/edit/{$account->id}/"));

                $sender = $this->getContainer()->get('notification_sender');

                try {
                    $sender->send($notification);
                } catch (\Exception $e) {
                    $logger->error('[USER] Subscribe', array('error' => $e->getMessage()));

                    if ($this->getRequest()->isAjax()) {
                        return $this->getHelper()->displayJsonError($e->getMessage());
                    }

                    $this->getHelper()->getTheme()
                        ->assignVar('errors', array($e->getMessage()));

                    return null;
                }

                // Se l'iscrizione è normale, metto subito l'utente come loggato
                if ($site->getOption('subscription_mode') === 'simple') {
                    $this->getHelper()->getUser()->forceLogin($account);
                }

                // Messaggio di registrazione avvenuta
                switch ($site->getOption('subscription_mode')) {
                    case 'simple':
                        $subscribe_complete_message = $lang->get('valid_subscribe_default_activation'); // TODO: Convertire con "valid_subscribe_simple_activation"
                        break;

                    case 'admin':
                        // TODO: Update per inserire la traduzione "valid_subscribe_admin_activation"
                        $subscribe_complete_message = $lang->get('valid_subscribe_admin_activation');
                        break;

                    default:
                        // TODO: Capire perché viene passato il nome del sito e se si può rimuovere
                        $subscribe_complete_message = sprintf(
                            $lang->get('valid_subscribe_email_activation'),
                            $site->getName()
                        );

                        break;
                }

                // Iscrizione in AJAX
                if ($this->getRequest()->isAjax()) {
                    return $this->getHelper()->displayMessage($subscribe_complete_message);
                }

                $this->getHelper()->getTheme()
                    ->assignVars([
                        'valid'                        => true,
                        'l_subscribe_complete_message' => $subscribe_complete_message,
                    ]);
            } else {
                $errors = $this->translateErrors($account->errors);

                if (!isset($_POST['username'])) {
                    unset($errors['username']);
                }

                $logger->notice(
                    '[USER] Subscribe FAIL',
                    array(
                        'errors' => $errors,
                    )
                );

                // Richiesta AJAX
                if ($this->getRequest()->isAjax()) {
                    return $this->getHelper()->displayJsonError(
                        $lang->get('error_title', count($errors)),
                        [
                            'errors' => $errors,
                        ]
                    );
                }

                $this->getHelper()->getTheme()
                    ->assignVar('errors', $errors);
            }
        }

        $account = $this->filterAccount($account);

        $this->getHelper()->getTheme()
            ->setTemplate('user/subscribe.php')
            ->assignVar('account', $account);

        // Intestazioni
        $this->setMetaTags(
            $lang->get('subscribe_title'),
            $lang->get('subscribe_meta_description')
        );

        // Evento di chiusura
        $this->getHelper()->notify($this, 'user.controller.subscribe.after_set_view', array('account' => $account));

        return null;
    }

    /**
     * Azione "activation" per l’attivazione del profilo utente.
     *
     * @return \Pongho\Http\Response
     */
    public function activationAction()
    {
        $key = isset($_GET['key']) ? htmlspecialchars(trim($_GET['key'])) : '';
        $user_id = isset($_GET['u']) ? intval($_GET['u']) : 0;

        /** @var \Application\Core\Model\Account $account */
        $account = Account::first(
            array(
                'conditions' => array('id = ? AND activation_key = ?', $user_id, $key)
            )
        );

        if (is_null($account)) {
            return $this->getHelper()->displayMessage($this->getHelper()->getLocalization()->get('activation_failed'));
        } else {
            if ($account->is_active === null) {
                $account->activate();
                $account->resetActivationKey();
                $account->save(false);

                $this->getHelper()->notify($this, 'account_activation', array('account' => $account));
            }

            $this->getHelper()->getTheme()
                ->assignVars(
                    array(
                        'failed'  => false,
                        'message' => $this->getHelper()->getLocalization()->get('valid_user_activation'),
                        'account' => $account,
                    )
                );

            // Login automatico
            $this->getSession()
                ->set('user_id', $account->id);
        }

        $this->getHelper()->getTheme()->setTemplate('user/activation.php');

        // Evento di chiusura
        $this->getHelper()->notify($this, 'user.controller.activation.after_set_view', array('account' => $account));

        return null;
    }

    /**
     * Azione "settings" per la modifica del profilo utente.
     *
     * @return \Pongho\Http\Response
     */
    public function settingsAction()
    {
        if (!$this->getHelper()->getUser()->isLogged()) {
            $redirect = '/user/login/?' . http_build_query(array('redirect' => '/user/settings/'));

            return $this->getHelper()->redirectResponse($redirect);
        }

        /** @var Account $account */
        $account = $this->getHelper()->getUser()->getAccount();

        // Carico i gruppi
        $groups_options = $this->getHelper()->filter(
            $this,
            'user.controller.settings.filter_groups_options',
            array('order' => 'name')
        );

        /** @var Group[] $groups */
        $groups = Group::all($groups_options);

        if ($this->getRequest()->getMethod() === 'POST') {
            // Account
            $account->username = isset($_POST['username']) ? htmlspecialchars(trim($_POST['username'])) : $account->username;
            $account->email = isset($_POST['email']) ? htmlspecialchars(trim($_POST['email'])) : $account->email;

            // Password
            $password = isset($_POST['password']) ? trim($_POST['password']) : '';
            $confirm_password = isset($_POST['confirm_password']) ? trim($_POST['confirm_password']) : '';

            if (!empty($password)) {
                $account->password = $password;
                $account->confirm_password = $confirm_password;

                if ($password === $confirm_password) {
                    $this->getHelper()->notify(
                        $this,
                        'user.controller.settings.change_password',
                        array('account' => $account, 'new_password' => $password)
                    );
                }
            }

            // Campi personali
            $account->name = isset($_POST['name']) ? htmlspecialchars(trim($_POST['name'])) : $account->name;
            $account->surname = isset($_POST['surname']) ? htmlspecialchars(trim($_POST['surname'])) : $account->surname;
            $account->company = isset($_POST['company']) ? htmlspecialchars(trim($_POST['company'])) : $account->company;
            $account->partita_iva = isset($_POST['partita_iva']) ? htmlspecialchars(trim($_POST['partita_iva'])) : $account->partita_iva;
            $account->codice_fiscale = isset($_POST['codice_fiscale']) ? htmlspecialchars(trim($_POST['codice_fiscale'])) : $account->codice_fiscale;

            // Indirizzo
            $account->address = isset($_POST['address']) ? htmlspecialchars(trim($_POST['address'])) : $account->address;
            $account->address2 = isset($_POST['address2']) ? htmlspecialchars(trim($_POST['address2'])) : $account->address;
            $account->city = isset($_POST['city']) ? htmlspecialchars(trim($_POST['city'])) : $account->city;
            $account->postcode = isset($_POST['postcode']) ? htmlspecialchars(trim($_POST['postcode'])) : $account->postcode;
            $account->province_id = isset($_POST['province']) && is_numeric($_POST['province']) ? intval($_POST['province']) : $account->province_id;
            $account->country_id = isset($_POST['country']) && is_numeric($_POST['country']) ? intval($_POST['country']) : $account->country_id;

            // Contatti
            $account->telephone = isset($_POST['telephone']) ? htmlspecialchars(trim($_POST['telephone'])) : $account->telephone;
            $account->fax = isset($_POST['fax']) ? htmlspecialchars(trim($_POST['fax'])) : $account->fax;
            $account->website = isset($_POST['website']) ? htmlspecialchars(trim($_POST['website'])) : $account->website;

            // Newsletter
            $account->newsletter = isset($_POST['newsletter']) && $_POST['newsletter'] ? true : false;

            // Gruppi
            if (isset($_POST['groups_enabled']) && $_POST['groups_enabled']) {
                $user_groups = (isset($_POST['groups']) && is_array($_POST['groups'])) ? $_POST['groups'] : array();
                $account->setGroups(array_keys($user_groups));
            }

            // Evento per filtrare l’account
            $account = $this->getHelper()->filter($this, 'user.controller.settings.filter_account', $account);
            $this->getHelper()->notify($this, 'user.controller.settings.before_save', array('account' => $account));

            // Salvataggio
            if ($account->save()) {
                // Evento
                $this->getHelper()->notify($this, 'user.controller.settings.after_save', array('account' => $account));

                $this->getHelper()->getTheme()
                    ->assignVar('valid', true);
            } else {
                $this->getHelper()->getTheme()
                    ->assignVar('errors', $this->translateErrors($account->errors));
            }
        }

        // Aggiungo il check sui gruppi
        $user_groups = $account->getGroups();

        foreach ($groups as $group) {
            $group->is_checked = array_key_exists($group->id, $user_groups);
        }

        $account = $this->filterAccount($account);

        // Vista
        $this->getHelper()->getTheme()
            ->setTemplate('user/settings.php')
            ->assignVars(
                array(
                    'groups'  => $groups,
                    'account' => $account,
                )
            );

        // Intestazioni
        $this->getHelper()->getHead()
            ->setTitle($this->getHelper()->getLocalization()->get('user_settings_title'));

        // Evento di chiusura
        $this->getHelper()->notify($this, 'user.controller.settings.after_set_view', array('account' => $account));

        return null;
    }

    /**
     * Azione "forgotpassword" per iniziare la procedura di recupero della password.
     */
    public function forgotpasswordAction()
    {
        /** @var \Psr\Log\LoggerInterface $logger */
        $logger = $this->getContainer()->get('logger');

        $lang = $this->getHelper()->getLocalization();
        $theme = $this->getHelper()->getTheme();

        $theme->setTemplate('user/forgotpassword.php');

        $errors = [];

        if ($this->getRequest()->getMethod() === 'POST') {
            // L'unico campo in arrivo via POST è l'indirizzo e-mail
            $email = isset($_POST['email']) ? $_POST['email'] : '';

            $logger->info(
                '[USER] Forgot password',
                array(
                    'email' => $email,
                )
            );

            /** @var Account $account */
            $account = null;

            // Controllo la validità
            if (empty($email)) {
                $errors['email'] = $lang->get('required_email');
            } else {
                if (!preg_match(REGEXP_VALIDATE_EMAIL, $email)) {
                    $errors['email'] = $lang->get('invalid_email_format');
                } else {
                    // Cerco l'account in base all'indirizzo e-mail
                    $account = Account::findByEmail($email);

                    // Se non trovo nessun account segnalo l'errore
                    if (is_null($account)) {
                        $errors['email'] = sprintf($lang->get('no_account_with_email'), $email);
                    } elseif (!$account->is_active) {
                        $errors['email'] = sprintf($lang->get('forgot_password_user_not_active'), $email);
                    }
                }
            }

            if (empty($errors)) {
                // Genero una chiave per il reset della password e la salvo nell'account
                $reset_password_key = random(32);
                $account->reset_password_key = $reset_password_key;
                $account->save(false);

                /** @var \Application\Core\Notification\NotificationCollection $collection */
                $collection = $this->getContainer()->get('notification_collection');

                /** @var \Application\Core\Notification\ForgotPasswordNotification $notification */
                $notification = $collection->get('core.user_forgot_password');

                $notification->setAccount($account);
                $notification->setResetPasswordLink(absolute_url("/user/resetpassword/{$reset_password_key}/"));

                /** @var \Application\Core\Notification\NotificationSender $sender */
                $sender = $this->getContainer()->get('notification_sender');
                try {
                    $sender->send($notification);
                } catch (\Exception $e) {
                    $logger->error('[USER] Forgot password', array('error' => $e->getMessage()));

                    if ($this->getRequest()->isAjax()) {
                        return $this->getHelper()->displayJsonError($e->getMessage());
                    }

                    $theme->assignVar('errors', array($e->getMessage()));

                    return null;
                }

                // Vista
                $theme->assignVar('valid', true);
            } else {
                $logger->notice(
                    '[USER] Forgot password',
                    array(
                        'errors' => $errors
                    )
                );

                // Richiesta AJAX
                if ($this->getRequest()->isAjax()) {
                    return $this->getHelper()->displayJsonError(
                        $lang->get('error_title', count($errors)),
                        [
                            'errors' => $errors,
                        ]
                    );
                }
            }
        }

        $theme->assignVar('errors', $errors);

        // Intestazioni
        $this->setMetaTags(
            $lang->get('forgot_password_title'),
            $lang->get('forgot_password_meta_description')
        );

        // Evento di chiusura
        $this->getHelper()->notify($this, 'user.controller.forgotpassword.after_set_view');

        return null;
    }

    /**
     * Azione "resetpassword" per reimpostare la password.
     *
     * Quando l’utente dimentica la password per il proprio account deve richiederne la modifica attraverso
     * `forgotpassword`. Gli viene inviata una e-mail con il collegamento a questa azione che gestirà la modifica
     * e quindi il ripristino.
     *
     * @return \Pongho\Http\Response
     * @throws HttpNotFoundException
     */
    public function resetpasswordAction()
    {
        $reset_key = $this->getParameter('id');

        if (empty($reset_key)) {
            throw new HttpNotFoundException();
        }

        if (strlen($reset_key) != 32) {
            return $this->getHelper()->displayError(
                'La chiave di riattivazione della tua password non è valida!<br />Prova a copiare ed incollare il link della e-mail che ti abbiamo inviato.'
            );
        }

        /**
         * Cerco l'account in base alla chiave di reset della password
         *
         * @var Account $account
         */
        $account = Account::findByResetPasswordKey($reset_key);

        if ($account === null) {
            return $this->getHelper()->displayError(
                'La chiave di riattivazione della tua password non è valida, oppure è già stata usata!'
            );
        }

        if ($this->getRequest()->getMethod() === 'POST') {
            // Password
            $password = isset($_POST['password']) ? trim($_POST['password']) : '';
            $confirm_password = isset($_POST['confirm_password']) ? trim($_POST['confirm_password']) : '';

            if (empty($password)) {
                $account->addError('password', 'required_password');
            }

            if (empty($confirm_password)) {
                $account->addError('confirm_password', 'required_confirm_password');
            }

            if ($password != $confirm_password) {
                $account->addError('confirm_password', 'password_mismatch');
            }

            $account->password = $password;
            $account->confirm_password = $confirm_password;

            if ($account->errors->is_empty() && $account->save(false)) {
                $account->reset_password_key = '';
                $account->save(false);

                $this->getHelper()->getTheme()->assignVar('valid', true);
            } else {
                $this->getHelper()->getTheme()->assignVar('errors', $this->translateErrors($account->errors));
            }
        }

        $account = $this->filterAccount($account);

        $this->getHelper()->getTheme()
            ->setTemplate('user/resetpassword.php')
            ->assignVars(
                array(
                    'url_resetpassword' => url(sprintf('user/resetpassword/%s/', $reset_key)),
                    'account'           => $account,
                )
            );

        // Intestazioni
        $this->setMetaTags(
            $this->getHelper()->getLocalization()->get('reset_password_title'),
            $this->getHelper()->getLocalization()->get('reset_password_meta_description')
        );

        // Evento di chiusura
        $this->getHelper()->notify($this, 'user.controller.resetpassword.after_set_view', array('account' => $account));

        return null;
    }

    /**
     * Azione "uploadavatar".
     *
     * Permette di caricare un avatar per il proprio profilo utente tramite le nuove funzionalità di HTML5
     * ed in particolare con il codice JS `avatar-upload.js`.
     *
     * @return JsonResponse
     * @throws HttpUnauthorizedException
     * @throws HttpNotFoundException
     */
    public function uploadavatarAction()
    {
        if (!$this->getHelper()->getUser()->isLogged()) {
            throw new HttpUnauthorizedException();
        }

        if ($this->getRequest()->getMethod() === 'POST' && $this->getRequest()->files->has('avatar')) {
            $avatar = $this->getRequest()->files->get('avatar');

            /** @var Account $account */
            $account = $this->getHelper()->getUser()->getAccount();
            $file = $account->getUserDetails()->getAvatar();

            $old_avatar_filepath = $file ? $file->filepath() : null;

            try {
                $uploader = new FileUploader($avatar);
                $uploader
                    ->setPath($this->getHelper()->getSite()->getUploadsPath('avatars'))
                    ->setAllowedExtensions(array('jpg', 'jpeg', 'gif', 'png'))
                    ->setMaxFileSize(102400)// 100 KB
                    ->setFileName($this->getHelper()->getUser()->getAccount()->getId() . '-' . time())
                    ->upload();
            } catch (FileUploaderException $e) {
                return $this->getHelper()->displayJsonError(
                    $this->getHelper()->getLocalization()->get($e->getMessage())
                );
            }

            // Attributi del file da salvare in database
            $file_attributes = $uploader->getFileData();
            $file_attributes['site_id'] = $this->getSiteId();
            $file_attributes['path'] = substr($file_attributes['path'], strlen($this->getHelper()->getSite()->getUploadsPath()));

            $file_attributes = $this->getHelper()->filter(
                $this,
                'user.avatar.filter_file_attributes',
                $file_attributes,
                array('uploader' => $uploader)
            );

            /**
             * @var File $file
             */

            if ($file === null) {
                $file = File::create($file_attributes);
                $account->getUserDetails()->setAvatar($file)->save();
            } else {
                if (file_exists($old_avatar_filepath)) {
                    unlink($old_avatar_filepath);
                }

                $file->updateAttributes($file_attributes);
                $file->save();
            }

            // Devo restituire un JSON con i dati dell’avatar
            $sizes = $this->getHelper()->getTheme()->getOption('image_sizes', array());
            if (array_key_exists('avatar-settings', $sizes)) {
                $avatar_size = $sizes['avatar-settings'];
            } else {
                $avatar_size = array(100, 100, 'fill');
            }

            return new JsonResponse(array(
                array(
                    'src'          => $this->getHelper()->filter($this, 'user.avatar.filter_file', src($file, 'avatar-settings')),
                    'width'        => $avatar_size[0],
                    'height'       => $avatar_size[1],
                    'delete_url'   => url('/user/deleteavatar/'),
                    'delete_label' => $this->getHelper()->getLocalization()->get('delete'),
                )
            ));
        }

        throw new HttpNotFoundException();
    }

    /**
     * Azione "deleteavatar" per eliminare il proprio avatar.
     *
     * @return \Pongho\Http\Response
     * @throws HttpUnauthorizedException
     */
    public function deleteavatarAction()
    {
        if (!$this->getHelper()->getUser()->isLogged()) {
            throw new HttpUnauthorizedException();
        }

        if ($this->getHelper()->getUser()->getAccount()->getUserDetails()->deleteAvatar()) {
            if ($this->getRequest()->isAjax()) {
                return $this->getHelper()->displayJsonMessage(
                    $this->getHelper()->getLocalization()->get('avatar_deleted')
                );
            } else {
                return $this->getHelper()->redirectResponse('/user/settings/');
            }
        } else {
            return $this->getHelper()->displayMessage(
                $this->getHelper()->getLocalization()->get('could_not_delete_avatar')
            );
        }
    }

    /**
     * @param string $method
     * @param array  $args
     * @return null|bool|\Pongho\Http\Response
     */
    public function __call($method, array $args)
    {
        if (substr($method, -6) === 'Action') {
            $page = substr($method, 0, -6);

            if ($this->getHelper()->getSite()->getOption('account_url')
                && in_array($page, $this->getPublicPages())
                && $this->hasParameter('account')) {
                return $this->displayPublicPage($page);
            }

            if (in_array($page, $this->getEditPages())) {
                return $this->displayEditPage($page);
            }
        }

        return parent::__call($method, $args);
    }

    /**
     * Filtra l’account da passare alla vista.
     *
     * @param AccountInterface $account
     * @return Account
     */
    protected function filterAccount(AccountInterface $account)
    {
        return $this->getHelper()->filter($this, 'user.controller.display_page.filter_account', $account);
    }

    /**
     * @return array
     */
    protected function getPublicPages()
    {
        return $this->getHelper()->filter($this, 'user.controller.filter_public_pages', array('view'));
    }

    /**
     * Visualizza una pagina pubblica del profilo.
     *
     * @param string $page_name
     * @return null
     * @throws HttpNotFoundException
     */
    protected function displayPublicPage($page_name)
    {
        $account = $this->getParameter('account', null, true);

        $this->getHelper()->notify($this, 'user.controller.display_public_page', array('page_name' => $page_name));

        $this->getHelper()->getTheme()
            ->setTemplate("user/pages/{$page_name}.php")
            ->assignVar('account', $this->filterAccount($account));

        return null;
    }

    /**
     * @return array
     */
    protected function getEditPages()
    {
        return $this->getHelper()->filter($this, 'user.controller.filter_edit_pages', array());
    }

    /**
     * Visualizza una pagina privata per la modifica dei dati del profilo
     *
     * @param string $page_name
     * @return null|\Pongho\Http\Response
     */
    protected function displayEditPage($page_name)
    {
        if (!$this->getHelper()->getUser()->isLogged()) {
            $redirect = '/user/login/?' . http_build_query(array('redirect' => "/user/{$page_name}/"));

            return $this->getHelper()->redirectResponse($redirect);
        }

        $this->getHelper()->notify($this, 'user.controller.display_edit_page', array('page_name' => $page_name));

        $this->getHelper()->getTheme()
            ->setTemplate("user/pages/{$page_name}.php")
            ->assignVar('account', $this->filterAccount($this->getHelper()->getUser()->getAccount()));

        return null;
    }

    /**
     * Carica e gestisce la form.
     *
     * @return \Pongho\Http\Response
     * @throws HttpNotFoundException
     */
    public function editpanelAction()
    {
        $request = $this->getRequest();
        $panel = $request->query->get('p', null);

        if ($panel === null || !array_key_exists($panel, $this->getPanels())) {
            throw new HttpNotFoundException();
        }

        $config = $this->getPanelOptions($panel);

        if (is_callable($config)) {
            $config = call_user_func($config, $this->container);
        }

        if ($config instanceof FormConfig) {
            /** @var \Pongho\Form\Form $form */
            $form = FormBuilder::buildConfig($config);
            $form->setAction($this->getRequest()->getRequestUri());

            $generator = new HtmlGenerator($this->getHelper()->getLocalization());

            $generator = $this->getHelper()->filter(
                $this,
                'user.controller.editpanel.filter_generator',
                $generator,
                array('panel' => $panel)
            );

            $generator = $this->getHelper()->filter(
                $this,
                'user.controller.editpanel_' . $panel . '.filter_generator',
                $generator
            );

            $form->setGenerator($generator);
        } else {
            $form = FormBuilder::build($panel, $config);
            $form->afterAddFieldToParent();
        }

        if ($request->getMethod() === 'POST') {
            $form->handleRequest($request);

            if (!$form->hasErrors()) {
                $account = $this->getHelper()->getUser()->getAccount();

                $template_path = '/user/panels/' . $panel . '.php';

                $template_path = $this->getHelper()->filter(
                    $this,
                    'user.controller.editpanel.filter_template_path',
                    $template_path,
                    array('panel' => $panel)
                );

                $template_path = $this->getHelper()->filter(
                    $this,
                    'user.controller.editpanel_' . $panel . '.filter_template_path',
                    $template_path
                );

                $view = $this->getHelper()->getTheme()
                    ->setTemplate($template_path)
                    ->assignVars(
                        array(
                            'account' => $this->filterAccount($account),
                            'model'   => $form->getSubject()->getModel(),
                        )
                    );

                // Notifiche
                $parameters = array(
                    'account' => $account,
                    'panel'   => $panel,
                    'form'    => $form,
                    'view'    => $view,
                );

                $this->getHelper()->notify($this, 'user.controller.editpanel.after_save', $parameters);
                $this->getHelper()->notify($this, 'user.controller.editpanel_' . $panel . '.after_save', $parameters);

                return new JsonResponse(array(
                    'errors' => false,
                    'html'   => $view->render(),
                ));
            }
        }

        $form = $this->getHelper()->filter($this, 'user.controller.editpanel_' . $panel . '.filter_form', $form);

        $json = $this->getHelper()->filter(
            $this,
            'user.controller.editpanel_' . $panel . '.filter_json',
            array(
                'errors' => $form->hasErrors(),
                'html'   => $form->render(),
            ),
            array('form' => $form)
        );

        return new JsonResponse($json);
    }

    /**
     * @return array
     * @throws \LogicException
     */
    protected function getPanels()
    {
        if ($this->panels === null) {
            $panels = $this->getHelper()->filter($this, 'user.controller.filter_panels_list', []);

            if (!is_array($panels)) {
                throw new \LogicException('The user panels list must be an array.');
            }

            $this->panels = $panels;
        }

        return $this->panels;
    }

    /**
     * @param string $panel
     * @return array
     * @throws \InvalidArgumentException
     */
    protected function getPanelOptions($panel)
    {
        $panels = $this->getPanels();

        if (!array_key_exists($panel, $panels)) {
            throw new \InvalidArgumentException(sprintf('Panel "%s" not exists.', $panel));
        }

        $options = $panels[$panel];

        if (!(is_callable($options) || $options instanceof FormConfig)) {
            $options['action'] = $this->getRequest()->getRequestUri();
            $options['method'] = 'POST';
        }

        return $options;
    }

    /**
     * @param string $title
     * @param string $description
     */
    protected function setMetaTags($title, $description)
    {
        $headHelper = $this->getHelper()->getThemeHeaderHelper();
        $headHelper->setTitle($title);
        $headHelper->metatags->add('description', $description);
        $headHelper->links->add(absolute_lang_url('/user/' . $this->getAction() . '/'), array('rel' => 'canonical'));
    }

    /**
     * @return \League\Tactician\CommandBus
     */
    private function getCommandBus()
    {
        return $this->container->get('command_bus');
    }
}
