<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Controller;

use Application\Core\Controller;
use Pongho\Http\Exception\HttpNotFoundException;
use Pongho\Imager\Imager;
use Pongho\Imager\ImagerInterface;

/**
 * Controller per la gestione delle immagini.
 */
class ImagesController extends Controller
{
    /**
     * Gestisce l’azione `display`.
     *
     * Carica la versione originale dell’immagine richiesta, la ridimensiona e la invia al browser.
     * In ambiente di produzione crea anche una copia su fylesystem dell’immagine ridimensionata in
     * modo da non doverla ricreare ogni volta.
     */
    public function displayAction()
    {
        $sizes = $this->getHelper()->getTheme()->getOption('image_sizes', []);

        if (empty($sizes)) {
            throw new \RuntimeException('Parameter "image_sizes" does not exist!');
        }

        $default = [
            'avatar-settings' => [100, 100, 'fill']
        ];

        $sizes = array_merge($default, $sizes);

        $imageRelativePath = $this->getParameter('image_path');
        $regex = '#^(/.*?)/([^/]+)\-(' . implode('|', array_map('preg_quote', array_keys($sizes))) . ').(jpg|jpeg|png|gif)$#';
        if (!preg_match($regex, $imageRelativePath, $matches)) {
            throw new HttpNotFoundException();
        }

        $path = $matches[1];
        $filename = $matches[2] . '.' . $matches[4];
        $size = $sizes[$matches[3]];

        $options = [
            'background'         => null,
            'watermark'          => null,
            'watermark_position' => ImagerInterface::WATERMARK_CENTER,
        ];

        switch (count($size)) {
            case 3:
                list($width, $height, $type) = $size;
                break;

            case 4:
                list($width, $height, $type, $opts) = $size;

                if (is_string($opts)) {
                    $options['background'] = $opts;
                } elseif (is_array($opts)) {
                    $options = array_merge($options, $opts);
                } else {
                    throw new \RuntimeException(sprintf(
                        'Parameter "image_sizes.%s" in themes is not valid: expected an array for the 4th parameter.',
                        $matches[3]
                    ));
                }

                break;

            default:
                throw new \RuntimeException(sprintf(
                    'Parameter "image_sizes.%s" in themes is not valid: expected 3 or 4 parameters!',
                    $matches[3]
                ));
        }

        $site = $this->getHelper()->getSite();
        $sourceFile = $site->getUploadsPath($path . '/' . $filename);

        $imager = Imager::load($sourceFile);

        if ($options['background']) {
            // "#FFFFFF" -> "FFFFFF"
            $background = ltrim($options['background'], '#');

            $imager->background($background);
        }

        if ($options['watermark']) {
            $imager->addWatermark($options['watermark'], $options['watermark_position']);
        }

        $imager->resize($width, $height, $type);

        if ($this->getHelper()->getEnvironment() === 'prod') {
            create_path($site->getImagesPath($path));
            $destinationFile = $site->getImagesPath($imageRelativePath);
            $imager->save($destinationFile);
        }

        return $imager->show();
    }
}
