<?php

namespace Application\Core\Controller\Admin;

use Application\Admin\Controller\AdminController;
use Application\Core\Model\Translation;
use Pongho\Http\Response;
use Pongho\Http\Exception\HttpNotFoundException;

class TranslationsController extends AdminController
{
    protected $languages;
    protected static $index_tab = 'search';
    protected $application;

    protected function getPartitions()
    {
        $partitions = $this->filter(
            'admin.translations.filter_partitions',
            array('theme', 'core', 'admin', 'messages', 'user')
        );

        $partitions = array_merge(array('search'), $partitions);

        return $partitions;
    }

    /**
     * @return \Application\Core\Model\LanguageSite[]|\Application\Core\Model\Language[]
     */
    protected function getLanguages()
    {
        if ($this->languages === null) {
            foreach ($this->getSite()->getLanguages() as $language_site) {
                /** @var \Application\Core\Model\LanguageSite|\Application\Core\Model\Language $language_site */
                $language_site->flag = sprintf(
                    '<img class="flag" src="%s" alt="%s" title="%s" />',
                    pongho_url("/Application/Core/Resources/flags/16/{$language_site->culture}.png"),
                    $language_site->iso,
                    $language_site->name
                );

                $this->languages[$language_site->language_site_id] = $language_site;
            }
        }

        return $this->languages;
    }

    /**
     * @param array $translations
     * @return array
     */
    protected function prepareTranslations(array $translations)
    {
        $_translations = array();

        $languages = $this->getLanguages();

        /** @var \Application\Core\Model\Translation $translation */

        // Preparo la struttura
        foreach ($translations as $translation) {
            $_translations[$translation->key] = array(
                'application'  => $translation->application,
                'translations' => array(),
            );
        }

        // Inserisco le traduzioni
        foreach ($translations as $translation) {
            if (isset($languages[$translation->language_site_id])) {
                $_translations[$translation->key]['translations'][$translation->language_site_id] = $translation->attributes();
            }
        }

        // Inserisco le traduzioni mancanti e gli elementi accessori (bandierine...)
        foreach ($languages as $language) {
            foreach ($_translations as $key => $_translation) {
                if (!isset($_translation['translations'][$language->language_site_id])) {
                    $_translations[$key]['translations'][$language->language_site_id] = array(
                        'key'              => $key,
                        'language_site_id' => $language->language_site_id,
                        'application'      => $_translation['application'],
                        'value'            => '',
                    );
                }

                $_translations[$key]['translations'][$language->language_site_id]['flag'] = $language->flag;
            }
        }

        return $_translations;
    }

    /**
     * Aggiorna, elimina o aggiunge nuove traduzioni
     *
     * @param array $translations
     * @return array
     */
    public function handleTranslations(array $translations)
    {
        if ($this->getRequest()->getMethod() === 'POST') {
            $lang = $this->container->get('language');
            $errors = array();
            $errors_translations = array();

            $languages = $this->getLanguages();

            /**
             * Aggiornamento delle traduzioni esistenti
             */
            foreach ($this->getRequest()->post->get('translations', array()) as $key => $translations_array) {
                // Se impostata, aggiorna l'applicazione della chiave
                if (isset($translations_array['application'])) {
                    $translations[$key]['application'] = $translations_array['application'];
                }

                $application = $translations[$key]['application'];

                /*  se ho segnato una chiave come "da cancellare", cancello tutte le traduzioni ad essa associate per il sito corrente */
                if (isset($translations_array['delete'])) {
                    /** @var \Application\Core\Model\Translation $translation */
                    foreach (Translation::findAllByKeyAndSite($key, $this->getSiteId()) as $translation) {
                        $translation->delete();
                    }

                    unset($translations[$translation->key]);
                } else {
                    foreach ($translations_array['translations'] as $language_site_id => $value) {
                        $attributes = array(
                            'key'              => $key,
                            'language_site_id' => $language_site_id,
                            'value'            => $value,
                        );

                        if ($this->application === 'index') {
                            // L'applicazione è definita dalla select
                            $attributes['application'] = $application;
                        } else {
                            // L'applicazione è definita dalla tab
                            $attributes['application'] = $this->application;
                        }

                        if (!empty($value)) {
                            if (($translation = Translation::find(array($key, $language_site_id))) === null) {
                                // qui gestisce l'inserimento di nuove traduzioni per una chiave già esistente
                                $translation = Translation::create($attributes);
                            } else {
                                $translation->updateAttributes($attributes);
                                $translation->save();
                            }

                            $translations[$key]['translations'][$language_site_id] = $translation->attributes();

                        } else {
                            // Se ho svuotato una chiave o non l'ho compilata, provo a cercare l'eventuale traduzione ed eliminarla
                            if (($translation = Translation::find(array($key, $language_site_id))) !== null) {
                                $translation->delete();
                            }

                            $translations[$key]['translations'][$language_site_id]['value'] = '';
                            $translations[$key]['translations'][$language_site_id]['application'] = $attributes['application'];
                        }

                        $translations[$key]['translations'][$language_site_id]['flag'] = $languages[$language_site_id]->flag;
                    }

                    $translations[$key]['application'] = $application;
                }
            }

            /**
             * Aggiunta di nuove traduzioni
             *
             * L'aggiunta è un po' particolare perché la chiave viene inserita da input e non è più nel nome del campo, inoltre,
             * essendo la chiave testuale e non un id numerico, non posso controllare semplicemente con is_numeric(), quindi
             * devo tener un progressivo al posto della chiave (new, new_1, new2... che posso tranquillamente scartare) altrimenti
             * le posizioni dell'array vengono sovrascritte e salvare la chiave come valore all'interno dell'array.
             */
            foreach ($this->getRequest()->post->get('newtranslations', array()) as $i => $key) {
                /* salto le righe con chiave vuota */
                if (!empty($key['key'])) {
                    /* controllo se la chiave esiste già in database per questo sito */
                    if (($translation = Translation::findIfKeyInSite($key['key'], $this->getSiteId())) === null) {
                        foreach ($key['languages'] as $language_site_id => $value) {
                            $attributes = array(
                                'key'              => $key['key'],
                                'language_site_id' => $language_site_id,
                                'application'      => $this->application,
                                'value'            => $value,
                            );

                            /* se il valore è vuoto, non salvo la riga */
                            if (!empty($value)) {
                                Translation::create($attributes);
                            }

                            $translations[$key['key']]['translations'][$language_site_id] = $attributes;
                            $translations[$key['key']]['translations'][$language_site_id]['flag'] = $languages[$language_site_id]->flag;
                        }
                    } else {
                        $errors[] = sprintf(
                            $lang->key_exists,
                            $translation->key,
                            $lang->get($translation->application)
                        );
                        $errors_translations[$i][$translation->key] = $key['languages'];
                    }
                }
            }

            /* rimetto l'array in ordine alfabetico per chiave */
            ksort($translations);

            $this->getBodyView()
                ->assignVars(
                    array(
                        'message'             => $this->getLocalization()->get('translations_updated'),
                        'errors_translations' => $errors_translations,
                        'errors'              => $errors,
                    )
                );
        }

        return $translations;
    }

    /**
     *  Mostra la pagina iniziale della ricerca
     */
    public function indexAction()
    {
        $this->application = $this->getAction();

        $translations = array();

        $query = $this->getRequest()->post->get('query', '');

        if ($query) {
            $translations = $this->prepareTranslations(Translation::search($query, $this->getSiteId()));
        }

        if ($this->getRequest()->getMethod() === 'POST' && !$this->getRequest()->post->has('search')) {
            $translations = $this->handleTranslations($translations);
        }

        // Select dell'applicazione
        $partitions = $this->getPartitions();

        $applications = array();
        foreach ($partitions as $application) {
            if ($application !== 'search') {
                $applications[$application] = $this->getLocalization()->get($application);
            }
        }

        $this->getBodyView()
            ->setTemplatePath(__DIR__ . '/../../Resources/views/translations.php')
            ->assignVars(
                array(
                    'action_form'  => $this->url('/' . $this->getParameter('path') . '/' . ($query ? '?query=' . $query : '')),
                    'translations' => $translations,
                    'languages'    => $this->getLanguages(),
                    'application'  => $this->application,
                    'tabs'         => $this->getTabs(),
                    'query'        => $query,
                    'applications' => $applications,
                )
            );

        $this->getHead()
            ->addCss(pongho_url('/Application/Admin/Resources/views/css/style-legacy.css?v=' . filemtime(__DIR__ . '/../../../Admin/Resources/views/css/style-legacy.css')))
            ->addJavascript(pongho_url('/Application/Core/Resources/views/js/jquery.translations.js?v=' . filemtime(__DIR__ . '/../../Resources/views/js/jquery.translations.js')));
    }

    /**
     * Gestisce le applicazioni dinamiche
     *
     * @param string $method
     * @param array  $arguments
     * @return bool|void
     * @throws \Pongho\Http\Exception\HttpNotFoundException
     */
    public function __call($method, array $arguments)
    {
        $action = $this->getAction();
        $this->application = $action;

        if (!in_array($this->application, $this->getPartitions())) {
            throw new HttpNotFoundException;
        }

        // Modifica / Aggiunta / Cancellazione
        $translations = $this->prepareTranslations(Translation::findAllBySiteAndApplication($this->getSiteId(), $this->application));

        if ($this->getRequest()->getMethod() === 'POST') {
            $translations = $this->handleTranslations($translations);
        }

        $this->getBodyView()
            ->setTemplatePath(__DIR__ . '/../../Resources/views/translations.php')
            ->assignVars(
                array(
                    'action_form'  => $this->url('/' . $this->getParameter('path') . "/{$this->application}/"),
                    'translations' => $translations,
                    'languages'    => $this->getLanguages(),
                    'application'  => $this->application,
                    'tabs'         => $this->getTabs(),
                    'query'        => $this->getRequest()->post->get('query'),
                )
            );

        $this->getHead()
            ->addCss(pongho_url('/Application/Admin/Resources/views/css/style-legacy.css?v=' . filemtime(__DIR__ . '/../../../Admin/Resources/views/css/style-legacy.css')))
            ->addJavascript(pongho_url('/Application/Core/Resources/views/js/jquery.translations.js?v=' . filemtime(__DIR__ . '/../../Resources/views/js/jquery.translations.js')));
    }

    /**
     * Restituisce la lista delle tabs disponibili
     *
     * @access protected
     */
    protected function getTabs()
    {
        $lang = $this->getLocalization();
        $path = $this->getParameter('path');

        $tabs = array();

        foreach ($this->getPartitions() as $application) {
            $action = $application === static::$index_tab ? '' : "{$application}/";
            $tabs[] = (object)array(
                'name'      => $lang->get($application),
                'link'      => $this->url("/{$path}/{$action}"),
                'is_active' => (($this->getAction() === 'index' && $application === static::$index_tab) || $this->getAction() == $application)
            );
        }

        return $tabs;
    }

    public function getSettings()
    {
        return $this->getSite();
    }
}
