<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author Daniele De Nobili
 * @package Application\Core
 */

namespace Application\Core\Controller\Admin;

use Application\Admin\Controller\CrudFormController;
use Application\Admin\Form\FormConfig;
use Application\Core\Entity\SiteInterface;
use Application\Core\Model\Language;
use Application\Core\Model\LanguageSite;
use Application\Core\Model\Manager\LanguageManager;
use Application\Core\Model\Manager\LanguageModuleManager;
use Application\Core\Model\Manager\LanguageSiteManager;
use Application\Core\Model\Manager\ModuleManager;
use Application\Core\Model\Manager\ModuleSiteManager;
use Application\Core\Model\Manager\SiteManager;
use Application\Core\Model\Module;
use Application\Core\Model\Translation;
use Pongho\Form\Form;
use Pongho\Form\Subject\ModelSubject;
use Pongho\Http\Exception\HttpNotFoundException;
use Pongho\Http\Exception\HttpUnauthorizedException;

/**
 * SitesController
 */
class SitesController extends CrudFormController
{
    /**
     * language_site_id della traduzione da copiare quando si crea un nuovo sito.
     *
     * @access private
     * @var int
     */
    private $source_language_site_id;

    /**
     * {@inheritdoc}
     */
    protected function boot()
    {
        if (!$this->getHelper()->getUser()->isFounder()) {
            throw new HttpUnauthorizedException();
        }

        parent::boot();
    }

    /**
     * {@inheritdoc}
     */
    function getNamespace()
    {
        return 'sites';
    }

    /**
     * {@inheritdoc}
     */
    public function getEventNamespace($event_name)
    {
        return 'admin.site.' . $event_name;
    }

    /**
     * {@inheritdoc}
     */
    public function getModelClass()
    {
        return 'Application\\Core\\Model\\Site';
    }

    /**
     * {@inheritdoc}
     */
    protected function getArchiveTitle()
    {
        return $this->getHelper()->getTranslator()->trans('Sites');
    }

    /**
     * {@inheritdoc}
     */
    public function hasAddButton()
    {
        return $this->getHelper()->getUser()->isFounder();
    }

    /**
     * {@inheritdoc}
     */
    protected function getArchiveAddButtonText()
    {
        return $this->getHelper()->getTranslator()->trans('Add site');
    }

    /**
     * {@inheritdoc}
     */
    protected function getArchiveAddFirstText()
    {
        return null;
    }

    /**
     * {@inheritdoc}
     */
    public function getAddEditTitle($model)
    {
        if ($this->getAction() === 'add') {
            return $this->getHelper()->getTranslator()->trans('Add site');
        }

        return $this->getHelper()->getTranslator()->trans('Edit the site “%title%”', array('%title%' => $model->name));
    }

    /**
     * {@inheritdoc}
     */
    protected function getTableColumns()
    {
        $translator = $this->getHelper()->getTranslator();

        return array(
            array(
                'name'      => 'name',
                'label'     => $translator->trans('Site'),
                'class'     => 'large',
                'orderable' => true,
                'render'    => function ($row) {
                    return '<strong>' . $row->name . '</strong><div class="actions"><span>' . $row->actions . '</span></div>';
                }
            ),
            array(
                'name'   => 'domain',
                'label'  => $translator->trans('Domain'),
                'class'  => 'main',
                'render' => function ($site) {
                    /** @var \Application\Core\Model\Site $site */
                    return $site->getPermalink();
                }
            ),
            array(
                'name'      => 'id',
                'label'     => 'ID',
                'class'     => 'id',
                'orderable' => true,
            ),
        );
    }

    /**
     * {@inheritdoc}
     */
    public function parseArchiveRow($row)
    {
        $path = $this->getParameter('path');

        $actions = array(
            array($this->getHelper()->getTranslator()->trans('Edit'), 'href' => $this->url("/{$path}/edit/{$row->id}/"), 'class' => 'edit'),
        );

        $row->actions = $this->parseActions($actions);
    }

    /**
     * {@inheritdoc}
     */
    protected function getFormsOptions()
    {
        $subject = $this->getSubject();
        $translator = $this->getHelper()->getTranslator();

        // Impostazioni principali
        $options = new FormConfig('sites', $subject, $this->getHelper()->getLocalization());

        $options->addBaseStructure('main', '');
        $options->addTab('content/main', $translator->trans('Main'));
        $options->addPanel('content/main/site');

        $options->addFields(
			'content/main/site',
			array(
                'name' => array(
                    'class'         => 'Pongho\\Form\\Field\\TextField',
                    'label'         => $translator->trans('Name'),
                    'attributes'    => array(
                        'class'         => 'input_text',
                        'required'      => 'required'
                    ),
                ),
                'domain' => array(
                    'class'         => 'Pongho\\Form\\Field\\TextField',
                    'label'         => $translator->trans('Domain'),
                    'attributes'    => array(
                        'class'         => 'input_text',
                        'required'      => 'required'
                    ),
                ),
                'content_path' => array(
                    'class'         => 'Pongho\\Form\\Field\\TextField',
                    'label'         => $translator->trans('content_path'),
                    'attributes'    => array(
                        'class'         => 'input_text',
                        'required'      => 'required'
                    ),
                ),
                'images_path' => array(
                    'class'         => 'Pongho\\Form\\Field\\TextField',
                    'label'         => $translator->trans('images_path'),
                    'attributes'    => array(
                        'class'         => 'input_text',
                        'required'      => 'required'
                    ),
                ),
                'images_url' => array(
                    'class'         => 'Pongho\\Form\\Field\\TextField',
                    'label'         => $translator->trans('images_url'),
                    'attributes'    => array(
                        'class'         => 'input_text',
                        'required'      => 'required'
                    ),
                ),
                'uploads_path' => array(
                    'class'         => 'Pongho\\Form\\Field\\TextField',
                    'label'         => $translator->trans('uploads_path'),
                    'attributes'    => array(
                        'class'         => 'input_text',
                        'required'      => 'required'
                    ),
                ),
                'uploads_url' => array(
                    'class'         => 'Pongho\\Form\\Field\\TextField',
                    'label'         => $translator->trans('uploads_url'),
                    'attributes'    => array(
                        'class'         => 'input_text',
                        'required'      => 'required'
                    ),
                ),
                'themes_path' => array(
                    'class'         => 'Pongho\\Form\\Field\\TextField',
                    'label'         => $translator->trans('themes_path'),
                    'attributes'    => array(
                        'class'         => 'input_text',
                        'required'      => 'required'
                    ),
                ),
                'themes_url' => array(
                    'class'         => 'Pongho\\Form\\Field\\TextField',
                    'label'         => $translator->trans('themes_url'),
                    'attributes'    => array(
                        'class'         => 'input_text',
                        'required'      => 'required'
                    ),
                ),
                'default_language_id'  => array(
                    'class'         => 'Pongho\\Form\\Field\\SelectField',
                    'label'         => $translator->trans('Default language'),
                    'attributes'    => array(
                        'required'      => 'required'
                    ),
                    'options' => array($this, 'getLanguageIdSelectOptions'),
                ),
            )
		);

        if ($this->getAction() === 'add') {
            $subject
                ->attachGetter('source_language_site', array($this, 'getSubjectSourceLanguageSiteCallback'))
                ->attachSetter('source_language_site', array($this, 'setSubjectSourceLanguageSiteCallback'))
            ;

            /** @var LanguageSiteManager $language_site_manager */
            $language_site_manager = $this->getContainer()->get('language_site_manager');

            $source_languages_sites = array();
            /** @var Languagesite $language_site */
            foreach ($language_site_manager->findAll() as $language_site) {
                $source_languages_sites[$language_site->getId()] = sprintf(
                    '%s - %s',
                    $language_site->getSite()->getName(),
                    $language_site->getLanguage()->getName()
                );
            }

            $options->addFields(
                'content/main/site',
                array(
                    'source_language_site'  => array(
                        'class'         => 'Pongho\\Form\\Field\\SelectField',
                        'label'         => $translator->trans('translation source'),
                        'attributes'    => array(
                            'required'      => 'required'
                        ),
                        'options' => $source_languages_sites,
                    ),
                )
            );
        }

        return $this->getHelper()->filter($this, $this->getEventNamespace('filter_form'), $options, array('subject' => $subject));
    }

    /**
     * @return array
     */
    public function getLanguageIdSelectOptions()
    {
        $ids = array();

        /** @var LanguageManager $language_manager */
        $language_manager = $this->getContainer()->get('language_manager');

        /** @var Language $language */
        foreach ($language_manager->findAll() as $language) {
            $ids[$language->getId()] = $language->getName();
        }

        return $ids;
    }

    /**
     * Callback di tipo getter per ottenere il `source_language_site_id` dalla form.
     *
     * @access public
     * @param \Pongho\Form\Subject\ModelSubject $subject
     * @return int
     */
    public function getSubjectSourceLanguageSiteCallback(ModelSubject $subject)
    {
        return $this->source_language_site_id;
    }

    /**
     * Callback di tipo setter per impostare il `source_language_site_id` nella form.
     *
     * @param \Pongho\Form\Subject\ModelSubject $subject
     * @param                                   $value
     */
    public function setSubjectSourceLanguageSiteCallback(ModelSubject $subject, $value)
    {
        $value = intval($value);

        $this->source_language_site_id = $value;
    }

    /**
     * Restituisce il subject della form.
     *
     * @access protected
     * @return \Pongho\Form\Subject\ModelSubject
     */
    protected function getSubject()
    {
        $subject = new ModelSubject($this->getModel());

        return $this->getHelper()->filter($this, $this->getEventNamespace('filter_subject'), $subject);
    }

    /**
     * {@inheritdoc}
     */
    public function deleteAction()
    {
        throw new HttpNotFoundException();
    }

    /**
     * {@inheritdoc}
     */
    protected function getMassActions()
    {
        return array();
    }

    /**
     * Risponde alla callback `after_save`.
     *
     * Imposto i parametri base dopo la creazione del sito.
     *
     * @access public
     * @param \Pongho\Form\Form $form
     */
    public function afterSaveCallback(Form $form)
    {
        if ($this->getAction() === 'add') {
            /** @var SiteManager $site_manager */
            $site_manager = $this->getContainer()->get('site_manager');

            /** @var \Application\Core\Model\Site $site */
            $site = $form->getSubject()->getModel();

            // todo: attualmente lo fa 2 volte: la prima nell'handle della Form, la seconda qui per svuotare la cache
            $site_manager->save($site);

            $this->enableCoreApplications($site);

            /** @var LanguageSiteManager $language_site_manager */
            $language_site_manager = $this->getContainer()->get('language_site_manager');
            $source_language_site = $language_site_manager->findById($this->source_language_site_id);

            /**
             * Aggiungo l’associazione tra sito e lingua di default
             */
            $language_site = $language_site_manager->create(
                $site->getDefaultLanguageId(),
                $site->getId(),
                '/',
                true
            );

            $language_site_manager->save($language_site);

            // Copio le traduzioni da un sito all’altro
            /** @var Translation $translation */
            foreach ($language_site->getTranslations() as $translation) {
                $attributes = $translation->attributes();

                $attributes['language_site_id'] = $language_site->id;

                Translation::create($attributes);
            }

            // Aggiungo la rotta degli utenti nei moduli
            $this->addModuleSite('Application\\Core\\Controller\\UserController', $language_site, $source_language_site);

            $this->getHelper()->notify(
                $this,
                'core.after_create_site',
                array(
                    'site_model'           => $site,
                    'language_site'        => $language_site,
                    'source_language_site' => $source_language_site,
                )
            );
        }
    }

    /**
     * @access public
     * @param string                               $controller_name
     * @param \Application\Core\Model\LanguageSite $language_site
     * @param \Application\Core\Model\LanguageSite $source_language_site
     * @return \Application\Core\Model\ModuleSite|null
     */
    public function addModuleSite($controller_name, LanguageSite $language_site, LanguageSite $source_language_site)
    {
        /** @var ModuleManager $module_manager */
        $module_manager = $this->getContainer()->get('module_manager');
        $modules = $module_manager->findAllByController($controller_name);

        /** @var Module $module */
        $module = reset($modules);

        /** @var LanguageModuleManager $language_module_manager */
        $language_module_manager = $this->getContainer()->get('language_module_manager');
        $user_language_module = $language_module_manager->findByLanguageSiteAndModule($source_language_site->getId(), $module->getId());

        $module_site = null;

        if ($module) {
            /** @var ModuleSiteManager $module_site_manager */
            $module_site_manager = $this->getContainer()->get('module_site_manager');
            $module_site = $module_site_manager->create(
                $user_language_module->getModule(),
                $language_site->getSite(),
                true
            );

            $module_site_manager->save($module_site);

            $user_language_module = $language_module_manager->create(
                $language_site,
                $user_language_module->getModule(),
                $user_language_module->getPath(),
                $user_language_module->getName(),
                $user_language_module->getPosition(),
                true
            );

            $language_module_manager->save($user_language_module);
        }

        return $module_site;
    }

    /**
     * @param SiteInterface $site
     */
    protected function enableCoreApplications(SiteInterface $site)
    {
        /** @var \Application\Core\Model\Manager\ApplicationManagerInterface $application_manager */
        $application_manager = $this->getContainer()->get('application_manager');

        // Abilito le applicazioni
        $application_manager->createApplicationSite('Core', $site->getId());
        $application_manager->createApplicationSite('Admin', $site->getId());
    }
}
