<?php

/**
 * Questo file è parte di Pongho 2.0-dev.
 *
 * @author    Daniele De Nobili
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Core
 */

namespace Application\Core\Controller\Admin;

use Application\Admin\Controller\AdminController;
use Application\Admin\Utilities\TablesManager;
use Application\Core\Model\Application;
use Application\Core\Application\Exception\CommandException;
use Pongho\Http\Exception\HttpNotFoundException;
use Pongho\Http\RedirectResponse;

/**
 * Class AppsController
 */
class AppsController extends AdminController
{
    /**
     * Elenco delle applicazioni
     */
    public function indexAction()
    {
        /** @var \Application\Core\Application\Facade $facade */
        $facade = $this->getContainer()->get('application_facade');

        $translator = $this->getHelper()->getTranslator();
        $site = $this->getHelper()->getSite();
        $apps = $facade->all();

        $path = $this->getParameter('path');

        foreach ($apps as $app) {
            $actions = array();

            $warnings = '';
            $enabled = false;
            $installed = false;

            if ($app->getVersion() !== '') {
                if ($facade->isInstalled($app->getName())) {
                    $installed = true;

                    if ($facade->isEnabled($app->getName(), $site)) {
                        $validator = $facade->getCouldBeDisabledValidator($app->getName());

                        if ($validator->isValid()) {
                            $actions[] = array($translator->trans('Disable'), 'href' => $this->url("/{$path}/disable/{$app->getName()}/"));
                        } else {
                            $warnings = $translator->trans('The application could not be disabled : %explanation%', array('%explanation%' => $validator->motivation()));
                        }

                        $enabled = true;
                    } else {
                        $validator = $facade->getCouldBeEnabledValidator($app->getName());

                        if ($validator->isValid()) {
                            $actions[] = array($translator->trans('Enable'), 'href' => $this->url("/{$path}/enable/{$app->getName()}/"));
                        } else {
                            $warnings = $translator->trans('The application could not be enabled : %explanation%', array('%explanation%' => $validator->motivation()));
                        }

                        $validator = $facade->getCouldBeUninstalledValidator($app->getName());

                        if ($validator->isValid()) {
                            $actions[] = array($translator->trans('Uninstall'), 'href' => $this->url("/{$path}/uninstall/{$app->getName()}/"));
                        } else {
                            $warnings = $translator->trans('The application could not be uninstalled : %explanation%', array('%explanation%' => $validator->motivation()));
                        }
                    }
                } else {
                    $validator = $facade->getCouldBeInstalledValidator($app->getName());

                    if ($validator->isValid()) {
                        $actions[] = array($translator->trans('Install'), 'href' => $this->url("/{$path}/install/{$app->getName()}/"));
                    } else {
                        $warnings = $translator->trans('The application could not be installed : %explanation%', array('%explanation%' => $validator->motivation()));
                    }
                }
            } else {
                $warnings = 'The application version is missing';
            }

            $app->actions = $this->parseActions($actions);
            $app->warnings = $warnings;
            $app->is_enabled = $enabled;
            $app->is_installed = $installed;
        }

        $table_options = array(
            'columns' => array(
                array(
                    'name'      => 'app_name',
                    'label'     => $translator->trans('Application name'),
                    'class'     => 'main',
                    'render'    => function (Application $app) {
                        return sprintf(
                            '<strong>%s</strong> (%s)<div class="actions"><span>%s</span></div>',
                            $app->getName(),
                            $app->getVersion() ?: '<span style="color: red;">N/A</span>',
                            $app->actions
                        );
                    },
                ),
                array(
                    'name'      => 'warnings',
                    'label'     => $translator->trans('Warnings'),
                    'class'     => 'main',
                    'render'    => function (Application $app) {
                            return $app->warnings;
                        }
                ),
                array(
                    'name'      => 'is_installed',
                    'label'     => $translator->trans('Is installed'),
                    'class'     => 'smallest',
                    'render'    => function (Application $app) {
                            return $app->is_installed ? '&#10004;' : '&nbsp;';
                        }
                ),
                array(
                    'name'      => 'is_enabled',
                    'label'     => $translator->trans('Is enabled'),
                    'class'     => 'smallest',
                    'render'    => function (Application $app) {
                            return $app->is_enabled ? '&#10004;' : '&nbsp;';
                        }
                ),
            ),
        );

        $table = new TablesManager($table_options, $this->getRequest(), $apps);

        $this->getHelper()->getBodyView()
            ->setTemplatePath(__DIR__ . '/../../Resources/views/applications.php')
            ->assignVar('table', $table);
    }

    /**
     * installAction
     *
     * @return \Pongho\Http\Response
     * @throws \Pongho\Http\Exception\HttpNotFoundException
     */
    public function installAction()
    {
        /** @var \Application\Core\Application\Facade $facade */
        $facade = $this->getContainer()->get('application_facade');

        $app_name = $this->getAppNameFromPathInfo();

        try {
            $facade->install($app_name, $this->getHelper()->getSite());
        } catch (CommandException $e) {
            return $this->getHelper()->displayError($e->getMessage());
        } catch (\Exception $e) {
            $text = <<<HTML
<p>Qualcosa è andato storto nella procedura di installazione dell’applicazione.</p>
<p><strong>Eccezione:</strong> {$e->getMessage()}</p>
<pre>{$e->getTraceAsString()}</pre>
HTML;

            return $this->getHelper()->displayError($text);
        }

        return new RedirectResponse($this->url('/' . $this->getParameter('path') . '/'));
    }

    /**
     * uninstallAction
     *
     * @return \Pongho\Http\Response
     * @throws \Pongho\Http\Exception\HttpNotFoundException
     */
    public function uninstallAction()
    {
        /** @var \Application\Core\Application\Facade $facade */
        $facade = $this->getContainer()->get('application_facade');

        $app_name = $this->getAppNameFromPathInfo();

        try {
            $facade->uninstall($app_name);
        } catch (CommandException $e) {
            return $this->getHelper()->displayError($e->getMessage());
        } catch (\Exception $e) {
            $text = <<<HTML
<p>Qualcosa è andato storto nella procedura di disinstallazione dell’applicazione. È necessario procedere manualmente.</p>
<p><strong>Eccezione:</strong> {$e->getMessage()}</p>
<pre>{$e->getTraceAsString()}</pre>
HTML;

            return $this->getHelper()->displayError($text);
        }

        return new RedirectResponse($this->url('/' . $this->getParameter('path') . '/'));
    }

    /**
     * @throws \Pongho\Http\Exception\HttpNotFoundException
     */
    public function enableAction()
    {
        /** @var \Application\Core\Application\Facade $facade */
        $facade = $this->getContainer()->get('application_facade');

        $app_name = $this->getAppNameFromPathInfo();

        try {
            $facade->enable($app_name, $this->getHelper()->getSite());
        } catch (CommandException $e) {
            return $this->getHelper()->displayError($e->getMessage());
        } catch (\Exception $e) {
            $text = <<<HTML
<p>Qualcosa è andato storto nella procedura di attivazione dell’applicazione. È necessario procedere manualmente.</p>
<p><strong>Eccezione:</strong> {$e->getMessage()}</p>
<pre>{$e->getTraceAsString()}</pre>
HTML;

            return $this->getHelper()->displayError($text);
        }

        return new RedirectResponse($this->url('/' . $this->getParameter('path') . '/'));
    }

    /**
     * @throws \Pongho\Http\Exception\HttpNotFoundException
     */
    public function disableAction()
    {
        /** @var \Application\Core\Application\Facade $facade */
        $facade = $this->getContainer()->get('application_facade');

        $app_name = $this->getAppNameFromPathInfo();

        try {
            $facade->disable($app_name, $this->getHelper()->getSite());
        } catch (CommandException $e) {
            return $this->getHelper()->displayError($e->getMessage());
        } catch (\Exception $e) {
            $text = <<<HTML
<p>Qualcosa è andato storto nella procedura di disattivazione dell’applicazione. È necessario procedere manualmente.</p>
<p><strong>Eccezione:</strong> {$e->getMessage()}</p>
<pre>{$e->getTraceAsString()}</pre>
HTML;

            return $this->getHelper()->displayError($text);
        }

        return new RedirectResponse($this->url('/' . $this->getParameter('path') . '/'));
    }

    /**
     * Ricavo il nome dell’applicazione dal path info
     *
     * @throws \Pongho\Http\Exception\HttpNotFoundException
     * @return string
     */
    protected function getAppNameFromPathInfo()
    {
        $path_info = $this->getRequest()->getPathInfo();
        $segments = explode('/', trim($path_info, '/'));
        $app_name = isset($segments[3]) ? $segments[3] : '';

        if (!$app_name) {
            throw new HttpNotFoundException();
        }

        return $app_name;
    }

    /**
     * @param string $app_name
     * @throws \Pongho\Http\Exception\HttpNotFoundException
     * @return \Pongho\Core\Plugin
     */
    protected function getAppKernelClass($app_name)
    {
        // Creo il nome del kernel
        $kernel_class = "Application\\{$app_name}\\Kernel";

        if (!class_exists($kernel_class)) {
            throw new HttpNotFoundException();
        }

        if (!defined("{$kernel_class}::VERSION")) {
            throw new HttpNotFoundException();
        }

        /**
         * @var \Pongho\Core\Plugin $kernel
         */
        return new $kernel_class($this->getContainer());
    }

    /**
     * @see \Application\Admin\Controller\Crud2Controller::parseActions()
     *
     * @param array $actions
     * @return string
     */
    protected function parseActions(array $actions)
    {
        $html = $this->getContainer()->get('template_html');

        $code = array();
        foreach ( $actions as $action )
        {
            $label = array_shift($action);
            $attributes = $action;

            $code[] = $html->generateTagBlock('a', $label, $attributes);
        }

        return implode(' | ', $code);
    }
}
