<?php

namespace Application\Calendar\Model;

use ActiveRecord\Base;
use Application\Core\Model\Account;

/**
 * @property int                                          $id
 * @property int                                          $calendar_id
 * @property \Application\Calendar\Model\Calendar         $calendar
 * @property int                                          $category_id
 * @property \Application\Calendar\Model\CalendarCategory $category
 * @property int                                          $author_id
 * @property bool                                         $is_all_day
 * @property \Pongho\Utilities\DateTime                   $start_date_at
 * @property \Pongho\Utilities\DateTime                   $end_date_at
 * @property string                                       $title
 * @property string                                       $content
 * @property string                                       $place
 * @property string                                       $status
 * @property \Pongho\Utilities\DateTime                   $created_at
 * @property \Pongho\Utilities\DateTime                   $updated_at
 */
class CalendarEvent extends Base
{
    public static $table_name = 'calendars_events';

    public static $belongs_to = [
        ['calendar', 'model' => \Application\Calendar\Model\Calendar::class],
        ['category', 'model' => \Application\Calendar\Model\CalendarCategory::class],
    ];

    protected $users;

    public function delete($in_recursion = false)
    {
        return $this->deleteCalendarEventUsers() && parent::delete();
    }

    public function deleteCalendarEventUsers()
    {
        $options = [
            'conditions' => ['calendar_event_id = :event', 'event' => $this->id],
        ];

        foreach (CalendarEventUser::all($options) as $row) {
            if (!$row->delete()) {
                return false;
            }
        }

        return true;
    }

    public function getParticipants()
    {
        if ($this->users === null) {
            $options = [
                'conditions' => ['calendar_event_id = :event', 'event' => $this->id],
            ];

            $this->users = CalendarEventUser::all($options);
        }

        return $this->users;
    }

    public function addParticipant($user_id, $status = 'active')
    {
        if (Account::find($user_id) === null) {
            return false;
        }

        foreach ($this->getParticipants() as $participant) {
            if ($participant->user_id === intval($user_id)) {
                return false;
            }
        }

        $attributes = [
            'calendar_event_id' => $this->id,
            'user_id'           => $user_id,
            'status'            => $status,
        ];

        $event_user = CalendarEventUser::create($attributes);

        return $event_user !== null;
    }

    public function removeParticipant($user_id)
    {
        if (Account::find($user_id) === null) {
            return false;
        }

        $options = [
            'conditions' => [
                'calendar_event_id = :event AND user_id = :user',
                'event' => $this->id,
                'user'  => $user_id,
            ],
        ];

        if ($participant = CalendarEventUser::first($options)) {
            return $participant->delete();
        }

        return false;
    }

    public function hasParticipants()
    {
        $participants = $this->getParticipants();

        return !empty($participants);
    }

    public function inParticipants($user_id)
    {
        foreach ($this->getParticipants() as $participant) {
            if ($participant->user_id === intval($user_id)) {
                return true;
            }
        }

        return false;
    }

    /**
     * @param int $author_id
     * @return $this
     */
    public function setAuthorId($author_id)
    {
        if ($author_id) {
            $this->author_id = $author_id;
        }

        return $this;
    }
}
