<?php

namespace Application\Calendar\Model;

use ActiveRecord\Base;
use Application\Core\Model\LanguageModule;
use Application\Core\Model\Module;
use Pongho\Core\Kernel;
use Pongho\Utilities\Inflector;

/**
 * @property int                             $id
 * @property int                             $module_site_id
 * @property int                             $owner_id
 * @property \Application\Core\Model\Account $owner
 * @property string                          $title
 * @property string                          $color
 * @property string                          $status
 * @property int                             $position
 * @property string                          $slug
 * @property bool $can_add
 * @property bool $can_edit_self
 * @property bool $can_edit_others
 * @property bool $can_delete_self
 * @property bool $can_delete_others
 * @property bool $can_view_others
 */
class Calendar extends Base
{
    protected $categories;

    protected $users;

    public static $table_name = 'calendars';

    protected $events;

    /**
     * Relazioni
     */
    public static $belongs_to = [
        ['owner', 'model' => \Application\Core\Model\Account::class],
    ];

    /**
     * Callback 'after_create'.
     *
     * @var array
     */
    public static $after_save = ['clearCache'];

    /**
     * @param array $options
     * @return CalendarEvent[]
     */
    public function getEvents(array $options)
    {
        if ($this->events === null) {
            if (!isset($options['conditions'])) {
                $options['conditions'] = ['calendar_id = :id', 'id' => $this->id];
            }

            $this->events = CalendarEvent::all($options);
        }

        return $this->events;
    }

    /**
     * {@inheritdoc}
     */
    public function delete($in_recursion = false)
    {
        return $this->deleteCalendarEvents() && $this->deleteCalendarCategories() && $this->deleteCalendarUsers() && parent::delete();
    }

    /**
     * @return bool
     */
    public function deleteCalendarEvents()
    {
        $options = [
            'conditions' => ['calendar_id = :calendar', 'calendar' => $this->id],
        ];

        foreach (CalendarEvent::all($options) as $row) {
            if (!$row->delete()) {
                return false;
            }
        }

        return true;
    }

    /**
     * @return bool
     */
    public function deleteCalendarCategories()
    {
        $options = [
            'conditions' => ['calendar_id = :calendar', 'calendar' => $this->id],
        ];

        foreach (CalendarCategory::all($options) as $row) {
            if (!$row->delete()) {
                return false;
            }
        }

        $this->categories = null;

        return true;
    }

    /**
     * @return bool
     */
    public function deleteCalendarUsers()
    {
        $options = [
            'conditions' => ['calendar_id = :calendar', 'calendar' => $this->id],
        ];

        foreach (CalendarUser::all($options) as $row) {
            if (!$row->delete()) {
                return false;
            }
        }

        $this->users = null;

        return true;
    }

    /**
     * @return \Application\Calendar\Model\Calendarcategory[]
     */
    public function getCategories()
    {
        if ($this->categories === null) {
            $options = [
                'conditions' => ['calendar_id = :calendar', 'calendar' => $this->id],
            ];

            $this->categories = [];

            /** @var \Application\Calendar\Model\CalendarCategory $category */
            foreach (CalendarCategory::all($options) as $category) {
                $this->categories[$category->id] = $category;
            }
        }

        return $this->categories;
    }

    /**
     * @return array
     */
    public function getUsers()
    {
        if ($this->users === null) {
            $options = [
                'conditions' => ['calendar_id = :calendar', 'calendar' => $this->id],
            ];

            $this->users = [];

            /** @var \Application\Calendar\Model\CalendarUser $user */
            foreach (CalendarUser::all($options) as $user) {
                $this->users[$user->user_id] = $user;
            }
        }

        return $this->users;
    }

    /**
     * @param $calendar_id
     * @return array
     */
    public static function getCategoriesSelectOptions($calendar_id)
    {
        $options = [
            'conditions' => ['calendar_id = :calendar', 'calendar' => $calendar_id],
        ];

        $categories = [];
        /** @var \Application\Calendar\Model\CalendarCategory $category */
        foreach (CalendarCategory::all($options) as $category) {
            $categories[$category->id] = $category->title;
        }

        return $categories;
    }

    protected $js_events = [];

    /**
     * @param array $events
     */
    public function setJsEvents(array $events)
    {
        $this->js_events = $events;
    }

    /**
     * @return mixed
     */
    public function getJsEvents()
    {
        return $this->js_events;
    }

    /**
     * Aggiorna il permalink dei calendari, ad esempio se è stato cambiato il percorso del calendario.
     */
    public function updatePermalink()
    {
        $this->createPermalink();
    }

    /**
     * Crea il permalink per il calendario.
     */
    public function createPermalink()
    {
        if (!empty($this->title)) {
            if (empty($this->slug)) {
                $this->slug = $this->title;
            }

            $this->slug = Inflector::urlize($this->slug);

            $this->checkPermalinkUniqueness();
        }
    }

    /**
     * Verifica l’unicità del permalink per il calendario.
     */
    protected function checkPermalinkUniqueness()
    {
        $options = [
            'conditions' => [
                'slug = :slug AND module_site_id = :module_site',
                'slug'        => $this->slug,
                'module_site' => $this->module_site_id,
            ],
        ];

        if ($this->id) {
            $options = Calendar::addCondition($options, ['id <> :calendar', 'calendar' => $this->id]);
        }

        if (static::count($options)) {
            $suffix = 1;

            do {
                $suffix++;
                $options['conditions']['slug'] = $this->slug . '-' . $suffix;
            } while (static::count($options));

            $this->slug .= '-' . $suffix;
        }
    }

    public static function findByPermalink($permalink)
    {
        $site_id = Kernel::instance()->getContainer()->get('site')->id;

        $options = [
            'select'     => '`from`.*',
            'joins'      => 'INNER JOIN ' . ModuleSite::tableName() . ' AS ms ON ms.id = `from`.module_site_id'
                         . ' INNER JOIN ' . Module::tableName() . ' AS m ON m.id = ms.module_id'
                         . ' INNER JOIN ' . LanguageModule::tableName() . ' AS lm ON lm.module_id = ms.module_id',

            'conditions' => [
                'CONCAT(lm.path, `from`.slug) = :permalink AND lm.is_enabled = :enabled AND m.controller = :controller AND ms.site_id = :site',
                'permalink'  => rtrim($permalink, '/'),
                'enabled'    => true,
                'site'       => $site_id,
                'controller' => \Application\Calendar\Controller\CalendarController::class,
            ],
        ];

        return Calendar::first($options);
    }

    /**
     * Svuota la cache dei modelli collegati
     */
    public function clearCache()
    {
        $this->users = null;
        $this->categories = null;
        $this->events = null;
    }
}
