<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author Daniele Termini
 * @package \Application\Calendar
 */

namespace Application\Calendar\Installer;

use Application\Calendar\Kernel;
use Application\Core\Model\Permit;

/**
 * Class Installer
 */
class Installer extends \Application\Core\Installer\Installer
{
    /**
     * Crea la struttura in database e prepara i moduli
     */
    public function install()
    {
        /** @var \Monolog\Logger $logger */
        $logger = $this->container->get('logger');

        $logger->info('[Calendar][Installer] Installazione calendario');

        $schema = $this->getSchema();

        // Calendari
        $calendars_table = $schema->createTable('pongho_calendars');
        $calendars_table->addColumn('id', 'integer', ['unsigned' => true, 'autoincrement' => true]);
        $calendars_table->addColumn('module_site_id', 'integer', ['unsigned' => true]);
        $calendars_table->addColumn('owner_id', 'integer', ['unsigned' => true]);
        $calendars_table->addColumn('title', 'string', ['length' => 255, 'default' => '']);
        $calendars_table->addColumn('color', 'string', ['length' => 50, 'default' => '']);
        $calendars_table->addColumn('status', 'string', ['length' => 20, 'default' => '']);
        $calendars_table->addColumn('position', 'integer', ['unsigned' => true, 'default' => 0]);
        $calendars_table->addColumn('slug', 'string', ['length' => 100, 'default' => '']);
        $calendars_table->addColumn('can_add', 'boolean', ['default' => true]);
        $calendars_table->addColumn('can_edit_self', 'boolean', ['default' => true]);
        $calendars_table->addColumn('can_edit_others', 'boolean', ['default' => false]);
        $calendars_table->addColumn('can_delete_self', 'boolean', ['default' => true]);
        $calendars_table->addColumn('can_delete_others', 'boolean', ['default' => false]);
        $calendars_table->addColumn('can_view_others', 'boolean', ['default' => true]);

        $calendars_table->setPrimaryKey(['id']);
        $calendars_table->addIndex(['position']);
        $calendars_table->addIndex(['slug']);
        $calendars_table->addForeignKeyConstraint($schema->getTable('pongho_modules_sites'), ['module_site_id'], ['id']);
        $calendars_table->addForeignKeyConstraint($schema->getTable('pongho_users'), ['owner_id'], ['id']);


        // Condivisioni dei calendari
        $calendars_users_table = $schema->createTable('pongho_calendars_users');
        $calendars_users_table->addColumn('calendar_id', 'integer', ['unsigned' => true]);
        $calendars_users_table->addColumn('user_id', 'integer', ['unsigned' => true]);
        $calendars_users_table->addColumn('can_add', 'boolean', ['default' => true]);
        $calendars_users_table->addColumn('can_edit_self', 'boolean', ['default' => true]);
        $calendars_users_table->addColumn('can_edit_others', 'boolean', ['default' => false]);
        $calendars_users_table->addColumn('can_delete_self', 'boolean', ['default' => true]);
        $calendars_users_table->addColumn('can_delete_others', 'boolean', ['default' => false]);
        $calendars_users_table->addColumn('can_view_others', 'boolean', ['default' => true]);

        $calendars_users_table->setPrimaryKey(['calendar_id', 'user_id']);
        $calendars_users_table->addForeignKeyConstraint($calendars_table, ['calendar_id'], ['id']);
        $calendars_users_table->addForeignKeyConstraint($schema->getTable('pongho_users'), ['user_id'], ['id']);


        // Categorie
        $categories_table = $schema->createTable('pongho_calendars_categories');
        $categories_table->addColumn('id', 'integer', ['unsigned' => true, 'autoincrement' => true]);
        $categories_table->addColumn('calendar_id', 'integer', ['unsigned' => true]);
        $categories_table->addColumn('title', 'string', ['length' => 255, 'default' => '']);

        $categories_table->setPrimaryKey(['id']);
        $categories_table->addForeignKeyConstraint($calendars_table, ['calendar_id'], ['id']);


        // Eventi
        $events_table = $schema->createTable('pongho_calendars_events');
        $events_table->addColumn('id', 'integer', ['unsigned' => true, 'autoincrement' => true]);
        $events_table->addColumn('calendar_id', 'integer', ['unsigned' => true]);
        $events_table->addColumn('category_id', 'integer', ['unsigned' => true, 'notnull' => false]);
        $events_table->addColumn('author_id', 'integer', ['unsigned' => true]);
        $events_table->addColumn('is_all_day', 'boolean', ['default' => false]);
        $events_table->addColumn('start_date_at', 'datetime');
        $events_table->addColumn('end_date_at', 'datetime');
        $events_table->addColumn('title', 'string', ['length' => 255, 'default' => '']);
        $events_table->addColumn('content', 'text', ['length' => 65535, 'default' => '']);
        $events_table->addColumn('place', 'string', ['length' => 255, 'default' => '']);
        $events_table->addColumn('status', 'string', ['length' => 20, 'default' => '']);
        $events_table->addColumn('created_at', 'datetime');
        $events_table->addColumn('updated_at', 'datetime');

        $events_table->setPrimaryKey(['id']);
        $events_table->addForeignKeyConstraint($calendars_table, ['calendar_id'], ['id']);
        $events_table->addForeignKeyConstraint($categories_table, ['category_id'], ['id']);
        $events_table->addForeignKeyConstraint($schema->getTable('pongho_users'), ['author_id'], ['id']);


        // Utenti invitati agli eventi
        $events_users_table = $schema->createTable('pongho_calendars_events_users');
        $events_users_table->addColumn('calendar_event_id', 'integer', ['unsigned' => true]);
        $events_users_table->addColumn('user_id', 'integer', ['unsigned' => true]);
        $events_users_table->addColumn('status', 'string', ['length' => 20, 'default' => '']);

        $events_users_table->setPrimaryKey(['calendar_event_id', 'user_id']);
        $events_users_table->addForeignKeyConstraint($events_table, ['calendar_event_id'], ['id']);
        $events_users_table->addForeignKeyConstraint($schema->getTable('pongho_users'), ['user_id'], ['id']);


        $this->updateSchema();

        $this->fixRegisteredUpdates();


        // Creo i permessi per i reminder
        $permits = [
            'calendar.admin',
        ];

        foreach ($permits as $key) {
            $this->getOrCreatePermit($key);
        }

        // Creo module e module_site
        $module = $this->getOrCreateModule(Kernel::FRONT_CONTROLLER);
        $this->getOrCreateModuleSite($module->id, false, \Application\Calendar\Model\ModuleSite::class);
        $this->getOrCreateLanguageModule($module->id, '/calendar/', 'Calendar');
    }

    /**
     *
     */
    public function uninstall()
    {
        /** @var \Monolog\Logger $logger */
        $logger = $this->container->get('logger');

        $logger->info('[Calendar][Installer] Disinstallazione calendario');

        $schema = $this->getSchema();

        if ($schema->hasTable('pongho_calendars_events_users')) {
            $schema->dropTable('pongho_calendars_events_users');
        }

        if ($schema->hasTable('pongho_calendars_events')) {
            $schema->dropTable('pongho_calendars_events');
        }

        if ($schema->hasTable('pongho_calendars_categories')) {
            $schema->dropTable('pongho_calendars_categories');
        }

        if ($schema->hasTable('pongho_calendars_users')) {
            $schema->dropTable('pongho_calendars_users');
        }

        if ($schema->hasTable('pongho_calendars')) {
            $schema->dropTable('pongho_calendars');
        }

        $this->updateSchema();

        // Elimino i permessi
        $options = [
            'conditions' => [
                'pongho_like(`key`, :key)',
                'key' => 'calendar.%',
            ],
        ];

        foreach (Permit::all($options) as $permit) {
            $permit->delete();
        }

        $this->deleteModule(Kernel::FRONT_CONTROLLER);
    }
}
