<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author Daniele Termini
 * @package Application\Calendar
 */

namespace Application\Calendar\Handler\View;

use Application\Calendar\Handler\DataInterface;
use Application\Calendar\Handler\PopupInterface;
use Application\Calendar\Handler\ViewInterface;
use Application\Calendar\Model\CalendarEvent;
use Pongho\Http\Request;
use Pongho\Template\HeadHelper;

class BaseView implements ViewInterface
{
    /**
     * @var DataInterface
     */
    protected $data;

    /**
     * @var PopupInterface
     */
    protected $popup;

    /**
     * @param DataInterface  $data
     * @param PopupInterface $popup
     */
    public function __construct(DataInterface $data, PopupInterface $popup)
    {
        $this->data = $data;
        $this->popup = $popup;
    }

    /**
     * {@inheritdoc}
     */
    public function getName()
    {
        return 'base';
    }

    /**
     * {@inheritdoc}
     */
    public function getPopupViewTemplatePath()
    {
        return PONGHO_PATH . '/Application/Calendar/Resources/public/event_view.php';
    }

    /**
     * {@inheritdoc}
     */
    public function getPopupEditTemplatePath()
    {
        return PONGHO_PATH . '/Application/Calendar/Resources/public/calendar_popup.php';
    }

    /**
     * {@inheritdoc}
     */
    public function getCalendars()
    {
        return $this->data->getCalendars();
    }

    /**
     * {@inheritdoc}
     */
    public function getEvents()
    {
        return $this->data->getEvents();
    }

    /**
     * {@inheritdoc}
     */
    public function populateHead(HeadHelper $head)
    {
        $this->popup->populateHead($head);

        $code = <<<JS
$(document).ready(function() {
    $.timepicker.setDefaults($.timepicker.regional['it']);

    var o = {$this->getJsonOptions()};

    o.popup = $.extend({}, Pongho.Calendar.Popup);

    {$this->includeJsInline()}

    $('#calendar').ponghoFullCalendar(o);
});
JS;

        $head
            ->addJavaScriptInline($code);
    }

    /**
     * Consente di inserire del codice javascript prima dell'esecuzione del calendario, ad esempio per sovrascrivere
     * le callback
     *
     * @return string
     */
    protected function includeJsInline()
    {
        return '';
    }

    /**
     * Restituisce la configurazione per gestire gli eventi
     *
     * @return array
     */
    protected function getJsEventsConfig()
    {
        return [
            'url'  => url('/api/calendar/events/'),
            'data' => [
                'calendars' => $this->getJsCalendars(),
                'handler'   => $this->getName(),
            ],
        ];
    }

    /**
     * Restituisce le opzioni in formato JSON
     *
     * @return string
     */
    protected function getJsonOptions()
    {
        return json_encode(
            $this->getJsCalendarOptions([
                'handler' => $this->getName(),
                'events'  => $this->getJsEventsConfig(),
            ]),
        );
    }

    /**
     * Restituisce l'elenco degli ID dei calendari visibili per questa vista da passare al JS.
     *
     * @return array
     */
    protected function getJsCalendars()
    {
        $calendars = [];
        foreach ($this->data->getCalendars() as $calendar) {
            $calendars[] = $calendar->id;
        }

        return $calendars;
    }

    /**
     * @param array $options
     * @return array
     */
    public function getJsCalendarOptions(array $options = [])
    {
        return array_merge($this->getDefaultJsOptions(), $options);
    }

    /**
     * @return array
     */
    protected function getDefaultJsOptions()
    {
        return [
            'aspect_ratio_handling' => 'fixed',
            'axisFormat'            => 'H:mm',
            'contentHeight'         => 800,
            'selectable'            => true,
            'selectHelper'          => true,
            'event_actions'         => [
                'add_url'    => url('/api/calendar/add_event/'),
                'edit_url'   => url('/api/calendar/edit_event/[id]/'),
                'view_url'   => url('/api/calendar/event/[id]/'),
                'delete_url' => url('/api/calendar/delete_event/[id]/'),
                'drop_url'   => url('/api/calendar/drop_event/[id]/'),
                'resize_url' => url('/api/calendar/resize_event/[id]/'),
            ],
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function getPopupForm(CalendarEvent $event, Request $request)
    {
        return $this->popup->getForm($event, $this->data, $request);
    }
}
