<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author Daniele Termini
 * @package Application\Calendar
 */

namespace Application\Calendar\Handler\Popup;

use Application\Admin\Form\Generator\AdminGenerator;
use Application\Calendar\Handler\DataInterface;
use Application\Calendar\Handler\PopupInterface;
use Application\Calendar\Model\CalendarEvent;
use Application\Core\Localization;
use Pongho\Form\Subject\ModelRowSubject;
use Pongho\Form\Subject\ModelSubject;
use Pongho\Form\Utilities\FormBuilder;
use Pongho\Form\Utilities\FormConfig;
use Pongho\Http\Request;
use Pongho\Template\HeadHelper;

class BasePopup implements PopupInterface
{
    /**
     * @var \Application\Core\Localization
     */
    protected $lang;

    /**
     * @var array
     */
    protected $calendar_event_states;

    /**
     * @var array
     */
    protected $calendar_colors;

    public function __construct(Localization $lang, array $calendar_event_states, array $calendar_colors)
    {
        $this->lang = $lang;
        $this->calendar_event_states = $calendar_event_states;
        $this->calendar_colors = $calendar_colors;
    }

    /**
     * {@inheritdoc}
     */
    public function populateHead(HeadHelper $head)
    {
        $head
            ->addJavaScript(pongho_url('/Application/Calendar/Resources/public/js/calendar-popup.js'));
    }

    /**
     * {@inheritdoc}
     */
    public function getForm(CalendarEvent $event, DataInterface $data, Request $request)
    {
        $subject = new ModelSubject($event);
        $subject->attachGetter('users', [$this, 'getEventParticipants']);

        $config = $this->getFormConfig($subject, $data, $request);

        /** @var \Pongho\Form\Form $form */
        $form = FormBuilder::buildConfig($config);
        $form
            ->setGenerator(new AdminGenerator($this->lang))
            ->setAction(url($request->getPathInfo()));

        return $form;
    }

    /**
     * Restituisce i partecipanti all'evento
     *
     * @param ModelSubject $subject
     * @return array
     */
    public function getEventParticipants(ModelSubject $subject)
    {
        $participants = [];
        if (!$subject->isNewRecord()) {
            /** @var \Application\Calendar\Model\CalendarEvent $model */
            $model = $subject->getModel();

            foreach ($model->getParticipants() as $participant) {
                $participants[$participant->id] = new ModelRowSubject($participant);
            }
        }

        return $participants;
    }

    /**
     * @param ModelSubject  $subject
     * @param DataInterface $data
     * @param Request       $request
     * @return FormConfig
     */
    protected function getFormConfig(ModelSubject $subject, DataInterface $data, Request $request)
    {
        $config = new FormConfig('event', $subject);

        $config->addFields(
            '',
            [
                'event'        => [
                    'class' => \Pongho\Form\Fieldset::class,
                    'label' => $this->lang->get('calendar_event'),
                ],
                'dates'        => [
                    'class' => \Pongho\Form\Fieldset::class,
                    'label' => $this->lang->get('calendar_event_dates'),
                ],
                'details'      => [
                    'class' => \Pongho\Form\Fieldset::class,
                    'label' => $this->lang->get('calendar_event_details'),
                ],
                'participants' => [
                    'class' => \Pongho\Form\Fieldset::class,
                    'label' => $this->lang->get('calendar_event_participants'),
                ],
                'actions'      => [
                    'class'    => \Pongho\Form\Fieldset::class,
                    'settings' => [
                        'hide_label' => true,
                    ],
                ],
            ]
        );

        $config->addFields(
            'actions',
            [
                'save'   => [
                    'class'    => \Pongho\Form\Field\ButtonField::class,
                    'label'    => $this->lang->get('save'),
                    'settings' => [
                        'type'  => 'submit',
                        'value' => 'save',
                    ],
                ],
                'cancel' => [
                    'class'    => \Pongho\Form\Field\ButtonField::class,
                    'label'    => $this->lang->get('cancel'),
                    'settings' => [
                        'value' => 'cancel',
                    ],
                ],
                'delete' => [
                    'class'    => \Pongho\Form\Field\ButtonField::class,
                    'label'    => $this->lang->get('delete'),
                    'settings' => [
                        'value' => 'delete',
                    ],
                ],
            ]
        );

        $config->addFields(
            'event',
            [
                'calendar_id' => [
                    'class'      => \Pongho\Form\Field\SelectField::class,
                    'label'      => $this->lang->get('calendar'),
                    'attributes' => ['required' => 'required'],
                    'options'    => $data->getCalendarSelectOptions(),
                    'settings'   => [
                        'placeholder' => $this->lang->get('select_calendar'),
                    ],
                ],
                'title'       => [
                    'class'      => \Pongho\Form\Field\TextField::class,
                    'label'      => $this->lang->get('title'),
                    'attributes' => ['required' => 'required'],
                ],
                'content'     => [
                    'class' => \Pongho\Form\Field\TextareaField::class,
                    'label' => $this->lang->get('description'),
                ],
                'is_all_day'  => [
                    'class' => \Pongho\Form\Field\CheckboxField::class,
                    'label' => $this->lang->get('calendar_event_full_day'),
                ],
            ]
        );

        $config->addFields(
            'dates',
            [
                'start_date_at' => [
                    'class'      => \Pongho\Form\Field\DateTimeField::class,
                    'label'      => $this->lang->get('calendar_event_start'),
                    'attributes' => [
                        'class' => 'datetimepicker',
                    ],
                ],
                'end_date_at'   => [
                    'class'      => \Pongho\Form\Field\DateTimeField::class,
                    'label'      => $this->lang->get('calendar_event_end'),
                    'attributes' => [
                        'class' => 'datetimepicker',
                    ],
                ],
            ]
        );

        // Prendo il calendar_id sempre da POST perché posso averlo modificato nell'evento, altrimenti caricherei
        // le categorie del precedente calendario
        $calendar_id = $request->post->get('event[calendar_id]', null, true);

        $config->addFields(
            'details',
            [
                'category_id' => [
                    'class'      => \Pongho\Form\Field\SelectField::class,
                    'label'      => $this->lang->get('category'),
                    'attributes' => [
                        'data-url' => url('/api/calendar/categories/[id]/'),
                    ],
                    'options'    => $data->getCategoriesSelectOptions($calendar_id),
                    'settings'   => [
                        'placeholder' => $this->lang->get('calendar_select_category'),
                    ],
                ],
                'status'      => [
                    'class'    => \Pongho\Form\Field\SelectField::class,
                    'label'    => $this->lang->get('status'),
                    'options'  => $this->calendar_event_states,
                    'settings' => [
                        'placeholder' => $this->lang->get('calendar_select_status'),
                    ],
                ],
            ]
        );

        $config->addField(
            'participants/users',
            [
                'class'      => \Pongho\Form\Repeater\ModelRepeater::class,
                'attributes' => ['class' => 'panel'],
                'settings'   => [
                    'filter_model' => \Application\Calendar\Model\CalendarEventUser::class,
                    'row_settings' => [
                        'show-delete'  => true,
                        'delete-label' => $this->lang->get('remove'),
                    ],
                ],
            ]
        );

        $config->addFields(
            'participants/users',
            [
                'main'    => [
                    'class' => \Application\Admin\Form\Repeater\RowMain::class,
                ],
                'content' => [
                    'class' => \Application\Admin\Form\Repeater\RowContent::class,
                ],
            ]
        );

        return $config;
    }

    /**
     * {@inheritdoc}
     */
    public function getView()
    {
    }
}
