<?php

namespace Application\Calendar\Handler\Data;

use Application\Calendar\Acl;
use Application\Calendar\Handler\DataInterface;
use Application\Calendar\Model\Calendar;
use Application\Calendar\Model\CalendarEvent;
use Application\Core\Controller;
use Application\Core\Model\Account;
use Application\Core\Model\Role;
use Pongho\Http\Request;
use Pongho\Utilities\DateTime;

class BaseData implements DataInterface
{
    /**
     * @var \Pongho\Http\Request
     */
    protected $request;

    /**
     * @var \Application\Core\Model\Account
     */
    protected $account;

    /**
     * @var \Application\Calendar\Model\Calendar[]
     */
    protected $calendars;

    /**
     * @var \Application\Calendar\Acl
     */
    protected $acl;

    /**
     * @var array
     */
    protected $colors;

    public function __construct(Request $request, Account $account, Acl $acl, array $colors)
    {
        $this->request = $request;
        $this->account = $account;
        $this->colors = $colors;
        $this->acl = $acl;
    }

    /**
     * {@inheritdoc}
     */
    public static function getClassPath()
    {
        return \Application\Calendar\Handler\Data\BaseData::class;
    }

    /**
     * @return array
     */
    protected function getCalendarsQueryOptions()
    {
        return [];
    }

    /**
     * @param \Application\Calendar\Model\Calendar $calendar
     * @return bool
     */
    protected function isEditable(Calendar $calendar)
    {
        // Se sono founder posso modificarlo sempre?
        //        if ($this->account->is_founder) {
        //            return true;
        //        }

        // Posso modificare un intero calendario solo se lo stato è attivo
        if ($calendar->status === 'disabled') {
            return false;
        }

        return (($calendar->owner_id === $this->account->id) || $this->account->role_id === Role::ADMIN);
    }

    /**
     * {@inheritdoc}
     */
    public function getCalendars()
    {
        if ($this->calendars === null) {
            $this->calendars = [];

            /** @var \Application\Calendar\Model\Calendar $calendar */
            foreach (Calendar::all($this->getCalendarsQueryOptions()) as $calendar) {
                $id = 'calendar-' . $calendar->id;

                $calendar->is_editable = $this->isEditable($calendar);
                //                $calendar->setJsEvents($this->getEvents(
                //                    array(
                //                        'query_options' => array(
                //                            'conditions' => array(
                //                                'calendar_id = :calendar',
                //                                'calendar' => $calendar->id
                //                            )
                //                        )
                //                    )
                //                ));

                $this->calendars[$id] = $calendar;
            }
        }

        return $this->calendars;
    }

    /**
     * {@inheritdoc}
     */
    public function getCalendarSelectOptions()
    {
        $calendar_select = [];
        $options = [];
        foreach (Calendar::all($options) as $calendar) {
            $calendar_select[$calendar->id] = $calendar->title;
        }

        return $calendar_select;
    }

    /**
     * @param int $calendar_id
     * @return array
     */
    public function getCategoriesSelectOptions($calendar_id)
    {
        return Calendar::getCategoriesSelectOptions($calendar_id);
    }

    /**
     * {@inheritdoc}
     * @throws \InvalidArgumentException
     */
    public function getEvents()
    {
        $str_start_date = $this->request->query->get('start');
        $str_end_date = $this->request->query->get('end');

        // Le date mi arrivano in Unix Timestamp Format
        $start_date = date('Y-m-d H:i:s', $str_start_date);
        $end_date = date('Y-m-d H:i:s', $str_end_date);

        $calendars = $this->request->query->get('calendars', []);

        if (empty($calendars)) {
            return [];
        }

        // L'evento potrebbe iniziare prima e finire dopo il giorno corrente
        // prima: 'start_date_at >= :start AND end_date_at < :end'
        $query_options = [
            'conditions' => [
                'calendar_id IN (' . implode(', ', array_map('intval', $calendars)) . ') AND start_date_at < :end AND end_date_at > :start',
                'start' => $start_date,
                'end'   => $end_date,
            ],
        ];

        $events = [];
        /** @var \Application\Calendar\Model\CalendarEvent $event */
        foreach (CalendarEvent::all($query_options) as $event) {
            $events[] = $this->makeEvent($event);
        }

        return $events;
    }

    protected function makeEvent(CalendarEvent $event, $is_editable = true)
    {
        $participants = [];
        foreach ($event->getParticipants() as $participant) {
            $p = [];
            foreach ($participant->attributes as $field => $value) {
                $p[$field] = ($value instanceof DateTime ? $value->format('Y-m-d') : $value);
            }

            // Campi supplementari
            $p['label'] = $participant->user->name(true);

            $participants[] = $p;
        }

        return [
            'id'           => $event->id,
            'author'       => $event->author_id,
            'editable'     => $is_editable,
            'calendar'     => $event->calendar_id,
            'category'     => $event->category_id,
            'title'        => $event->title,
            'start'        => $event->start_date_at->format('Y-m-d H:i'),
            'end'          => $event->end_date_at->format('Y-m-d H:i'),
            'place'        => $event->place,
            'description'  => $event->content,
            'allDay'       => $event->is_all_day,
            'status'       => $event->status,
            'color'        => $this->colors[$event->calendar->color]['bg'],
            'textColor'    => $this->colors[$event->calendar->color]['text'],
            'participants' => $participants,
            'resourceId'   => $event->calendar_id,
        ];
    }
}
