<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Calendar
 */

namespace Application\Calendar\DependencyInjection;

use Application\Calendar\Acl;
use Application\Calendar\Handler\Data\BaseData;
use Application\Calendar\Handler\Popup\AdminPopup;
use Application\Calendar\Handler\Popup\AdminSinglePopup;
use Application\Calendar\Handler\Popup\BasePopup;
use Application\Calendar\Handler\Popup\SinglePopup;
use Application\Calendar\Handler\View\AdminSingleView;
use Application\Calendar\Handler\View\AdminView;
use Application\Calendar\Handler\View\BaseView;
use Application\Calendar\Handler\View\ResourcesView;
use Application\Calendar\Handler\View\SingleView;
use Application\Calendar\Kernel;
use Application\Calendar\Model\ModuleSite;
use Application\Core\Model\Module;
use Pongho\DependencyInjection\Container;
use Pongho\DependencyInjection\ServiceProviderInterface;
use Pongho\EventDispatcher\Event;

class CalendarServicesProvider implements ServiceProviderInterface
{
    protected ModuleSite $module_site;

    public function register(Container $container): void
    {
        $container->share('calendar_views', function (Container $container): array {
            /** @var \Application\Core\I18n\Translation\Translator $lang */
            $translator = $container->get('translator');

            $options = [
                'agendaDay'   => $translator->trans('Agenda day'),
                'agendaWeek'  => $translator->trans('Agenda week'),
                'month'       => $translator->trans('Month'),
                'resourceDay' => $translator->trans('Resource day'),
            ];

            return $this->filter($container, 'calendar.filter_views', $options);
        });

        $container->share('calendar_colors', function (Container $container): array {
            return $this->filter(
                $container,
                'calendar.filter_colors',
                [
                    'lightsteelblue' => ['bg' => '#B0C4DE', 'text' => '#666'],
                    'white'          => ['bg' => '#f5f5f5', 'text' => '#666'],
                    'silver'         => ['bg' => '#a5a5a5', 'text' => '#fff'],
                    'gray'           => ['bg' => '#888888', 'text' => '#fff'],
                    'black'          => ['bg' => '#333333', 'text' => '#fff'],
                    'red'            => ['bg' => '#C10031', 'text' => '#fff'],
                    'maroon'         => ['bg' => '#A6493E', 'text' => '#fff'],
                    'yellow'         => ['bg' => '#DF9E43', 'text' => '#fff'],
                    'olive'          => ['bg' => '#8AA857', 'text' => '#fff'],
                    'lime'           => ['bg' => '#A1DE44', 'text' => '#fff'],
                    'green'          => ['bg' => '#4CA64C', 'text' => '#fff'],
                    'cyan'           => ['bg' => '#59BFB3', 'text' => '#fff'],
                    'teal'           => ['bg' => '#3366CC', 'text' => '#fff'],
                    'blue'           => ['bg' => '#33497A', 'text' => '#fff'],
                    'navy'           => ['bg' => '#2623A8', 'text' => '#fff'],
                    'magenta'        => ['bg' => '#A64C79', 'text' => '#fff'],
                    'purple'         => ['bg' => '#8342A6', 'text' => '#fff'],
                ],
            );
        });

        $container->share('calendar_states', function (Container $container): array {
            $cal_states = [];

            $states = $this->filter(
                $container,
                'calendar.filter_states',
                [
                    'active'   => 'calendar_status_active',
                    'disabled' => 'calendar_status_disabled',
                ],
            );

            /** @var \Application\Core\Localization $lang */
            $lang = $container->get('language');
            foreach ($states as $name => $state) {
                $cal_states[$name] = $lang->get($state);
            }

            return $cal_states;
        });

        $container->share('calendar_event_states', function (Container $container): array {
            $event_states = [];

            $states = $this->filter(
                $container,
                'calendar.filter_event_states',
                [
                    'open'      => 'calendar_status_open',
                    'closed'    => 'calendar_status_closed',
                    'confirmed' => 'calendar_status_confirmed',
                ],
            );

            /** @var \Application\Core\Localization $lang */
            $lang = $container->get('language');
            foreach ($states as $name => $state) {
                $event_states[$name] = $lang->get($state);
            }

            return $event_states;
        });

        $container->share('calendar_access_control', function (Container $container): Acl {
            /**
             * @var \Application\Core\User          $user
             * @var \Application\Core\Model\Account $account
             */
            $user = $container->get('current_user');
            $account = $user->getAccount();

            return new Acl($account);
        });

        /**
         * Calendario di base
         */

        $container->set('calendar_view_handler.base', function (Container $container): BaseView {
            return new BaseView(
                $container->get('calendar_data_handler.base'),
                $container->get('calendar_popup_handler.base'),
            );
        });

        $container->set('calendar_data_handler.base', function (Container $container): BaseData {
            return new BaseData(
                $container->get('request'),
                $container->get('current_user')->getAccount(),
                $container->get('calendar_access_control'),
                $container->get('calendar_colors'),
            );
        });

        $container->set('calendar_popup_handler.base', function (Container $container): BasePopup {
            return new BasePopup(
                $container->get('language'),
                $container->get('calendar_event_states'),
                $container->get('calendar_colors'),
            );
        });

        /**
         * Calendario globale amministrazione
         */

        $container->set('calendar_view_handler.admin', function (Container $container): AdminView {
            return new AdminView(
                $container->get('calendar_data_handler.base'),
                $container->get('calendar_popup_handler.admin'),
                $this->getSiteModule($container),
            );
        });

        $container->set('calendar_popup_handler.admin', function (Container $container): AdminPopup {
            return new AdminPopup(
                $container->get('language'),
                $container->get('calendar_event_states'),
                $container->get('calendar_colors'),
            );
        });

        /**
         * Calendario singolo amministrazione in modifica
         */

        $container->set('calendar_view_handler.admin_single', function (Container $container): AdminSingleView {
            return new AdminSingleView(
                $container->get('calendar_data_handler.base'),
                $container->get('calendar_popup_handler.admin_single'),
                $this->getSiteModule($container),
            );
        });

        $container->set('calendar_popup_handler.admin_single', function (Container $container): AdminSinglePopup {
            return new AdminSinglePopup(
                $container->get('language'),
                $container->get('calendar_event_states'),
                $container->get('calendar_colors'),
            );
        });

        /**
         * Calendario singolo pubblico in modifica
         */

        $container->set('calendar_view_handler.public_single', function (Container $container): SingleView {
            return new SingleView(
                $container->get('calendar_data_handler.base'),
                $container->get('calendar_popup_handler.public_single'),
            );
        });

        $container->set('calendar_popup_handler.public_single', function (Container $container): SinglePopup {
            return new SinglePopup(
                $container->get('language'),
                $container->get('calendar_event_states'),
                $container->get('calendar_colors'),
            );
        });

        /**
         * Calendario con vista risorse
         */

        $container->set('calendar_view_handler.resources', function (Container $container): ResourcesView {
            return new ResourcesView(
                $container->get('calendar_data_handler.base'),
                $container->get('calendar_popup_handler.admin_single'),
            );
        });
    }

    private function getSiteModule(Container $container): ModuleSite
    {
        if ($this->module_site === null) {
            $options = [
                'joins'      => 'INNER JOIN ' . Module::tableName() . ' AS m ON m.id = `from`.module_id',
                'conditions' => [
                    'm.controller = :controller AND `from`.site_id = :site',
                    'controller' => Kernel::FRONT_CONTROLLER,
                    'site'       => $container->get('site')->id,
                ],
            ];

            /** @var ModuleSite $site_module */
            $this->module_site = ModuleSite::first($options);
        }

        return $this->module_site;
    }

    /**
     * Applica un filtro.
     *
     * @param Container $container
     * @param string    $event_name
     * @param mixed     $value
     * @param array     $parameters
     *
     * @return mixed
     */
    private function filter(Container $container, $event_name, $value, array $parameters = [])
    {
        /** @var \Pongho\EventDispatcher\EventDispatcher $event_dispatcher */
        $event_dispatcher = $container->get('event_dispatcher');

        return $event_dispatcher->filter(new Event($this, $event_name, $parameters), $value)->getReturnValue();
    }
}
