<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author Daniele Termini
 * @package \Application\Calendar
 */

namespace Application\Calendar\Controller;

use Application\Calendar\Model\CalendarCategory;
use Application\Calendar\Model\CalendarEvent;
use Application\Core\Controller;
use Pongho\Core\TemplateResponse;
use Pongho\Http\Exception\HttpNotFoundException;
use Pongho\Http\JsonResponse;
use Pongho\Http\Response;
use Pongho\Template\View;
use Pongho\Utilities\DateTime;

class ApiController extends Controller
{
    public function calendarsAction()
    {
    }

    public function calendarAction()
    {
    }

    /**
     * Restituisce le categorie per il calendario selezionato
     *
     * @param null $calendar_id
     * @return array
     */
    protected function getCategoriesSelectOptions($calendar_id = null)
    {
        $categories = [];

        $options = [
            'conditions' => ['calendar_id = :calendar_id', 'calendar_id' => $calendar_id],
            'order'      => 'title ASC',
        ];

        /** @var \Application\Calendar\Model\CalendarCategory $category */
        foreach (CalendarCategory::all($options) as $category) {
            $categories[$category->id] = $category->title;
        }

        return $categories;
    }

    /**
     * Restituisce le categorie associate ad un calendario
     */
    public function categoriesAction()
    {
        $categories = [
            null => $this->getHelper()->getLocalization()->get('select_category'),
        ];

        $command = $this->getParameter('command');

        $segments = explode('/', $command);
        $calendar_id = array_shift($segments);

        if ($calendar_id) {
            foreach ($this->getCategoriesSelectOptions($calendar_id) as $category_id => $category) {
                $categories[$category_id] = $category;
            }
        }

        /** @var \Pongho\Template\Html $html */
        $html = $this->getContainer()->get('template_html');

        return new Response($html->select($categories));
    }

    /**
     * @return JsonResponse
     */
    public function eventsAction()
    {
        $handler_name = $this->getRequest()->query->get('handler', 'base');

        /** @var \Application\Calendar\Handler\ViewInterface $handler */
        $handler = $this->getContainer()->get('calendar_view_handler.' . $handler_name);

        return new JsonResponse($handler->getEvents());
    }

    /**
     * @return JsonResponse|Response
     * @throws \Pongho\Http\Exception\HttpNotFoundException
     */
    public function eventAction()
    {
        $event = $this->getEvent();

        if (empty($segments)) {
            $format = $this->getParameter('format');

            if ($format === 'json') {
                return new JsonResponse($event->attributes());
            }

            $view = new View($this->getHandler()->getPopupViewTemplatePath());
            $view->assignVars([
                'event' => $event,
                'lang'  => $this->getHelper()->getLocalization(),
            ]);

            return new Response($view->render());
        } else {
            $action = array_shift($segments);
            $method = $action . 'Command';

            if (method_exists($this, $method)) {
                return $this->$method($event, $segments);
            }

            throw new HttpNotFoundException();
        }
    }

    public function add_eventAction()
    {
        $request = $this->getRequest();

        $calendar_id = $request->query->get('calendar_id', null);

        // Se ho la gestione delle risorse, quando creo un evento seleziono il calendario della colonna cliccata
        if (!is_numeric($calendar_id)) {
            $command = $this->getParameter('command');

            $segments = explode('/', $command);
            $calendar_id = array_shift($segments);
        }

        if (!is_numeric($calendar_id)) {
            $calendar_id = null;
        }

        if ($request->getMethod() === 'POST') {
            $start = $request->post->get('event[start_date_at]', null, true);
            $end = $request->post->get('event[end_date_at]', null, true);
        } else {
            $start = $request->query->get('start');
            $end = $request->query->get('end');
        }

        $event = new CalendarEvent([
            'calendar_id'   => $calendar_id,
            'author_id'     => $this->getUser()->id,
            'is_all_day'    => (bool) $request->query->get('is_all_day', false),
            'start_date_at' => new DateTime($start),
            'end_date_at'   => new DateTime($end),
        ]);

        return $this->addEditEvent($event);
    }

    public function edit_eventAction()
    {
        return $this->addEditEvent($this->getEvent());
    }

    protected function addEditEvent(CalendarEvent $event)
    {
        $form = $this->getHandler()->getPopupForm($event, $this->getRequest());
        $form->handleRequest($this->getRequest());

        $view = new View($this->getHandler()->getPopupEditTemplatePath());
        $view->assignVars($this->getViewGlobalVariables());
        $view->assignVar('form', $form);

        if ($this->getRequest()->getMethod() === 'POST') {
            /** @var \Application\Calendar\Model\CalendarEvent $event */
            $event = $form->getSubject()->getModel();
            $event_data = $event->attributes();

            $color = $this->getColorData($event->calendar->color);
            $event_data['text_color'] = $color['text'];
            $event_data['bg_color'] = $color['bg'];

            /**
             * Converto le date in formato JSON corretto.
             *
             * @link https://git.metaline.it/pongho/pongho/-/issues/6
             */
            foreach (['start_date_at', 'end_date_at'] as $field) {
                if ($event_data[$field] instanceof DateTime) {
                    $event_data[$field] = $event_data[$field]->format('Y-m-d\TH:i:s\.000P');
                }
            }

            $response = [
                'event'   => $event_data,
                'form'    => $view->render(),
                'success' => $form->hasHandled() && !$form->hasErrors(),
                'action'  => $this->getAction(),
            ];

            return new JsonResponse($response);
        }

        return new TemplateResponse($view);
    }

    /**
     * Restituisce i dati del colore del calendario
     *
     * @param $calendar_color
     * @return array
     * @throws \DomainException
     */
    public function getColorData($calendar_color)
    {
        $colors = $this->getContainer()->get('calendar_colors');

        if (!array_key_exists($calendar_color, $colors)) {
            throw new \DomainException(sprintf('The color "%s" is not defined', $calendar_color));
        }

        return $colors[$calendar_color];
    }

    /**
     * Gestisce il ridimensionamento di un evento nel calendario
     *
     * @return JsonResponse
     */
    public function resize_eventAction()
    {
        return $this->updateEvent();
    }

    /**
     * Gestisce lo spostamento di un evento nel calendario
     *
     * @return JsonResponse
     */
    public function drop_eventAction()
    {
        return $this->updateEvent();
    }

    /**
     * Aggiorna gli attributi di un evento
     *
     * @return JsonResponse
     */
    protected function updateEvent()
    {
        $event = $this->getEvent();

        /** @var \Pongho\Http\Request $request */
        $request = $this->getRequest();

        $event->updateAttributes($request->post->all());
        if ($event->save()) {
            return new JsonResponse(['success' => true]);
        }

        return new JsonResponse(['error' => true, 'message' => 'È avvenuto un errore durante la modifica dell\'evento']);
    }

    /**
     * @return JsonResponse
     */
    public function delete_eventAction()
    {
        $event = $this->getEvent();

        $response = [
            'action'  => 'delete',
            'success' => $event->delete(),
        ];

        return new JsonResponse($response);
    }

    protected function partecipantsCommand(CalendarEvent $event, array $segments = [])
    {
    }

    protected function add_partecipantCommand(CalendarEvent $event, array $segments = [])
    {
    }

    protected function edit_partecipantCommand(CalendarEvent $event, array $segments = [])
    {
        $partecipant_id = array_shift($segments);
    }

    protected function delete_partecipantCommand(CalendarEvent $event, array $segments = [])
    {
        $partecipant_id = array_shift($segments);
    }

    /**
     * @return array
     */
    protected function getViewGlobalVariables()
    {
        return [
            'head'     => $this->getHelper()->getThemeHeaderHelper(),
            'html'     => $this->container->get('template_html'),
            'lang'     => $this->getHelper()->getLocalization(),
            'language' => $this->getHelper()->getLanguage(),
            'site'     => $this->getHelper()->getSite(),
            'user'     => $this->getHelper()->getUser(),
        ];
    }

    /**
     * @return \Application\Calendar\Model\CalendarEvent
     * @throws \Pongho\Http\Exception\HttpNotFoundException
     */
    protected function getEvent()
    {
        $command = $this->getParameter('command');

        $segments = explode('/', $command);
        $event_id = array_shift($segments);

        $event = CalendarEvent::find($event_id);

        if ($event === null) {
            throw new HttpNotFoundException();
        }

        return $event;
    }

    /**
     * @return \Application\Calendar\Handler\ViewInterface
     */
    public function getHandler()
    {
        $handler_name = 'calendar_view_handler.' . $this->getRequest()->query->get('handler', 'base');

        return $this->getContainer()->get($handler_name);
    }
}
