<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Calendar
 */

namespace Application\Calendar\Controller\Admin;

use Application\Admin\Controller\ModuleOptionsController;
use Application\Core\Model\LanguageModule;

/**
 * CalendarOptionsController
 */
class CalendarOptionsController extends ModuleOptionsController
{
    /**
     * Opzioni principali.
     */
    public function indexAction()
    {
        $calendar_views = $this->getContainer()->get('calendar_views');

        $options = [
            'enabled_views'               => 'checklist',
            'default_calendar_admin_view' => 'string',
            'min_calendar_day_time'       => 'string',
            'max_calendar_day_time'       => 'string',
            'first_day'                   => 'integer',
        ];

        $this->handleRequest($options);

        $this->getHelper()->getBodyView()
            ->setTemplatePath(__DIR__ . '/../../Resources/views/calendar_options.php')
            ->assignVars([
                'action_form'                  => $this->url('/' . $this->getParameter('path') . '/'),
                'default_calendar_admin_views' => $calendar_views,
                'enabled_views'                => $calendar_views,
            ]);

        $this->displayTabs();
    }

    /**
     * Lingue e traduzioni.
     */
    public function languagesAction()
    {
        $this->handleLanguages();

        $enabled_languages = [];
        foreach ($this->getHelper()->getSite()->getLanguages() as $language_site) {
            if (!$language_site->is_enabled) {
                continue;
            }

            $language_site->flag = sprintf(
                '<img class="flag" src="%s" alt="%s" title="%s" />',
                pongho_url("/Application/Core/Resources/flags/16/{$language_site->culture}.png"),
                $language_site->iso,
                $language_site->name
            );

            $language_module = new LanguageModule();
            $language_module->is_enabled = false;

            $enabled_languages[$language_site->language_site_id] = [
                'language' => $language_site,
                'module'   => $language_module,
            ];
        }

        $languages_modules = LanguageModule::allByModuleAndSite($this->getHelper()->getModuleId(), $this->getSiteId());

        foreach ($languages_modules as $language_module) {
            if (array_key_exists($language_module->language_site_id, $enabled_languages)) {
                $enabled_languages[$language_module->language_site_id]['module'] = $language_module;
            }
        }

        $this->getHelper()->getBodyView()
            ->setTemplatePath(__DIR__ . '/../../Resources/views/calendar_options_languages.php')
            ->assignVars([
                'action_form'       => $this->url('/' . $this->getParameter('path') . '/languages/'),
                'enabled_languages' => $enabled_languages,
            ]);

        $this->displayTabs();
    }

    /**
     * @param array $post_options
     * @param       $field_name
     * @return array
     */
    protected function getFieldChecklist(array $post_options, $field_name)
    {
        return isset($post_options[$field_name]) && is_array($post_options[$field_name]) ? $post_options[$field_name] : [];
    }

    /**
     * Salva le lingue abilitate o disabilitate per il modulo corrente.
     */
    protected function handleLanguages()
    {
        if (isset($_POST['languages']) && is_array($_POST['languages'])) {
            foreach ($_POST['languages'] as $language_site_id => $attributes) {
                // Salto le righe vuote
                if (empty($attributes['name']) || empty($attributes['path'])) {
                    continue;
                }

                $options = [
                    'conditions' => [
                        'position < :pages AND language_site_id = :language',
                        'pages'    => 99999,
                        'language' => $language_site_id,
                    ],
                    'order'      => 'position DESC',
                    'limit'      => 1,
                ];

                // Fallback, se non trovo nessun modulo parto dalla posizione 10 (anche se è impossibile)
                $position = 10;
                /** @var \Application\Core\Model\LanguageModule $last_module */
                if (($last_module = LanguageModule::first($options)) !== null) {
                    $position = $last_module->position + 10;
                }

                $attributes['is_enabled'] = isset($attributes['is_enabled']);

                if (($language = LanguageModule::findByLanguageSiteAndModule(
                    $language_site_id,
                    $this->getHelper()->getModuleId()
                )) === null
                ) {
                    $attributes['language_site_id'] = $language_site_id;
                    $attributes['module_id'] = $this->getHelper()->getModuleId();
                    $attributes['position'] = $position;

                    $language = LanguageModule::create($attributes);
                } else {
                    $language->updateAttributes($attributes);
                }

                $language->save();
            }
        }
    }

    /**
     * Visualizza le tabs con le varie categorie di opzione.
     */
    protected function displayTabs()
    {
        $this->getHelper()->getBodyView()
            ->assignVar('tabs', $this->getTabs());
    }

    /**
     * Restituisce la lista delle tabs disponibili.
     */
    protected function getTabs()
    {
        /** @var \Application\Core\I18n\Translation\Translator $translator */
        $translator = $this->getContainer()->get('translator');
        $path = $this->getParameter('path');

        return [
            (object) [
                'name'      => $translator->trans('Main'),
                'link'      => $this->url("/{$path}/"),
                'is_active' => ($this->getAction() == 'index'),
            ],
            (object) [
                'name'      => $translator->trans('Languages'),
                'link'      => $this->url("/{$path}/languages/"),
                'is_active' => ($this->getAction() == 'languages'),
            ],
        ];
    }

    /**
     * Restituisce l'helper per l'admin.
     *
     * @return \Application\Cms\Utilities\NodeTypeHelper
     */
    public function getHelper()
    {
        /** @var \Application\Cms\Utilities\NodeTypeHelper $helper */
        $helper = parent::getHelper();

        $helper->setModuleSiteClass(\Application\Calendar\Model\ModuleSite::class);

        return $helper;
    }
}
