<?php

namespace Application\Calendar\Controller\Admin;

use Application\Admin\Controller\CrudFormController;
use Application\Admin\Form\FormConfig;
use Application\Calendar\Handler\View\AdminSingleView;
use Application\Calendar\Model\Calendar;
use Application\Calendar\Model\CalendarCategory;
use Application\Calendar\Model\CalendarUser;
use Application\Core\Model\Account;
use Pongho\Form\Field;
use Pongho\Form\Field\ChecklistField;
use Pongho\Form\Repeater\ModelRepeater;
use Pongho\Form\Repeater\RepeaterRow;
use Pongho\Form\Subject\ModelRowSubject;
use Pongho\Form\Subject\ModelSubject;
use Pongho\Form\Subject\SubjectInterface;
use Pongho\Http\RedirectResponse;
use Pongho\Template\View;
use Pongho\Template\ViewClosure;

class CalendarController extends CrudFormController
{
    /**
     * {@inheritdoc}
     */
    public function getNamespace()
    {
        return 'calendar';
    }

    /**
     * {@inheritdoc}
     */
    public function getEventNamespace($event_name)
    {
        return 'admin.calendar.' . $event_name;
    }

    /**
     * {@inheritdoc}
     */
    protected function getArchiveTitle()
    {
        return $this->getContainer()->get('translator')->trans('Calendars');
    }

    /**
     * {@inheritdoc}
     */
    protected function getArchiveAddButtonText()
    {
        return $this->getContainer()->get('translator')->trans('Add calendar');
    }

    /**
     * {@inheritdoc}
     */
    protected function getArchiveAddFirstText()
    {
        return $this->getContainer()->get('translator')->trans('Add the first calendar');
    }

    /**
     * {@inheritdoc}
     */
    public function getAddEditTitle($model)
    {
        /** @var \Application\Core\I18n\Translation\Translator $translator */
        $translator = $this->getContainer()->get('translator');

        if ($model->isNewRecord()) {
            return $translator->trans('Add calendar');
        }

        return $translator->trans('Edit the calendar “%title%”', ['%title%' => $model->title]);
    }

    /**
     * {@inheritdoc}
     */
    protected function hasAddButton()
    {
        return true;
    }

    /**
     * {@inheritdoc}
     */
    protected function getModel()
    {
        /** @var Calendar $model */
        $model = parent::getModel();

        $model->module_site_id = $this->getHelper()->getModuleSite()->id;

        return $model;
    }

    /**
     * Restituisce la chiave della tab che deve essere attivata
     *
     * @return string
     */
    protected function getActiveTab()
    {
        $default_tab = $this->getAction() === 'add' ? 'main' : 'calendar';

        $tab = $this->getRequest()->post->get('tabs[tab]', $default_tab, true);
        $tab = $this->getRequest()->query->get('tabs[tab]', $tab, true);

        return $tab;
    }

    /**
     * {@inheritdoc}
     */
    protected function getFormsOptions()
    {
        $subject = new ModelSubject($this->getModel());
        $subject->attachGetter('categories', [$this, 'getCalendarCategoriesCallback']);
        $subject->attachGetter('shares', [$this, 'getCalendarUsersCallback']);
        $subject->attachSetter('permissions', [$this, 'setPermissionsCallback']);
        $subject->attachGetter('permissions', [$this, 'getPermissionsCallback']);

        if ($this->getAction() === 'add') {
            $subject->set('color', 'lightsteelblue');
        }

        /** @var \Application\Core\I18n\Translation\Translator $translator */
        $translator = $this->getContainer()->get('translator');

        $config = new FormConfig('calendar', $subject, $this->getHelper()->getLocalization());

        $config->addBaseStructure($this->getActiveTab(), $this->currentUrl());
        $config->addTab('content/main', $translator->trans('Main'));
        $config->addTab('content/shares', $translator->trans('Calendar shares and permissions'));

        $config->addPanel('content/main/calendar');
        $config->addPanel('content/main/categories', $translator->trans('Event categories'));

        $config->addPanel(
            'content/shares/public-permissions',
            $translator->trans('Public permissions'),
            false,
            [
                'description' => $translator->trans('Configure the public permissions for the calendar, all the users will be able to perform the following enabled actions:'),
            ]
        );
        $config->addPanel(
            'content/shares/shares-panel',
            $translator->trans('Calendar share'),
            false,
            [
                'description' => $translator->trans('Configure shares and permissions for specific users.<br>The specific permissions completely override the public permissions.'),
            ]
        );

        /**
         * Visualizzazione del calendario
         */
        if ($this->getAction() === 'edit') {
            $config->addTabBefore('content/main', 'calendar', $translator->trans('Calendar'));
            $config->addField(
                'content/calendar/calendar-view',
                [
                    'class'    => \Pongho\Form\Fieldset::class,
                    'settings' => [
                        'view' => new ViewClosure([$this, 'getCalendarViewCallback']),
                    ],
                ]
            );

            $this->getHelper()
                ->addCss(pongho_url('/Application/Admin/Resources/vendor/fullcalendar/fullcalendar.css'))
                ->addJavaScript(pongho_url('/Application/Admin/Resources/vendor/xdate/xdate.js'))
                ->addJavaScript(pongho_url('/Application/Admin/Resources/vendor/fullcalendar/fullcalendar.js'))
                ->addJavaScript(pongho_url('/Application/Calendar/Resources/public/js/jquery.calendar.js'));
        }

        /**
         * Configurazione del calendario
         */
        $config->addFields(
            'content/main/calendar',
            [
                'permalink-editor' => [
                    'class'      => \Pongho\Form\Fieldset::class,
                    'attributes' => [
                        'class' => 'permalink-editor',
                    ],
                    'settings'   => [
                        'hide_label' => true,
                    ],
                ],
                'owner_id'         => [
                    'class'       => \Pongho\Form\Field\TextField::class,
                    'label'       => $translator->trans('Owner'),
                    'attributes'  => [
                        'required' => 'required',
                    ],
                    'description' => $translator->trans('Search for name, surname or email'),
                    'settings'    => [
                        'view' => [$this, 'getUserSelectView'],
                    ],
                ],
                'color'            => [
                    'class'      => \Pongho\Form\Field\ColorPickerField::class,
                    'label'      => $translator->trans('Color'),
                    'attributes' => [
                        'data-show-palette-only' => true,
                        'data-show-palette'      => true,
                        'data-palette'           => html_escape(json_encode(array_keys($this->getContainer()->get('calendar_colors')))),
                        'data-preferred-format'  => 'name',
                    ],
                ],
                'status'           => [
                    'class'   => \Pongho\Form\Field\SelectField::class,
                    'label'   => $translator->trans('Status'),
                    'options' => $this->getContainer()->get('calendar_states'),
                ],
            ]
        );


        $view = new View(PONGHO_PATH . '/Application/Calendar/Resources/views/edit_permalink_editor.php');
        $view
            ->assignVars($this->getHelper()->getViewGlobalVariables())
            ->assignVars(
                [
                    'model'  => $subject->getModel(),
                    'action' => $this->getAction(),
                    'path'   => trim($this->getHelper()->getPath(), '/'),
                ]
            );
        $config->addFields(
            'content/main/calendar/permalink-editor',
            [
                'title' => [
                    'class'      => \Pongho\Form\Field\TextField::class,
                    'label'      => $translator->trans('Calendar name'),
                    'attributes' => [
                        'class' => 'field-title input_text',
                    ],
                    'settings'   => [
                        'view' => [$this, 'getTitleViewCallback'],
                    ],
                ],
                'slug'  => [
                    'class'      => \Pongho\Form\Field\TextField::class,
                    'label'      => $translator->trans('Calendar url'),
                    'attributes' => [
                        'class' => 'slug-field input_text',
                    ],
                    'settings'   => [
                        'view' => $view,
                    ],
                ],
            ]
        );

        /**
         * Categorie
         */
        $config->addField(
            'content/main/categories/categories',
            [
                'class'    => \Pongho\Form\Repeater\ModelRepeater::class,
                'settings' => [
                    'filter_subject' => function ($key, $request, ModelRepeater $repeater) {
                        if (!is_numeric($key)) {
                            $row_subject = new ModelRowSubject(new CalendarCategory([
                                'calendar_id' => $repeater->getSubject()->get('id'),
                            ]));
                        } else {
                            $row_subject = new ModelRowSubject(CalendarCategory::find($key));
                        }

                        return $row_subject;
                    },
                    'row_settings'   => [
                        'show-delete'  => true,
                        'delete-label' => $translator->trans('Remove'),
                    ],
                ],
            ]
        );

        $config->addRepeaterRow('content/main/categories/categories');

        $config->addField(
            'content/main/categories/categories/main/title',
            [
                'class' => \Pongho\Form\Field\Textfield::class,
                'label' => $translator->trans('Category name'),
            ]
        );

        /**
         * Configurazione delle condivisioni e dei permessi
         */
        $config->addFields(
            'content/shares/public-permissions',
            [
                'permissions' => [
                    'class'   => \Pongho\Form\Field\ChecklistField::class,
                    'label'   => $translator->trans('Public permissions'),
                    'options' => [
                        'can_add'           => $translator->trans('Can add'),
                        'can_edit_self'     => $translator->trans('Can edit self'),
                        'can_delete_self'   => $translator->trans('Can delete self'),
                        'can_view_others'   => $translator->trans('Can view all'),
                        'can_edit_others'   => $translator->trans('Can edit all'),
                        'can_delete_others' => $translator->trans('Can delete all'),
                    ],
                ],
            ]
        );

        $config->addFields(
            'content/shares/shares-panel',
            [
                'user_id' => [
                    'class'       => \Pongho\Form\Field\TextField::class,
                    'label'       => $translator->trans('Share the calendar with'),
                    'description' => $translator->trans('Search for name, surname or email'),
                    'attributes'  => ['data-url' => url('/api/core/autocomplete/user_id/')],
                ],
                'shares'  => [
                    'class'    => \Pongho\Form\Repeater\ModelRepeater::class,
                    'settings' => [
                        'filter_subject' => [$this, 'filterSharesCallback'],
                        'row_settings'   => [
                            'show-delete'  => true,
                            'delete-label' => $translator->trans('Remove'),
                            'title'        => function (RepeaterRow $row) {
                                $row_subject = $row->getSubject();

                                /** @var \Application\Core\Model\Account $account */
                                $account = Account::find($row_subject->get('user_id'));
                                $name = '';
                                if ($account) {
                                    $name = $account->name();
                                }

                                return '<span class="shares-user-name">' . $name . '</span>';
                            },
                        ],
                    ],
                ],
            ]
        );

        $config->addRepeaterRow('content/shares/shares-panel/shares');

        $config->addFields(
            'content/shares/shares-panel/shares/main',
            [
                'user_id'     => [
                    'class'      => \Pongho\Form\Field\HiddenField::class,
                    'attributes' => ['class' => 'user-id'],
                ],
                'permissions' => [
                    'class'    => \Pongho\Form\Field\ChecklistField::class,
                    'options'  => [
                        'can_add'           => $translator->trans('Can add'),
                        'can_edit_self'     => $translator->trans('Can edit self'),
                        'can_delete_self'   => $translator->trans('Can delete self'),
                        'can_view_others'   => $translator->trans('Can view all'),
                        'can_edit_others'   => $translator->trans('Can edit all'),
                        'can_delete_others' => $translator->trans('Can delete all'),
                    ],
                    'settings' => [
                        'view' => new ViewClosure(function (ChecklistField $field) {
                            return '<div class="shares-controls">' . $field->getTag() . '</div>';
                        }),
                    ],
                ],
            ]
        );

        return $config;
    }

    /**
     * Callback per ottenere il template del campo del titolo
     *
     * @return \Pongho\Template\View
     */
    public function getTitleViewCallback()
    {
        $view = new View(PONGHO_PATH . '/Application/Calendar/Resources/views/edit_title.php');

        $view->assignVars($this->getHelper()->getViewGlobalVariables());

        $view->assignVars(
            [
                // @todo: se fallisce la transazione in "add", non deve darmi l'id del modello in quanto non esiste. Questo fix è sbagliato qui, dovrebbe essere compito di ActiveRecord dirmi che il modello non è stato salvato
                'model_id'    => $this->getAction() === 'add' ? null : $this->getModel()->id,
                'slug_action' => $this->url(sprintf('/' . $this->getParameter('path') . '/slug/')),
            ]
        );

        return $view;
    }

    /**
     * Azione per l'impostazione dello slug del calendario.
     *
     * @access public
     */
    public function slugAction()
    {
        if (($object = $this->getRequest()->query->get('object')) === null) {
            return $this->getHelper()->displayJsonError('The "object" parameter is missing!');
        }

        if (($title = $this->getRequest()->query->get('title')) === null) {
            return $this->getHelper()->displayJsonError('The "title" parameter is missing!');
        }

        $model_id = intval($object);

        /** @var Calendar $model */
        if ($model_id) {
            if (($model = Calendar::find($model_id)) === null) {
                return $this->getHelper()->displayJsonError('The model does not exist!');
            }
        } else {
            $model = new Calendar();
            $model->module_site_id = $this->getHelper()->getModuleSite()->id;
        }

        $model->title = $title;
        $model->createPermalink();

        return $this->getHelper()->displayJsonMessage('ok!', false, ['slug' => $model->slug]);
    }

    /**
     * @param ModelSubject $subject
     * @return array
     */
    public function getCalendarCategoriesCallback(ModelSubject $subject)
    {
        /** @var Calendar $model */
        $model = $subject->getModel();

        $categories = [];
        foreach ($model->getCategories() as $category) {
            $categories[$category->id] = new ModelRowSubject($category);
        }

        return $categories;
    }

    /**
     * @param ModelSubject $subject
     * @return array
     */
    public function getCalendarUsersCallback(ModelSubject $subject)
    {
        /** @var Calendar $model */
        $model = $subject->getModel();

        $users = [];
        foreach ($model->getUsers() as $user) {
            $row_subject = new ModelRowSubject($user);

            $row_subject->attachSetter('permissions', [$this, 'setPermissionsCallback']);
            $row_subject->attachGetter('permissions', [$this, 'getPermissionsCallback']);

            $users[$user->user_id] = $row_subject;
        }

        return $users;
    }

    /**
     * Filtra i subject delle righe del Repeater delle condivisioni
     *
     * @param               $key
     * @param               $request
     * @param ModelRepeater $repeater
     * @return ModelRowSubject
     */
    public function filterSharesCallback($key, $request, ModelRepeater $repeater)
    {
        $calendar_id = $repeater->getSubject()->get('id');

        if (!is_numeric($key)) {
            $row_subject = new ModelRowSubject(new CalendarUser(['calendar_id' => $calendar_id]));
        } else {
            $row_subject = new ModelRowSubject(CalendarUser::find([$key, $calendar_id]));
        }

        $row_subject->attachSetter('permissions', [$this, 'setPermissionsCallback']);
        $row_subject->attachGetter('permissions', [$this, 'getPermissionsCallback']);

        return $row_subject;
    }

    /**
     * Gestisce la trasformazione da checklist a colonne
     *
     * @param SubjectInterface $subject
     * @param                 $values
     */
    public function setPermissionsCallback(SubjectInterface $subject, $values)
    {
        $fields = [
            'can_add',
            'can_edit_self',
            'can_delete_self',
            'can_view_others',
            'can_edit_others',
            'can_delete_others',
        ];

        foreach ($fields as $field) {
            $subject->set($field, in_array($field, $values));
        }
    }

    /**
     * Gestisce la trasformazione da colonne a checklist
     *
     * @param SubjectInterface $subject
     * @return array
     */
    public function getPermissionsCallback(SubjectInterface $subject)
    {
        $values = [];
        $fields = [
            'can_add',
            'can_edit_self',
            'can_delete_self',
            'can_view_others',
            'can_edit_others',
            'can_delete_others',
        ];

        foreach ($fields as $field) {
            if ($subject->get($field)) {
                $values[] = $field;
            }
        }

        return $values;
    }

    /**
     * Vista per il campo di selezione utente
     *
     * @param Field $field
     * @return View
     */
    public function getUserSelectView(Field $field)
    {
        /** @var \Application\Core\I18n\Translation\Translator $translator */
        $translator = $this->getContainer()->get('translator');
        $view = new View(__DIR__ . '/../../../Admin/Resources/views/autosuggest.php');

        if ($this->getAction() === 'add') {
            $selected = $translator->trans('Search for a user to select');
        } else {
            /** @var \Application\Core\Model\Account $account */
            $account = Account::find($field->getForm()->getSubject()->get('owner_id'));

            $selected = $account->name();
        }

        $view->assignVars([
            'lang'             => $this->getHelper()->getLocalization(),
            'translator'       => $translator,
            'autocomplete_url' => url('/api/core/autocomplete/user_id/'),
            'selected'         => $selected,
        ]);

        return $view;
    }

    /**
     * Callback per impostare il calendario all'interno della form di modifica
     *
     * @throws \LogicException
     * @return string
     */
    public function getCalendarViewCallback()
    {
        $subject = $this->getForm()->getSubject();
        $calendar_id = $subject->get('id');

        $handler_name = 'calendar_view_handler.' . $this->getHelper()->filter(
                $this,
                $this->getEventNamespace('filter_edit_calendar_handler'),
                'admin_single'
            );

        /** @var \Application\Calendar\Handler\View\AdminSingleView $handler */
        $handler = $this->getContainer()->get($handler_name);

        if (!($handler instanceof AdminSingleView)) {
            throw new \LogicException('View Handler must be an instance of \Application\Calendar\Handler\View\AdminSingleView');
        }

        $handler->setCalendar($calendar_id);
        $handler->populateHead($this->getHelper()->getHead());

        return '<div id="calendar" style="min-height: 800px; margin: 8px;"></div>';
    }

    /**
     * {@inheritdoc}
     */
    public function getModelClass()
    {
        return Calendar::class;
    }

    /**
     * Azione per la vista del calendario
     */
    public function viewAction()
    {
        $path = $this->getParameter('path');

        $handler_name = 'calendar_view_handler.' . $this->getHelper()->filter(
                $this,
                $this->getEventNamespace('filter_calendar_handler'),
                'admin'
            );

        /** @var \Application\Calendar\Handler\ViewInterface $handler */
        $handler = $this->getContainer()->get($handler_name);

        $calendars = $handler->getCalendars();

        if (count($calendars)) {
            $add_first_calendar = '';
        } else {
            $add_first_calendar = sprintf(
                '<div class="empty"><a href="%1$s" title="%2$s">%2$s</a></div>',
                $this->url("/{$path}/add/"),
                $this->getContainer()->get('translator')->trans('Add the first calendar')
            );
        }

        $this->getHelper()
            ->addCss(pongho_url('/Application/Admin/Resources/vendor/fullcalendar/fullcalendar.css?v=' . filemtime(PONGHO_PATH . '/Application/Admin/Resources/vendor/fullcalendar/fullcalendar.css')))
            ->addJavaScript(pongho_url('/Application/Admin/Resources/vendor/xdate/xdate.js?v=' . filemtime(PONGHO_PATH . '/Application/Admin/Resources/vendor/xdate/xdate.js')))
            ->addJavaScript(pongho_url('/Application/Admin/Resources/vendor/fullcalendar/fullcalendar.js?v=' . filemtime(PONGHO_PATH . '/Application/Admin/Resources/vendor/fullcalendar/fullcalendar.js')))
            ->addJavaScript(pongho_url('/Application/Admin/Resources/vendor/jqueryui-nestedsortable/jquery.ui.nestedSortable.js?v=' . filemtime(PONGHO_PATH . '/vendor/jQueryUI-widgets/nestedSortable/jquery.ui.nestedSortable.js')))
            ->addCss(pongho_url('/Application/Calendar/Resources/views/css/style.css?v=' . filemtime(PONGHO_PATH . '/Application/Calendar/Resources/views/css/style.css')))
            ->addJavaScript(pongho_url('/Application/Core/Resources/public/js/polyfill.js?v=' . filemtime(PONGHO_PATH . '/Application/Core/Resources/public/js/polyfill.js')))
            ->addJavaScript(pongho_url('/Application/Calendar/Resources/public/js/jquery.calendar.js?v=' . filemtime(PONGHO_PATH . '/Application/Calendar/Resources/public/js/jquery.calendar.js')))
            ->addJavaScript(pongho_url('/Application/Admin/Resources/views/js/plugins.js?v=' . filemtime(PONGHO_PATH . '/Application/Admin/Resources/views/js/plugins.js')))
            ->addJavaScript(pongho_url('/Application/Admin/Resources/views/js/scripts.js?v=' . filemtime(PONGHO_PATH . '/Application/Admin/Resources/views/js/scripts.js')))
        ;

        $handler->populateHead($this->getHelper()->getHead());

        $this->getHelper()->getBodyView()
            ->setTemplatePath(__DIR__ . '/../../Resources/views/calendar.php')
            ->assignVar('add_first_calendar', $add_first_calendar);

        $this->getHelper()->notify($this, $this->getEventNamespace('after_set_view'));
    }

    /**
     * @return RedirectResponse
     */
    public function moveupAction()
    {
        /** @var Calendar $calendar */
        $calendar = Calendar::find($this->getParameter('id', null, true));

        return $this->move($calendar, -15);
    }

    /**
     * @return RedirectResponse
     */
    public function movedownAction()
    {
        /** @var Calendar $calendar */
        $calendar = Calendar::find($this->getParameter('id', null, true));

        return $this->move($calendar, 15);
    }

    /**
     * @param Calendar $calendar
     * @param          $direction
     * @return RedirectResponse
     */
    protected function move(Calendar $calendar, $direction)
    {
        Calendar::transaction(
            function () use ($calendar, $direction): void {
                $calendar->position += $direction;
                $calendar->save();

                $position = 10;
                /** @var Calendar $cal */
                foreach (Calendar::all(['order' => 'position']) as $cal) {
                    $cal->position = $position;
                    $cal->save();

                    $position += 10;
                }
            }
        );

        return new RedirectResponse($this->url('/' . $this->getParameter('path') . '/'));
    }

    /**
     * @access protected
     * @var int
     */
    protected $parse_row_counter = 1;

    /**
     * {@inheritdoc}
     */
    protected function parseArchiveRowOrderActions(Calendar $row)
    {
        /** @var \Application\Core\I18n\Translation\Translator $translator */
        $translator = $this->getContainer()->get('translator');

        $path = $this->getParameter('path');

        // Aggiungo l’opzione di ordinamento
        $order_actions = [];

        if ($this->parse_row_counter !== 1) {
            $order_actions[] = [
                $translator->trans('Up'),
                'href'  => $this->url("/{$path}/moveup/{$row->id}/"),
                'title' => $translator->trans('Move up'),
                'class' => 'move-up',
            ];
        }

        if ($this->parse_row_counter !== $this->getArchiveManager()->getTotalRecords()) {
            $order_actions[] = [
                $translator->trans('Down'),
                'href'  => $this->url("/{$path}/movedown/{$row->id}/"),
                'title' => $translator->trans('Move down'),
                'class' => 'move-down',
            ];
        }

        $row->order_actions = $this->parseActions($order_actions);
    }

    /**
     * {@inheritdoc}
     */
    public function parseArchiveRow($row)
    {
        parent::parseArchiveRow($row);
        $this->parseArchiveRowOrderActions($row);

        $this->getHelper()->filter($this, $this->getEventNamespace('filter_calendar'), $row);

        $this->parse_row_counter++;
    }

    /**
     * {@inheritdoc}
     */
    protected function getModelOptions()
    {
        $options = parent::getModelOptions();

        $options['order'] = 'position ASC';

        return $options;
    }


    /**
     * {@inheritdoc}
     */
    protected function getTableColumns()
    {
        /** @var \Application\Core\I18n\Translation\Translator $translator */
        $translator = $this->getContainer()->get('translator');

        $columns = [
            [
                'name'   => 'order',
                'label'  => $translator->trans('Order'),
                'class'  => 'order',
                'render' => function ($row) {
                    return $row->order_actions;
                },
            ],
            [
                'name'      => 'title',
                'label'     => $translator->trans('Title'),
                'class'     => 'main',
                'orderable' => true,
                'render'    => function ($row) {
                    return '<strong>' . $row->title . '</strong><div class="actions"><span>' . $row->actions . '</span></div>';
                },
            ],
            [
                'name'   => 'color',
                'label'  => $translator->trans('Color'),
                'render' => [$this, 'getColorColumnCallback'],
            ],
            [
                'name'      => 'id',
                'label'     => 'ID',
                'class'     => 'id',
                'orderable' => true,
            ],
        ];

        return $columns;
    }

    /**
     * Callback per il render della colonna dei colori nell'elenco dei calendari
     *
     * @param Calendar $row
     * @return string
     */
    public function getColorColumnCallback(Calendar $row)
    {
        $colors = $this->getContainer()->get('calendar_colors');
        $color_code = $colors[$row->color]['bg'];

        return '<div style="background-color: ' . $color_code . '; border: 1px solid #CCC; height: 2em; width: 2em;" ></div>';
    }

    /**
     * Callback
     */
    public function beforeSetTemplateCallback(/* Calendar $calendar */)
    {
        $this->getHelper()
            ->addCss(pongho_url('/Application/Calendar/Resources/views/css/style.css?v=' . __DIR__ . '/../../Resources/views/css/style.css'))
            ->addJavaScript(pongho_url('/Application/Calendar/Resources/views/js/jquery.calendar-edit.js?v=' . __DIR__ . '/../../Resources/views/js/jquery.calendar-edit.css'));
    }

    /**
     * Callback
     * @param Calendar $node
     */
    protected function beforeAddCallback(Calendar $node)
    {
        $node->module_site_id = $this->getSiteModuleId();
    }

    /**
     * {@inheritdoc}
     */
    protected function getSiteModuleClassName()
    {
        return \Application\Calendar\Model\ModuleSite::class;
    }

    /**
     * Restituisce l'elenco delle opzioni della select dei calendari disponibili
     *
     * @return array
     */
    public function getCalendarsSelectOptions()
    {
        $calendars = [];
        $colors = $this->getContainer()->get('calendar_colors');

        /** @var Calendar $calendar */
        foreach (Calendar::all(['options' => 'title ASC']) as $calendar) {
            $calendars[$calendar->id] = [
                'title'           => $calendar->title,
                'data-text-color' => $colors[$calendar->color]['text'],
                'data-bg-color'   => $colors[$calendar->color]['bg'],
            ];
        }

        return $calendars;
    }
}
