<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author Daniele Termini
 * @package @todo
 */

namespace Application\Calendar;

use Application\Calendar\Model\Calendar;
use Application\Calendar\Model\CalendarEvent;
use Application\Calendar\Model\CalendarUser;
use Application\Core\Model\Account;

class Acl
{
    /**
     * @var \Application\Core\Model\Account
     */
    protected $account;

    /**
     * @var CalendarUser[]
     */
    protected $privileges;

    public function __construct(Account $account)
    {
        $this->account = $account;

        $this->loadPrivileges();
    }

    protected function loadPrivileges()
    {
        $options = [
            'conditions' => [
                'user_id = :user',
                'user' => $this->account->id,
            ],
        ];

        /** @var CalendarUser $calendar_user */
        foreach (CalendarUser::all($options) as $calendar_user) {
            $this->privileges[$calendar_user->calendar_id] = $calendar_user;
        }
    }

    public function isActive($calendar)
    {
        return $calendar->status === 'active';
    }

    /**
     * @return bool
     */
    public function isOwner($calendar)
    {
        return $this->account->is_founder || $calendar->owner_id === $this->account->id;
    }

    /**
     * @return array
     */
    public function getEditableCalendars()
    {
        $calendar_select = [];
        $options = [];

        /** @var Calendar $calendar */
        foreach (Calendar::all($options) as $calendar) {
            if ($this->canAdd($calendar)) {
                $calendar_select[$calendar->id] = $calendar->title;
            }
        }

        return $calendar_select;
    }

    /**
     * @return bool
     */
    public function canAdd(Calendar $calendar)
    {
        if ($this->isOwner($calendar)) {
            return true;
        }

        if (array_key_exists($calendar->id, $this->privileges)) {
            return $this->privileges[$calendar->id]->can_add;
        }

        return $calendar->can_add;
    }

    /**
     * @return bool
     */
    public function canEdit(CalendarEvent $event)
    {
        $calendar = $event->calendar;

        if ($this->isOwner($calendar)) {
            return true;
        }

        if (array_key_exists($calendar->id, $this->privileges)) {
            if ($event->author_id === $this->account->id) {
                if ($this->privileges[$calendar->id]->can_edit_self) {
                    return true;
                }
            } else {
                if ($this->privileges[$calendar->id]->can_edit_others) {
                    return true;
                }
            }
        }

        if ($event->author_id === $this->account->id) {
            if ($calendar->can_edit_self) {
                return true;
            }
        } else {
            if ($calendar->can_edit_others) {
                return true;
            }
        }

        return false;
    }

    /**
     * @return bool
     */
    public function canDelete(CalendarEvent $event)
    {
        $calendar = $event->calendar;

        if ($this->isOwner($calendar)) {
            return true;
        }

        if (array_key_exists($calendar->id, $this->privileges)) {
            if ($event->author_id === $this->account->id) {
                if ($this->privileges[$calendar->id]->can_delete_self) {
                    return true;
                }
            } else {
                if ($this->privileges[$calendar->id]->can_delete_others) {
                    return true;
                }
            }
        }

        if ($event->author_id === $this->account->id) {
            if ($calendar->can_delete_self) {
                return true;
            }
        } else {
            if ($calendar->can_delete_others) {
                return true;
            }
        }

        return false;
    }

    /**
     * @return bool
     */
    public function canView(CalendarEvent $event)
    {
        $calendar = $event->calendar;

        if ($this->isOwner($calendar)) {
            return true;
        }

        if (array_key_exists($calendar->id, $this->privileges)) {
            if ($event->author_id === $this->account->id) {
                return true;
            } else {
                if ($this->privileges[$calendar->id]->can_view_others) {
                    return true;
                }
            }
        }

        if ($event->author_id === $this->account->id) {
            return true;
        } else {
            if ($calendar->can_view_others) {
                return true;
            }
        }

        return false;
    }
}
