<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Calendar
 */

namespace Application\Calendar\Model;

use ActiveRecord\Base;
use Application\Core\Model\Account;
use Pongho\Utilities\DateTime;

/**
 * CalendarEvent
 *
 * @property int      $id
 * @property int      $calendar_id
 * @property int      $category_id
 * @property int      $author_id
 * @property bool     $is_all_day
 * @property DateTime $start_date_at
 * @property DateTime $end_date_at
 * @property string   $title
 * @property string   $content
 * @property string   $place
 * @property string   $status
 * @property DateTime $created_at
 * @property DateTime $updated_at
 *
 * @method static CalendarEvent[] all(array $options)
 * @method static CalendarEvent find(int $id)
 */
class CalendarEvent extends Base
{
    /**
     * Tabella
     */
    public static $table_name = 'calendars_events';

    /**
     * Relazioni
     */
    public static $belongs_to = array(
        array('calendar', 'model' => 'Application\\Calendar\\Model\\Calendar'),
        array('category', 'model' => 'Application\\Calendar\\Model\\CalendarCategory'),
    );

    protected $users;

    /**
     * @inherits
     */
    public function delete()
    {
        return $this->deleteCalendarEventUsers()
            && parent::delete();
    }

    public function deleteCalendarEventUsers()
    {
        $options = array(
            'conditions' => array('calendar_event_id = :event', 'event' => $this->id),
        );

        foreach (CalendarEventUser::all($options) as $row) {
            if (!$row->delete()) {
                return false;
            }
        }

        return true;
    }

    public function getParticipants()
    {
        if ($this->users === null) {
            $options = array(
                'conditions' => array('calendar_event_id = :event', 'event' => $this->id),
                'order'      => 'created_at ASC',
            );

            $this->users = CalendarEventUser::all($options);
        }

        return $this->users;
    }

    /**
     * @param int    $user_id
     * @param string $status
     * @return bool
     */
    public function addParticipant($user_id, $status = 'active')
    {
        if (Account::find($user_id) === null) {
            return false;
        }

        foreach ($this->getParticipants() as $participant) {
            if ($participant->user_id === intval($user_id)) {
                return true;
            }
        }

        $attributes = array(
            'calendar_event_id' => $this->id,
            'user_id'           => $user_id,
            'status'            => $status,
        );

        $event_user = CalendarEventUser::create($attributes);

        if ($event_user !== null) {
            return true;
        }

        return false;
    }

    public function removeParticipant($user_id)
    {
        if (Account::find($user_id) === null) {
            return false;
        }

        $options = array(
            'conditions' => array(
                'calendar_event_id = :event AND user_id = :user',
                'event' => $this->id,
                'user'  => $user_id
            ),
        );

        if ($participant = CalendarEventUser::first($options)) {
            return $participant->delete();
        }

        return false;
    }

    public function hasParticipants()
    {
        $participants = $this->getParticipants();

        return !empty($participants);
    }

    public function inParticipants($user_id)
    {
        foreach ($this->getParticipants() as $participant) {
            if ($participant->user_id === intval($user_id)) {
                return true;
            }
        }

        return false;
    }
}
