<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Calendar
 */

namespace Application\Calendar\Model;

use ActiveRecord\Base;
use Pongho\Core\Kernel;
use Pongho\EventDispatcher\Event;

/**
 * Calendar
 *
 * @property int    $id
 * @property int    $module_site_id
 * @property int    $owner_id
 * @property string $title
 * @property string $color
 * @property string $status
 * @property int    $position
 *
 * @method static Calendar[] all(array $options)
 * @method static Calendar find(int $id)
 */
class Calendar extends Base
{
    /**
     * Tabella
     */
    public static $table_name = 'calendars';

    /**
     * Impostazioni
     */
    public static $colors = array(
        'default' => array('bg' => '#E3E3E3', 'text' => '#666'),
        'white'   => array('bg' => '#f5f5f5', 'text' => '#666'),
        'silver'  => array('bg' => '#a5a5a5', 'text' => '#fff'),
        'gray'    => array('bg' => '#888888', 'text' => '#fff'),
        'black'   => array('bg' => '#333333', 'text' => '#fff'),
        'red'     => array('bg' => '#C10031', 'text' => '#fff'),
        'maroon'  => array('bg' => '#A6493E', 'text' => '#fff'),
        'yellow'  => array('bg' => '#DF9E43', 'text' => '#fff'),
        'olive'   => array('bg' => '#8AA857', 'text' => '#fff'),
        'lime'    => array('bg' => '#A1DE44', 'text' => '#fff'),
        'green'   => array('bg' => '#4CA64C', 'text' => '#fff'),
        'cyan'    => array('bg' => '#59BFB3', 'text' => '#fff'),
        'teal'    => array('bg' => '#3366CC', 'text' => '#fff'),
        'blue'    => array('bg' => '#33497A', 'text' => '#fff'),
        'navy'    => array('bg' => '#2623A8', 'text' => '#fff'),
        'magenta' => array('bg' => '#A64C79', 'text' => '#fff'),
        'purple'  => array('bg' => '#8342A6', 'text' => '#fff'),
    );

    public static $calendar_states = array(
        'active'   => 'status_active',
        'disabled' => 'status_disabled',
    );

    public static $event_states = array(
        'open'      => 'status_open',
        'closed'    => 'status_closed',
        'confirmed' => 'status_confirmed',
    );

    protected $events;

    /**
     * Relazioni
     */
    public static $belongs_to = array(
        array('owner', 'model' => 'Application\\Core\\Model\\Account'),
    );

    public static function getControllerClassName()
    {
        return 'Application\\Calendar\\Controller\\CalendarController';
    }

    public static function getAdminControllerClassName()
    {
        return 'Application\\Calendar\\Controller\\Admin\\CalendarController';
    }

    /**
     *
     */
    public function getEvents(array $options)
    {
        if ($this->events === null) {
            if (!isset($options['conditions'])) {
                $options['conditions'] = array('calendar_id = :id', 'id' => $this->id);
            }

            $this->events = CalendarEvent::all($options);
        }

        return $this->events;
    }

    /**
     * @inherits
     */
    public function delete($in_recursion = false)
    {
        if ($this->deleteCalendarEvents() && $this->deleteCalendarCategories() && $this->deleteCalendarUsers() && parent::delete()) {
            return true;
        }

        return false;
    }

    public function deleteCalendarEvents()
    {
        $options = array(
            'conditions' => array('calendar_id = :calendar', 'calendar' => $this->id),
        );

        foreach (CalendarEvent::all($options) as $row) {
            if (!$row->delete()) {
                return false;
            }
        }

        return true;
    }

    public function deleteCalendarCategories()
    {
        $options = array(
            'conditions' => array('calendar_id = :calendar', 'calendar' => $this->id),
        );

        foreach (CalendarCategory::all($options) as $row) {
            if (!$row->delete()) {
                return false;
            }
        }

        return true;
    }

    public function deleteCalendarUsers()
    {
        $options = array(
            'conditions' => array('calendar_id = :calendar', 'calendar' => $this->id),
        );

        foreach (CalendarUser::all($options) as $row) {
            if (!$row->delete()) {
                return false;
            }
        }

        return true;
    }

    public static function getCategoriesSelectOptions($calendar_id)
    {
        $options = array(
            'conditions' => array('calendar_id = :calendar', 'calendar' => $calendar_id),
        );

        $categories = array();
        foreach (CalendarCategory::all($options) as $category) {
            $categories[$category->id] = $category->title;
        }

        return $categories;
    }

    public function getColor($type)
    {
        $colors = Kernel::instance()->getContainer()->get('event_dispatcher')
            ->filter(new Event($this, 'calendar.filter_colors'), static::$colors)
            ->getReturnValue();

        return $colors[$this->color][$type];
    }
}
