<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Calendar
 */

namespace Application\Calendar\Handler\Data;

use Application\Calendar\Model\Calendar;
use Application\Calendar\Model\CalendarEvent;
use Application\Core\Model\Role;
use Application\Core\User;
use Pongho\DependencyInjection\Container;
use Pongho\Utilities\DateTime;

class Base
{
    protected $user;

    public function __construct(User $user)
    {
        $this->user = $user;
    }

    public static function getClassPath()
    {
        return 'Application\\Calendar\\Handler\\Data\\Base';
    }

    public function getCalendars()
    {
        $calendars = array();

        foreach (Calendar::all() as $calendar) {
            $id = 'calendar-' . $calendar->id;
            $is_editable = ($calendar->owner_id === $this->user->id) || $this->user->role_id === Role::ADMIN;

            $calendar->is_editable = $is_editable;
            $calendar->events = $this->getEvents(array(
                'conditions' => array(
                    'calendar_id = :calendar',
                    'calendar' => $calendar->id
                )
            ));

            $calendars[$id] = $calendar;
        }

        return $calendars;
    }

    protected static function getCalendarsToLoad()
    {
        $options = array();

        return Calendar::all($options);
    }

    /**
     * Restituisce gli eventi per il calendario, metodo estendibile
     *
     * @throws \InvalidArgumentException
     * @internal param array $options
     *
     * @return array
     */
    public function getEvents(/* args */)
    {
        $options = func_get_arg(0);

        if (!is_array($options)) {
            throw new \InvalidArgumentException('The first argument must be an array');
        }

        $events = array();
        foreach (CalendarEvent::all($options) as $event) {
            $events[] = $this->makeEvent($event, false);
        }

        return $events;
    }

    protected function makeEvent(CalendarEvent $event, $is_editable = true)
    {
        $participants = array();
        foreach ($event->getParticipants() as $participant) {
            $p = array();
            foreach ($participant->attributes as $field => $value) {
                $p[$field] = ($value instanceof DateTime ? $value->format('Y-m-d') : $value);
            }

            // Campi supplementari
            $p['label'] = $participant->user->name(true);

            $participants[] = $p;
        }

        return array(
            'id'           => $event->id,
            'author'       => $event->author_id,
            'editable'     => $is_editable,
            'calendar'     => $event->calendar_id,
            'category'     => $event->category_id,
            'title'        => $event->title,
            'start'        => $event->start_date_at->format('Y-m-d H:i'),
            'end'          => $event->end_date_at->format('Y-m-d H:i'),
            'place'        => $event->place,
            'description'  => $event->content,
            'allDay'       => $event->is_all_day,
            'status'       => $event->status,
            'color'        => Calendar::$colors[$event->calendar->color]['bg'],
            'textColor'    => Calendar::$colors[$event->calendar->color]['text'],
            'participants' => $participants,
        );
    }

    /**
     * Funzione per gestire eventuali custom fields
     */
    public static function parseFields($post)
    {
        return $post;
    }

    /**
     * Carica il template del popup, utilizzato dalla chiamata AJAX in AjaxController
     *
     * @static
     * @access public
     *
     * @param array                                 $data i dati provenienti dal POST, gli eventi sono già gestiti dal parsePost()
     * @param \Pongho\DependencyInjection\Container $container
     */
    public static function getPopupTemplate(array $data, Container $container)
    {
        $path = $container->getParameter('path');

        if ($data['action'] === 'add') {
            $event = new CalendarEvent();
        } else {
            $event = CalendarEvent::find($data['event_id']);
        }

        $calendar_select = array();
        foreach (static::getCalendarsToLoad() as $calendar) {
            $calendar_select[$calendar->id] = $calendar->title;
        }

        $container->get('admin_view_body')
            ->setTemplatePath(__DIR__ . '/../../Resources/views/calendar_popup.php')
            ->assignVars(array(
                'form_action'      => url("{$path}/event/"),
                'autocomplete_url' => url("{$path}/autocomplete/"),
                'categories_url'   => url("{$path}/categories/"),
                'event'            => $event,
                'calendar_select'  => $calendar_select,
            ));
    }
}
