<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Calendar
 */

namespace Application\Calendar\Controller;

use Application\Calendar\Model\Calendar;
use Pongho\Http\Exception\HttpNotFoundException;

class CalendarController extends AjaxController
{
    public function indexAction()
    {
        if ($response = $this->notifyUntil('calendar.index_action')) {
            return $response;
        }

        $site_module = $this->getSiteModule();
        $path = $site_module->path;
        $user = $this->getUser();

        $calendar_options = array(
            'header'               => array(
                'left'  => 'prev,next,today title',
                'right' => 'agendaDay,agendaWeek,month',
            ),
            'firstDay'             => $site_module->getOption('first_day'),
            'minTime'              => $site_module->getOption('min_calendar_day_time'),
            'maxTime'              => $site_module->getOption('max_calendar_day_time'),
            'defaultView'          => $site_module->getOption('default_calendar_admin_view'),
            'popup_selector'       => '#calendar-popup',
            'popup_actions_url'    => url("{$path}event/"),
            'popup_users_url'      => url("{$path}autocomplete/"),
            'popup_categories_url' => url("{$path}categories/"),
        );
        $calendar_options = $this->filter("calendar.index.filter_js_options", $calendar_options);

        $calendar_container = $this->filter("calendar.index.filter_js_container", '#calendar');

        $data_handler_class = $this->filter('calendar.index.filter_data_handler', 'Application\\Calendar\\Handler\\Data\\Base');
        $data_handler = new $data_handler_class($user);
        $handler_class = $this->filter('calendar.index.filter_view_handler', 'Application\\Calendar\\Handler\\View\\Single');
        /** @var $handler \Application\Calendar\Handler\ViewInterface */
        $handler = new $handler_class($this->getView(), $this->getHead(), $site_module, $calendar_container, $calendar_options, $data_handler, $user);
        $calendars = $handler->getView();

        $current_calendar = Calendar::first(array(
            'conditions' => array('owner_id = :user', 'user' => $user->id),
        ));

        $this->getHead()
            ->addCss(pongho_url('/Application/Calendar/Resources/views/css/style.css?v=' . filemtime(__DIR__ . '/../Resources/views/css/style.css')));

        $this->getView()
            ->setTemplate('calendar/calendar.php')
            ->assignVars(array(
                'calendars'         => $calendars,
                'calendar_select'   => $this->getCalendarsSelectOptions(),
                'selected_calendar' => ($current_calendar === null ? 0 : $current_calendar->id),
                'category_select'   => array(),
                'status_select'     => $this->getEventStatesSelectOptions(),
            ));

        $this->notify('calendar.index.after_set_view');
    }

    /**
     *
     */
    private function getEventStatesSelectOptions()
    {
        $states = array();
        $lang = $this->getLocalization();

        foreach ($this->getEventStates() as $key => $status) {
            $states[$key] = array($lang->get($status));
        }

        return $states;
    }

    /**
     *
     */
    public function getCalendarsSelectOptions()
    {
        $calendars = array();
        $colors = $this->getColors();

        $options = $this->filter("calendar.filter_calendars_select_options", array('order' => 'title ASC'));

        foreach (Calendar::all($options) as $calendar) {
            $calendars[$calendar->id] = array(
                'title'           => $calendar->title,
                'data-text-color' => $colors[$calendar->color]['text'],
                'data-bg-color'   => $colors[$calendar->color]['bg'],
            );
        }

        return $calendars;
    }

    /**
     *
     */
    public function getColors()
    {
        $cal_colors = Calendar::$colors;
        $cal_colors = $this->filter("calendar.filter_colors", $cal_colors);

        return $cal_colors;
    }

    /**
     *
     */
    public function getStates()
    {
        $states = Calendar::$calendar_states;
        $states = $this->filter("calendar.filter_states", $states);

        return $states;
    }

    /**
     *
     */
    public function __call($method, array $arguments)
    {
        if ($response = $this->notifyUntil('calendar.method_not_found',
            array('method' => $method, 'arguments' => $arguments))
        ) {
            return $response;
        }

        throw new HttpNotFoundException();
    }

    /**
     *
     */
    public function getEventStates()
    {
        $states = Calendar::$event_states;
        $states = $this->filter("calendar.filter_event_states", $states);

        return $states;
    }
}
