<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Calendar
 */

namespace Application\Calendar\Controller;

use Application\Calendar\Model\Calendar;
use Application\Calendar\Model\CalendarCategory;
use Application\Calendar\Model\CalendarEvent;
use Application\Calendar\Model\CalendarEventUser;
use Application\Core\Controller as Base;
use Application\Core\Model\Account;
use ActiveRecord\Column;
use Pongho\Http\Exception\HttpNotFoundException;
use Pongho\Http\JsonResponse;
use Pongho\Http\Request;
use Pongho\Utilities\DateTime;

class AjaxController extends Base
{
    /**
     * Azioni
     */
    public function eventAction()
    {
        $request = $this->getRequest();
        $action = $request->post->get('action');

        $attributes = $request->post->all();
        unset($attributes['action']);
        unset($attributes['participants']);

        if ($action) {
            // Se non ho un attributo id, forzo l'aggiunta perché potrei voler modificare un evento "virtuale" che
            // deve ancora essere salvato in database
            if (!$request->post->has('id')) {
                $action = 'add';
            }

            // Potrei aver definito un handler per gestire questo evento diversamente dagli altri (solitamente per
            // dare un errore in caso certe condizioni non vengano soddisfatte)
            if ($request->post->has('handler')) {
                /** @var  $handler */
                $handler = $request->post->get('handler');
                $attributes = $handler::parseFields($attributes);
            }
        } else {
            return $this->getHelper()->displayError('The "action" parameter is not defined!');
        }

        // In mancanza di un autore, assegno l'utente corrente
        if ($request->post->has('author_id')) {
            $attributes['author_id'] = $request->post->get('author_id');
        } else {
            $attributes['author_id'] = $this->getHelper()->getUserId();
        }

        $attributes = $this->parseFields($attributes, CalendarEvent::columns());

        switch ($action) {
            case 'add':
                // l'id non mi serve in aggiunta
                unset($attributes['id']);

                /** @var CalendarEvent $event */
                $event = CalendarEvent::create($attributes);
                $attributes = $event->attributes();

                $attributes['participants'] = array_values($this->updateParticipants($event, $request));

                break;

            case 'edit':
                /** @var CalendarEvent $event */
                $event = CalendarEvent::find($attributes['id']);

                if ($event === null) {
                    throw new HttpNotFoundException();
                }

                $old_attributes = $event->attributes();
                $event->updateAttributes($attributes);

                if ($event->save()) {
                    $attributes = $event->attributes();
                } else {
                    $attributes = $old_attributes;
                }

                if ($request->post->has('participants')) {
                    $attributes['participants'] = array_values(
                        $this->updateParticipants($event, $request)
                    );
                } else {
                    $attributes['participants'] = [];
                }

                $this->getHelper()->notify($this, 'calendar.after_edit_event', [
                    'event'        => $event,
                    'participants' => $attributes['participants'],
                ]);

                break;

            case 'move':
                /** @var CalendarEvent $event */
                $event = CalendarEvent::find($attributes['id']);

                $event->calendar_id = $attributes['calendar_id'];
                $event->start_date_at = $attributes['start_date_at'];
                $event->end_date_at = $attributes['end_date_at'];

                $event->save();

                break;

            case 'delete':
                /** @var CalendarEvent $event */
                $event = CalendarEvent::find($attributes['id']);

                if ($event === null || $event->delete()) {
                    return new JsonResponse(array('result' => 'ok'));
                }

                return new JsonResponse(array('error' => true, 'message' => 'fail'));

            default:
                throw new HttpNotFoundException('unknown action');
                break;
        }

        // Correggo le date da PonghoDateTime a string e aggiungo i colori
        $attributes['text_color'] = Calendar::$colors[$event->calendar->color]['text'];
        $attributes['bg_color'] = Calendar::$colors[$event->calendar->color]['bg'];
        $attributes['start_date_at'] = $attributes['start_date_at']->format('Y-m-d H:i');
        $attributes['end_date_at'] = $attributes['end_date_at']->format('Y-m-d H:i');

        return new JsonResponse($attributes);
    }

    /**
     *
     */
    public function categoriesAction()
    {
        $id = isset($_POST['calendar_id']) ? $_POST['calendar_id'] : 0;

        $options = array(
            'conditions' => array('calendar_id = :calendar_id', 'calendar_id' => $id),
            'order'      => 'title ASC',
        );

        $categories = array();
        $code = '<option value="">Scegli una categoria</option>';
        foreach (CalendarCategory::all($options) as $category) {
            $categories[$category->id] = $category->title;
            $code .= '<option value="' . $category->id . '">' . $category->title . '</option>';
        }

        return new JsonResponse($code);
    }

    /**
     *
     */
    public function autocompleteAction()
    {
        $term = isset($_GET['term']) ? $_GET['term'] : '';
        $not_in = isset($_GET['not_in']) && is_array($_GET['not_in']) && $_GET['not_in'] ? $_GET['not_in'] : '';
        $filter_roles = isset($_GET['filter_roles']) && is_array($_GET['filter_roles']) && $_GET['filter_roles'] ? $_GET['filter_roles'] : '';

        if ($not_in) {
            $not_in = ' AND id NOT IN(' . implode(',', $not_in) . ')';
        }

        if ($filter_roles) {
            $filter_roles = ' AND role_id IN (' . implode(',', $filter_roles) . ')';
        }

        $options = array(
            'conditions' => array(
                '(pongho_like(name, :term) OR pongho_like(surname, :term) OR pongho_like(fullname, :term))' . $not_in . $filter_roles,
                'term' => "%{$term}%"
            ),
            'order'      => 'surname ASC, name ASC',
        );

        $options = $this->filter('calendar.filter_autocomplete_options', $options);

        $users = array();
        foreach (Account::all($options) as $user) {
            $users[] = array(
                'id'    => $user->id,
                'label' => $user->name(true),
            );
        }

        return new JsonResponse($users);
    }

    /**
     *
     */
    public function geteventsAction()
    {
        $start_date = date('Y-m-d H:i:s', $_POST['start']);
        $end_date = date('Y-m-d H:i:s', $_POST['end']);
        $calendars = isset($_POST['calendars']) ? $_POST['calendars'] : array();

        $handler_class = isset($_POST['handler']) ? $_POST['handler'] : '\\Application\\Calendar\\Handler\\Data\\Base';
        $filters = isset($_POST['filters']) ? $_POST['filters'] : array();

        $handler = new $handler_class($this->getUser());
        if ($filters) {
            $handler->setFilters($filters);
        }

        // L'evento potrebbe iniziare prima e finire dopo il giorno corrente
        // prima: 'start_date_at >= :start AND end_date_at < :end'
        $options = array(
            'conditions' => array(
                'calendar_id IN (' . implode(', ',
                    array_map('intval', $calendars)) . ') AND start_date_at < :end AND end_date_at > :start',
                'start' => $start_date,
                'end'   => $end_date
            ),
        );

        return new JsonResponse($handler->getEvents($options, $_POST['start'], $_POST['end'], $calendars));
    }

    /**
     *
     */
    public function getpopupAction()
    {
        if (!isset($_POST['handler'])) {
            throw new HttpNotFoundException('Handler del calendario non impostato');
        }

        $post = $_POST;

        $handler = $post['handler'];

        //$post['event'] = $this->parsePost($post['event']);
        try {
            $post = $handler::parseFields($post);
        } catch (\Exception $e) {
            return $this->displayError($e->getMessage);
        }

        $template = $handler::getPopupTemplate($post, $this->getContainer());
    }

    /**
     * @param CalendarEvent $event
     * @param Request       $request
     * @return CalendarEventUser[]|array
     */
    protected function updateParticipants(CalendarEvent $event, Request $request)
    {
        /** @var CalendarEventUser[] $eventParticipants */
        $eventParticipants = [];
        foreach ($event->getParticipants() as $participant) {
            $eventParticipants[$participant->user_id] = $participant;
        }

        if (!$request->post->has('participants')) {
            return array_map(function (CalendarEventUser $participant) {
                return $participant->attributes();
            }, $eventParticipants);
        }

        $participants = [];
        foreach ($request->post->get('participants', []) as $participant) {
            if (isset($participant['user_id']) && is_numeric($participant['user_id'])) {
                $participant['user_id'] = (int) $participant['user_id'];
                $participant['status'] = 'active'; // todo: non mi è chiaro a cosa serva, ma per sicurezza lo forzo

                $participants[$participant['user_id']] = $participant;
            }
        }

        // Cancellazione e aggiornamento
        foreach ($eventParticipants as $userId => $eventParticipant) {
            if (!isset($participants[$userId])) {
                $eventParticipant->delete();

                continue;
            }

            $attributes = $eventParticipant->attributes();
            foreach ($participants[$userId] as $attributeName => $attributeValue) {
                $attributes[$attributeName] = $attributeValue === 'null' ? null : $attributeValue;
            }

            $attributes = $this->getHelper()->filter($this, 'calendar.edit_participant.filter_attributes', $attributes);

            $eventParticipant->updateAttributes(
                $this->parseFields($attributes, CalendarEventUser::columns())
            );

            if ($eventParticipant->save()) {
                $participants[$userId] = $eventParticipant->attributes();
            }
        }

        // Aggiunta
        foreach ($participants as $userId => $participant) {
            if (isset($eventParticipants[$userId])) {
                continue;
            }

            $attributes = [];
            foreach ($participant as $attributeName => $attributeValue) {
                $attributes[$attributeName] = $attributeValue === 'null' ? null : $attributeValue;
            }

            if (!isset($attributes['calendar_event_id'])) {
                $attributes['calendar_event_id'] = $event->id;
            }

            $attributes = $this->parseFields($attributes, CalendarEventUser::columns());
            $attributes = $this->getHelper()->filter($this, 'calendar.add_participant.filter_attributes', $attributes);

            $new = new CalendarEventUser(
                $attributes
            );

            if ($new->save()) {
                $participants[$userId] = $new->attributes();
            }
        }

        return $participants;
    }

    /**
     * @param array    $raw
     * @param Column[] $columns
     * @return array
     */
    protected function parseFields($raw, $columns)
    {
        $attributes = [];

        // Ciclo le colonne del database
        foreach ($columns as $name => $column) {
            switch ($column->type) {
                case Column::STRING:
                    if (array_key_exists($name, $raw)) {
                        $attributes[$name] = $raw[$name];
                    }
                    break;

                case Column::DECIMAL:
                case Column::INTEGER:
                    if (array_key_exists($name, $raw)) {
                        $attributes[$name] = intval($raw[$name]);

                        if (!$attributes[$name]) {
                            $attributes[$name] = null;
                        }
                    }
                    break;

                case Column::BOOLEAN:
                    if (array_key_exists($name, $raw)) {
                        //attenzione con i POST in Ajax, il valore è 'true' o 'false' in stringa!
                        $attributes[$name] = $raw[$name] && $raw[$name] !== 'false' ? true : false;
                    } else {
                        $attributes[$name] = false;
                    }
                    break;

                case Column::DATETIME:
                case Column::DATE:
                    if (array_key_exists($name, $raw)) {
                        if ($raw[$name] instanceof DateTime) {
                            $attributes[$name] = $raw[$name];
                        } elseif (is_array($raw[$name]) && isset($raw[$name]['date'])) {
                            // FIXME: C'è un javascript da qualche parte che converte le date in array [date, timezone_type, timezone]
                            $attributes[$name] = new DateTime($raw[$name]['date']);
                        } else {
                            $attributes[$name] = new DateTime($raw[$name]);
                        }
                    }
                    break;
            }
        }

        return $attributes;
    }
}
