<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Calendar
 */

namespace Application\Calendar\Controller\Admin;

use Application\Calendar\Model\Calendar;
use Application\Calendar\Model\CalendarCategory;
use Application\Calendar\Model\CalendarUser;
use Application\Core\Model\Account;
use Pongho\Http\RedirectResponse;

class CalendarController extends AjaxController
{
    /**
     * @implements
     */
    public function getModelClass()
    {
        return 'Application\\Calendar\\Model\\Calendar';
    }

    public function viewAction()
    {
        $path = $this->getParameter('path');
        $site_module = $this->getHelper()->getModuleSite();
        $lang = $this->getHelper()->getLocalization();
        $user = $this->getHelper()->getUser();

        $data_handler_class = $this->filter('admin.calendar.filter_calendar_data_handler', 'Application\\Calendar\\Handler\\Data\\Base');
        $data_handler = new $data_handler_class($user);

        if (count($data_handler->getCalendars())) {
            $calendar_options = $this->filter("admin.calendar.filter_calendar_options", array(
                'aspect_ratio_handling' => 'fixed',
                'axisFormat'            => 'H:mm',
                'contentHeight'         => 800,
                'firstDay'              => $site_module->getOption('first_day'),
                'header'                => array(
                    'left'  => 'prev,next,today title',
                    'right' => 'agendaDay,agendaWeek,month',
                ),
                'minTime'               => $site_module->getOption('min_calendar_day_time'),
                'maxTime'               => $site_module->getOption('max_calendar_day_time'),
                'defaultView'           => $site_module->getOption('default_calendar_admin_view'),
                'selectable'            => true,
                'selectHelper'          => true,
                'popup_selector'        => pongho_url("/{$path}/getpopup/"),
                'popup_actions_url'     => $this->url("/{$path}/event/"),
                'popup_users_url'       => $this->url("/{$path}/autocomplete/"),
                'popup_categories_url'  => $this->url("/{$path}/categories/"),
                'event_save_url'        => $this->url("/{$path}/event/"),
            ));

            $calendar_container = $this->filter("admin.calendar.filter_calendar_container", '#calendar');

            $handler_class = $this->filter('admin.calendar.filter_calendar_view_handler', 'Application\\Calendar\\Handler\\View\\Single');
            $handler = new $handler_class($this->getHelper()->getBodyView(), $this->getHelper()->getThemeHeaderHelper(), $site_module, $calendar_container, $calendar_options, $data_handler, $user);
            $calendars = $handler->getView();

            $add_first_calendar = '';
        } else {
            $add_first_calendar = sprintf(
                '<div class="empty"><a href="%1$s" title="%2$s">%2$s</a></div>',
                $this->url("/{$path}/add/"),
                $lang->get('add_first_calendar')
            );
        }

        $this->getHelper()
            ->addCss(pongho_url('/Application/Calendar/Resources/views/css/style.css?v=' . filemtime(__DIR__ . '/../../Resources/views/css/style.css')))
            ->addJavascript(pongho_url('/vendor/jQuery/UI/widgets/timepicker/jquery-ui-timepicker-addon.js'))
            ->addJavaScriptInline(<<<JS
$.timepicker.setDefaults($.timepicker.regional['it']);
datetimepickerOptions = {
	hourMin: 8,
	hourMax: 20,
	stepMinute: 30,
	dateFormat: 'yy-mm-dd',

	// dispositivi touch
	addSliderAccess: true,
	sliderAccessArgs: { touchonly: false }
};
JS
            );

        $this->getHelper()->getBodyView()
            ->setTemplatePath(__DIR__ . '/../../Resources/views/calendar.php')
            ->assignVars(array(
                'add_first_calendar' => $add_first_calendar,
                'calendar_select'    => $this->getCalendarsSelectOptions(),
                'category_select'    => array('0' => $lang->select_calendar_first),
                'status_select'      => $this->getEventStatesSelectOptions(),
            ));

        $this->notify('admin.calendar.after_set_view');
    }

    /**
     *
     */
    public function moveupAction()
    {
        $calendar = Calendar::find($this->getParameter('id', null, true));

        return $this->move($calendar, -15);
    }

    /**
     *
     */
    public function movedownAction()
    {
        $calendar = Calendar::find($this->getParameter('id', null, true));

        return $this->move($calendar, 15);
    }

    /**
     *
     */
    protected function move(Calendar $calendar, $direction)
    {
        Calendar::transaction(function () use ($calendar, $direction) {
            $calendar->position += $direction;
            $calendar->save();

            $position = 10;
            foreach (Calendar::all(array('order' => 'position')) as $cal) {
                $cal->position = $position;
                $cal->save();

                $position += 10;
            }
        });

        return new RedirectResponse($this->url('/' . $this->getParameter('path') . '/'));
    }

    /**
     * @inherits
     */
    public function displayArchive($page = 1)
    {
        $path = $this->getParameter('path');
        $pagination_query = array();
        $options = array();
        $colors = $this->getColors();

        // Query di ricerca
        if (isset($_GET['q']) && !empty($_GET['q'])) {
            $query = htmlspecialchars(trim($_GET['q']));
            $pagination_query['q'] = $query;

            $options = array(
                'select'     => '`from`.*',
                'conditions' => array('pongho_like(`from`.title, :query)', 'query' => "%$query%"),
            );
        } else {
            $query = '';
        }

        //$options['order'] = 'created_at DESC';

        $options['include'] = array('owner');
        $options['order'] = 'position';
        $options = $this->filter("admin.calendar.filter_archive_options", $options, array('page' => $page));

        $rows = Calendar::page($page, $this->records_per_page, $options);
        $total = Calendar::count($options);

        $i = 1;
        foreach ($rows as $row) {
            $actions = array(
                array('edit', 'href' => $this->url("/{$path}/edit/{$row->id}/")),
                array('delete', 'href' => $this->url("/{$path}/delete/{$row->id}/"), 'class' => 'delete'),
            );

            $row = $this->filter("admin.calendar.filter_row", $row);
            $actions = $this->filter("admin.calendar.filter_row_actions", $actions);

            $row->actions = $this->parseActions($actions);
            $row->color_code = $colors[$row->color]['bg'];

            $order_actions = array();

            if ($i > 1) {
                $order_actions[] = array('up', 'href' => $this->url("/{$path}/moveup/{$row->id}/"));
            }

            if ($i < $total) {
                $order_actions[] = array('down', 'href' => $this->url("/{$path}/movedown/{$row->id}/"));
            }

            $row->order_actions = $this->parseActions($order_actions);

            $i++;
        }

        $this->getHelper()
            ->addCss(pongho_url('/Application/Admin/Resources/views/css/style-legacy.css?v=' . filemtime(__DIR__ . '/../../../Admin/Resources/views/css/style-legacy.css')));

        // Vista
        $lang = $this->getLang();
        $this->getHelper()->getBodyView()
            ->setTemplatePath(__DIR__ . '/../../Resources/views/calendars.php')
            ->assignVars(array(
                'add_url'    => $this->url("/{$path}/add/"),
                'search_url' => $this->url("/{$path}/"),
                'rows'       => $rows,
                'query'      => $query,
            ));

        // Paginazione
        $pagination_query['site'] = $this->getSiteId();
        $this->displayPagination($page, $total, $this->records_per_page, $pagination_query);
    }

    /**
     * @implements
     */
    public function getTitle($model)
    {
        $lang = $this->getHelper()->getLocalization();

        return $this->getAction() === 'add' ? $lang->get('add_calendar') : sprintf($lang->get('edit_calendar'), $model->title);
    }

    /**
     * @implements
     */
    public function getFields($node)
    {
        return $this->filter("admin.node_calendar.filter_fields", array(
            array('title'),
            array(
                'owner_id',
                'class' => 'Application\\Admin\\Form\\SelectField',
                'items' => $this->getOwnersSelectOptions()
            ),
            array(
                'color',
                'class' => 'Application\\Admin\\Form\\SelectField',
                'items' => $this->getColorsSelectOptions()
            ),
            array(
                'status',
                'class' => 'Application\\Admin\\Form\\SelectField',
                'items' => $this->getStatesSelectOptions()
            ),
        ));
    }

    /**
     * @callback
     */
    public function beforeSetTemplateCallback(Calendar $calendar)
    {
        $path = $this->getParameter('path');

        $calendar_categories = CalendarCategory::all(array(
            'conditions' => array('calendar_id = :calendar', 'calendar' => $calendar->id),
        ));

        $calendar_users = CalendarUser::all(array(
            'conditions' => array('calendar_id = :calendar', 'calendar' => $calendar->id),
        ));

        $this->getHelper()->getBodyView()
            ->assignVars(array(
                'calendar_categories' => $calendar_categories,
                'calendar_users'      => $calendar_users,
                'autocomplete'        => $this->url("/{$path}/autocomplete/"),
            ));

        $this->getHelper()
            ->addJavaScript(pongho_url('/Application/Calendar/Resources/views/js/jquery.calendar-edit.js'));
    }

    /**
     * @callback
     */
    protected function beforeAddCallback(Calendar $node)
    {
        $node->module_site_id = $this->getHelper()->getModuleSite()->id;
    }

    /**
     * @callback
     */
    protected function afterSaveCallback(Calendar $node)
    {
        $users = array_key_exists('users', $_POST) ? $_POST['users'] : array();
        $categories = array_key_exists('categories', $_POST) ? $_POST['categories'] : array();

        foreach ($users as $id => $user) {
            $attributes = array(
                'calendar_id'       => $node->id,
                'user_id'           => array_key_exists('user_id', $user) ? $user['user_id'] : null,
                'can_add'           => array_key_exists('can_add', $user),
                'can_edit_self'     => array_key_exists('can_edit_self', $user),
                'can_edit_others'   => array_key_exists('can_edit_others', $user),
                'can_delete_self'   => array_key_exists('can_delete_self', $user),
                'can_delete_others' => array_key_exists('can_delete_others', $user),
                'can_view_others'   => array_key_exists('can_view_others', $user),
            );

            if (!is_numeric($id)) {
                if ($attributes['user_id']) {
                    CalendarUser::create($attributes);
                }
            } else {
                $model = CalendarUser::findByCalendarAndUser($node->id, $user['user_id']);

                if ($model === null) {
                    throw new \RuntimeException;
                }

                if (array_key_exists('deletable', $user)) {
                    $model->delete();
                } else {
                    $model->updateAttributes($attributes);
                    $model->save();
                }
            }
        }

        foreach ($categories as $id => $category) {
            $attributes = array(
                'calendar_id' => $node->id,
                'title'       => $category['title'],
            );

            if (!is_numeric($id)) {
                if (!empty($attributes['title'])) {
                    CalendarCategory::create($attributes);
                }
            } else {
                $model = CalendarCategory::find($id);

                if (!$model) {
                    continue;
                }

                if (isset($category['delete']) || isset($category['deleteCategoryAndEvents'])) {
                    if (isset($category['deleteCategoryAndEvents'])) {
                        $model->deleteEvents();
                    }

                    $model->delete();

                    continue;
                }

                $model->updateAttributes($attributes);
                $model->save();
            }
        }
    }

    /**
     * Restituisce il path del modulo corrente.
     */
    public function getModulePath()
    {
        return $this->getParameter('module', null, true)->path;
    }

    /**
     * @inherits
     */
    protected function getSiteModuleClassName()
    {
        return 'Application\\Calendar\\Model\\ModuleSite';
    }

    /**
     *
     */
    private function getOwnersSelectOptions()
    {
        $owners = array();
        $options = array(
            'conditions' => array('is_active = :active AND role_id > :role', 'active' => true, 'role' => 1),
        );

        $options = $this->filter("admin.calendar.filter_owners_options", $options);

        foreach (Account::all($options) as $user) {
            $owners[$user->id] = $user->name(true);
        }

        return $owners;
    }

    /**
     *
     */
    private function getColorsSelectOptions()
    {
        $colors = array();
        $lang = $this->getHelper()->getLocalization();

        foreach ($this->getColors() as $key => $color) {
            $colors[$key] = array(
                $lang->get("color_{$key}"),
                'style' => "background-color: {$color['bg']}; color: {$color['text']};"
            );
        }

        return $colors;
    }

    /**
     *
     */
    private function getStatesSelectOptions()
    {
        $states = array();
        $lang = $this->getHelper()->getLocalization();

        foreach ($this->getStates() as $key => $status) {
            $states[$key] = array($lang->get($status));
        }

        return $states;
    }

    /**
     *
     */
    private function getEventStatesSelectOptions()
    {
        $states = array();
        $lang = $this->getHelper()->getLocalization();

        foreach ($this->getEventStates() as $key => $status) {
            $states[$key] = array($lang->get($status));
        }

        return $states;
    }

    /**
     *
     */
    public function getCalendarsSelectOptions()
    {
        $calendars = array();
        $colors = $this->getColors();

        foreach (Calendar::all(array('options' => 'title ASC')) as $calendar) {
            $calendars[$calendar->id] = array(
                'title'           => $calendar->title,
                'data-text-color' => $colors[$calendar->color]['text'],
                'data-bg-color'   => $colors[$calendar->color]['bg'],
            );
        }

        return $calendars;
    }

    /**
     *
     */
    public function getColors()
    {
        $cal_colors = Calendar::$colors;
        $cal_colors = $this->filter("admin.calendar.filter_calendar_colors", $cal_colors);

        return $cal_colors;
    }

    /**
     *
     */
    public function getStates()
    {
        $states = Calendar::$calendar_states;
        $states = $this->filter("admin.calendar.filter_calendar_states", $states);

        return $states;
    }

    /**
     *
     */
    public function getEventStates()
    {
        $states = Calendar::$event_states;
        $states = $this->filter("admin.calendar.filter_calendar_event_states", $states);

        return $states;
    }
}
