<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Autoscuola
 */

namespace Application\Autoscuola\Model;

use ActiveRecord\Base;
use Application\Core\Model\Account;
use Pongho\Utilities\DateTime;

/**
 * Student
 *
 * @property int                                    $id
 * @property int                                    $user_id
 * @property \Pongho\Utilities\DateTime             $subscription_date
 * @property int                                    $location_id
 * @property string                                 $typology
 * @property string                                 $license_type
 * @property \Pongho\Utilities\DateTime             $license_date
 * @property \Pongho\Utilities\DateTime             $temp_license_date
 * @property string                                 $status
 * @property string                                 $request_period_1
 * @property int                                    $request_confirm_1
 * @property \Pongho\Utilities\DateTime             $request_date_1
 * @property string                                 $request_result_1
 * @property string                                 $request_period_2
 * @property int                                    $request_confirm_2
 * @property \Pongho\Utilities\DateTime             $request_date_2
 * @property string                                 $request_result_2
 * @property \Pongho\Utilities\DateTime             $created_at
 * @property \Pongho\Utilities\DateTime             $updated_at
 * @property \Application\Core\Model\Account        $user
 * @property \Application\Autoscuola\Model\Location $location
 */
class Student extends Base
{
    /**
     * Nome della tabella.
     *
     * @var string
     */
    public static $table_name = 'autoscuola_students';

    /**
     * Relazione `belongs_to`.
     *
     * @var array
     */
    public static $belongs_to = [
        ['user', 'model' => \Application\Core\Model\Account::class],
        ['location', 'model' => \Application\Autoscuola\Model\Location::class],
    ];

    /**
     * Validazioni.
     *
     * @var array
     */
    public static $validates_presence_of = [
        ['location_id', 'message' => 'Il campo sede è obbligatorio.', 'on' => 'update'],
    ];

    /**
     * @var StudentComment[]
     */
    protected $comments;

    /**
     * @param Account $account
     * @return Student
     */
    public static function findByAccount(Account $account)
    {
        switch ($account->role_id) {
            case ModuleSite::ROLE_STUDENT_P:
                $typology = 'pratica';
                break;

            case ModuleSite::ROLE_STUDENT_T:
                $typology = 'teoria';
                break;

            default:
                throw new \RuntimeException('L’utente ha un ruolo non valido');
        }

        /** @var Student $student */
        $student = Student::first([
            'conditions' => [
                'user_id = :user AND typology = :typology AND status = :status',
                'user'     => $account->id,
                'status'   => 'active',
                'typology' => $typology,
            ],
        ]);

        return $student;
    }

    /**
     * @return StudentComment[]
     */
    public function getComments()
    {
        if ($this->comments === null) {
            $this->comments = StudentComment::all([
                'conditions' => ['student_id = :student', 'student' => $this->id],
                'order'      => 'created_at ASC',
            ]);
        }

        return $this->comments;
    }

    /**
     * @return float|int
     */
    public function getElapsedTime()
    {
        $now = new DateTime();

        $date = $this->typology === 'pratica' ? $this->temp_license_date : $this->subscription_date;

        if (!$date instanceof DateTime) {
            return 0;
        }

        $diff = $now->diff($date)->format('%a');

        $due_date = clone ($date);

        $due_date->add(new \DateInterval('P6M'));

        $diff_due_date = $due_date->diff($date)->format('%a');

        // Limito la proporzione in modo da non andare oltre al 100% per le date passate
        return ceil($diff_due_date > 0 ? 100 * $diff / $diff_due_date : 100);
    }

    /**
     * Restituisce il tentativo corrente (su due possibili prima di fallire il corso)
     *
     * @param string $period
     * @return array un array contenente i dettagli del tentativo corrente
     */
    public function getCurrentTry($period = null)
    {
        $try = [];

        if (($period === null && $this->request_period_1 !== '' && $this->request_result_1 === '' && $this->request_period_2 === '') || ($this->request_period_1 === $period)) {
            $try = [
                'try'     => 1,
                'period'  => $this->request_period_1,
                'confirm' => $this->request_confirm_1,
                'date'    => $this->request_date_1,
                'result'  => $this->request_result_1,
                'booked'  => $this->request_date_1 !== null,
            ];
        } elseif (($period === null && $this->request_period_2 !== '' && $this->request_result_2 === '') || ($this->request_period_2 === $period)) {
            $try = [
                'try'     => 2,
                'period'  => $this->request_period_2,
                'confirm' => $this->request_confirm_2,
                'date'    => $this->request_date_2,
                'result'  => $this->request_result_2,
                'booked'  => $this->request_date_2 !== null,
            ];
        }

        return $try;
    }

    /**
     * Ricava il primo tentativo libero
     *
     * @param int $try controlla se il tentativo è libero, valori ammessi [null, 1, 2]
     * @return integer il numero del tentativo libero, false se nessun tentativo è libero o ci sono tentativi in corso
     */
    public function getFreeTry($try = null)
    {
        switch ($try) {
            case null:
                if ($this->request_period_1 === '') {
                    return 1;
                }

                if (($this->request_result_1 !== '' || $this->request_result_1 === 'idoneo') && $this->request_period_2 === '') {
                    return 2;
                }

                return false;

            case 1:
                if ($this->request_period_1 === '') {
                    return 1;
                }

                return false;

            case 2:
                if (($this->request_result_1 !== '' || $this->request_result_1 === 'idoneo') && $this->request_period_2 === '') {
                    return 2;
                }

                return false;

            default:
                throw new \RuntimeException('Numero del tentativo non valido');
        }
    }

    /**
     * Restituisce un valore booleano che indica se la richiesta dello studente è stata accettata dall'insegnante
     *
     * @return bool
     */
    public function isConfirmed()
    {
        $try = $this->getCurrentTry();

        return (bool) $try['confirm'];
    }

    /**
     * Restituisce un valore booleano che indica se lo studente è correntemente iscritto ad una sessione di esame
     *
     * @return bool
     */
    public function isBooked()
    {
        $try = $this->getCurrentTry();

        if (!empty($try) && $try['date'] instanceof DateTime) {
            if ($try['date']->isNull()) {
                return false;
            }

            // Se ho la data ma non ho l'esito, sono ancora nella sessione
            return ($try['result'] === '');
        }

        return false;
    }

    /**
     * Imposta una prenotazione ad un esame
     *
     * @param DateTime $date
     * @return bool valore che indica se la prenotazione è avvenuta con successo
     */
    public function setBooking(DateTime $date)
    {
        if (!$this->isBooked()) {
            $try = $this->getCurrentTry();

            $request_field = "request_date_{$try['try']}";
            $this->$request_field = $date;

            return $this->save();
        }

        return false;
    }

    /**
     * Imposta una prenotazione ad un esame
     *
     * @return boolean valore che indica se la prenotazione è avvenuta con successo
     */
    public function removeBooking()
    {
        if ($this->isBooked()) {
            $try = $this->getCurrentTry();

            $request_field = "request_date_{$try['try']}";
            $this->$request_field = null;

            return $this->save();
        }

        return false;
    }

    /**
     * Richiede una prenotazione ad un esame
     *
     * @param string $period
     * @return boolean valore che indica se la richiesta è avvenuta con successo
     */
    public function setNewTry($period)
    {
        if (!$this->isBooked()) {
            $try = $this->getFreeTry();

            if (!$try) {
                throw new \RuntimeException('C\'è già una prenotazione o sono stati esauriti i tentativi disponibili');
            }

            $request_field = "request_period_{$try}";
            $this->$request_field = $period;

            return $this->save();
        }

        return false;
    }

    /**
     * Cancella una richiesta di prenotazione ad un esame
     *
     * @return bool Valore che indica se la cancellazione è avvenuta con successo
     */
    public function cancelTry()
    {
        if (!$this->isBooked()) {
            $try = $this->getCurrentTry();

            if ($try && !$try['confirm']) {
                $request_field = "request_period_{$try['try']}";
                $this->$request_field = '';

                return $this->save();
            }
        }

        return false;
    }

    /**
     * Accetta una richiesta di prenotazione.
     *
     * @param int $user_id
     * @return bool
     */
    public function acceptConfirm($user_id)
    {
        $try = $this->getCurrentTry();

        if ($try && !$try['confirm'] && $try['date'] === null) {
            $request_field = "request_confirm_{$try['try']}";
            $this->$request_field = $user_id;

            return $this->save();
        }

        return false;
    }

    /**
     * Rifiuta una richiesta di prenotazione precedentemente accettata.
     *
     * @return bool
     */
    public function cancelConfirm()
    {
        $try = $this->getCurrentTry();
        if ($try && $try['confirm'] && $try['date'] === null) {
            $request_field = "request_confirm_{$try['try']}";
            $this->$request_field = null;

            return $this->save();
        }

        return false;
    }

    /**
     * Restituisce l'esito dell'ultimo esame (primo o secondo tentativo), o niente se non ha conseguito esami
     *
     * @return string
     */
    public function getLastResult()
    {
        $try = $this->getCurrentTry();

        if (!empty($try) && $try['date'] instanceof DateTime) {
            if ($try['date']->isNull()) {
                return '';
            }

            // Se ho la data ma non ho l'esito, sono ancora nella sessione
            return $try['result'];
        }

        return '';
    }

    /**
     * Restituisce la data dell'ultimo esame prenotato
     *
     * @return DateTime
     */
    public static function getLastBookingDate()
    {
        // Ricavo la data dell'ultimo esame fissato
        $student_table = self::tableName();
        $sql = <<<SQL
SELECT t.request_date
FROM (
	SELECT request_date_1 AS request_date FROM {$student_table} WHERE YEAR(request_date_1) > 0 AND YEAR(request_date_2) = 0 AND status = 'active'
	UNION
	SELECT request_date_2 AS request_date FROM {$student_table} WHERE YEAR(request_date_1) > 0 AND YEAR(request_date_2) > 0 AND status = 'active'
) AS t
ORDER BY t.request_date DESC
LIMIT 0,1
SQL;
        $last_booking = self::findBySql($sql);
        if (count($last_booking)) {
            $last_booking = reset($last_booking);
            $last_booking_date = new DateTime($last_booking->request_date);
        } else {
            $last_booking_date = new DateTime();
        }

        return $last_booking_date;
    }

    /**
     * Trasforma il periodo dal formato 'id' ad una stringa formattata e leggibile dall'utente
     *
     * @param string $period
     * @return string
     */
    public static function getReadablePeriod($period)
    {
        if (!$period) {
            return '';
        }

        $year = intval(substr($period, 0, 4));
        $week = intval(substr($period, 4, 2));

        $monday = DateTime::getDateFromWeekNumber($year, $week);
        $period = $monday->getWeekInMonth() . '° sett. di ' . $monday->localeFormat('%B');

        return $period;
    }

    /**
     * @param string $try
     * @return string
     */
    public function requestPeriod($try)
    {
        $request_field = "request_period_{$try}";

        return static::getReadablePeriod($this->$request_field);
    }

    /**
     * @param string $try
     * @return Account
     */
    public function requestConfirm($try)
    {
        $request_field = "request_confirm_{$try}";

        return Account::find($this->$request_field);
    }
}
