<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Autoscuola
 */

namespace Application\Autoscuola\Model;

use ActiveRecord\Base;
use Pongho\Utilities\DateTime;

/**
 * LocationMap
 *
 * @property int    $location1_id
 * @property int    $location2_id
 * @property int    $value
 * @property string $typology
 * @property \Application\Autoscuola\Model\Location $location1
 * @property \Application\Autoscuola\Model\Location $location2
 */
class LocationMap extends Base
{
    /**
     * Nome della tabella.
     *
     * @var string
     */
    public static $table_name = 'autoscuola_locations_map';

    /**
     * Chiave primaria.
     *
     * @var array
     */
    public static $primary_key = ['location1_id', 'location2_id', 'typology'];

    /**
     * Relazione `belongs_to`.
     *
     * @var array
     */
    public static $belongs_to = [
        ['location1', 'model' => \Application\Autoscuola\Model\Location::class],
        ['location2', 'model' => \Application\Autoscuola\Model\Location::class],
    ];

    /**
     * @var array
     */
    public static $values = [
        ['Non impostato', 'blank'],
        ['OK', 'green'],
        ['NO', 'red'],
        ['OK/NO 18 20', 'yellow'],
    ];

    /**
     * Cerco il valore in base ai due id
     *
     * @param array  $ids
     * @param string $typology
     * @return LocationMap
     */
    public static function findByIdsAndTypology(array $ids, $typology)
    {
        if (!is_array($ids) || !isset($ids[1]) || !(is_numeric($ids[0]) && is_numeric($ids[1]))) {
            throw new \RuntimeException('findByIds requires an array with 2 integer values as argument');
        }

        $options = [
            'conditions' => [
                'location1_id = :loc1 AND location2_id = :loc2 AND typology = :typology',
                'loc1'     => $ids[0],
                'loc2'     => $ids[1],
                'typology' => $typology,
            ],
        ];

        return LocationMap::first($options);
    }

    /**
     * Restituisce un valore che indica se il luogo di partenza è compatibile.
     *
     * @param int      $event_start_place_id
     * @param int      $start_place_id
     * @param DateTime $booking_time
     * @param array    $location_map
     * @return bool
     */
    public static function checkLocationCompatibility($event_start_place_id, $start_place_id, DateTime $booking_time, array $location_map)
    {
        $compatible = true;

        if (array_key_exists($event_start_place_id, $location_map)) {
            if (array_key_exists($start_place_id, $location_map[$event_start_place_id])) {
                $value = $location_map[$event_start_place_id][$start_place_id];
            } else {
                $value = 0;
            }

            switch ($value) {
                case 0:
                case 2:
                    // Non settato, non compatibile
                    $compatible = false;
                    break;

                case 1:
                    // Compatibile, non faccio niente
                    break;

                case 3:
                    // Compatibile fino alle 18
                    if (intval($booking_time->format('H')) >= 18) {
                        $compatible = false;
                    }

                    break;
            }
        } else {
            // Il luogo di partenza dell'evento non c'è nella mappa, potrebbe essere stato cancellato dopo
            // la creazione dell'evento. Imposto come non compatibile (lo stesso che non settato).
            //$compatible = false;
        }

        return $compatible;
    }
}
