<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Cms
 */

namespace Application\Autoscuola;

use Application\Autoscuola\Handler\View\ResourceDay;
use Application\Autoscuola\Installer\Installer;
use Application\Autoscuola\Model\Location;
use Application\Autoscuola\Model\ModuleSite as Autoscuola;
use Application\Autoscuola\Model\Student;
use Application\Calendar\Model\Calendar;
use Application\Core\Model\Account;
use Application\Core\Model\Role;
use Pongho\Core\Plugin;
use Pongho\EventDispatcher\Event;
use Pongho\Http\Exception\HttpUnauthorizedException;
use Pongho\Http\RedirectResponse;

/**
 * Kernel
 */
class Kernel extends Plugin
{
    public const VERSION = '1.0.0-dev';

    public const FRONT_CONTROLLER = 'Application\\Autoscuola\\Controller\\AutoscuolaController';

    /**
     * {@inheritdoc}
     */
    public static function depends()
    {
        return ['Cms', 'Calendar'];
    }

    /**
     * @var array
     */
    public static $options = [
        'calendario_esami' => 6,
    ];

    /**
     * {@inheritdoc}
     */
    public function setUp()
    {
        return true;
    }

    /**
     * {@inheritdoc}
     */
    public function load()
    {
        $this->connect('admin.translations.filter_partitions', 'filterTranslationsPartitions');

        // Calendario
        $this->connect('calendar.index_action', 'redirectAnon');
        $this->connect('calendar.index.filter_view_handler', 'filterViewHandlers');
        $this->connect('calendar.filter_autocomplete_options', 'filterAutocompleteOptions');
        $this->connect('admin.calendar.filter_owners_options', 'filterAdminCalendarOwnerSelectOptions');

        // Campo sede di riferimento per gli insegnanti
        $this->connect('admin.users.filter_fields', 'setAutoscuolaCustomFieldsAdminAccount');
        $this->connect('admin.user.view_edit.after_address_panel', 'displayAutoscuolaCustomFieldsAdminAccount');
        $this->connect('admin.user.view_edit.after_user_info', 'displayBirthCustomFieldsAdminAccount');

        // Pagine private
        $this->connect('cms.privatepage.rewrite_node_action', 'accessLimit');
        $this->connect('cms.privatepage.rewrite_archive_action', 'accessLimit');
    }

    /**
     * {@inheritdoc}
     */
    public function install()
    {
        Installer::newInstance($this->container)->install();
    }

    /**
     * {@inheritdoc}
     */
    public function uninstall()
    {
        Installer::newInstance($this->container)->uninstall();
    }

    /**
     * {@inheritdoc}
     */
    public function enable()
    {
    }

    /**
     * {@inheritdoc}
     */
    public function disable()
    {
    }

    /**
     * @param Event $event
     * @param array $partitions
     * @return array
     */
    public function filterTranslationsPartitions(Event $event, array $partitions)
    {
        $partitions[] = 'autoscuola';

        return $partitions;
    }

    /**
     * @param Event $event
     * @param array $options
     * @return array
     */
    public function filterAdminCalendarOwnerSelectOptions(Event $event, array $options)
    {
        $roles = [Autoscuola::ROLE_TEACHER, Autoscuola::ROLE_TEACHER_P, Autoscuola::ROLE_ADMIN, Role::ADMIN];

        $options = [
            'conditions' => [
                'is_active = :active AND role_id IN (' . implode(',', $roles) . ') AND is_founder = :founder',
                'active'  => 1,
                'founder' => 0,
            ],
            'order'      => 'CASE WHEN role_id = ' . Role::ADMIN . ' THEN 0 ELSE 1 END, fullname ASC',
            // tanto per la cronaca, sì, si può fare, metto prima gli amministratori e poi gli altri a seguire
        ];

        return $options;
    }

    /**
     * Filtra i risultati dell'autocomplete per la selezione degli studenti nel calendario
     *
     * @param Event $event
     * @param array $options
     * @return array
     */
    public function filterAutocompleteOptions(Event $event, array $options)
    {
        $students_table = Student::tableName();

        $options['select'] = '`from`.*';
        $options['joins'] = "INNER JOIN {$students_table} AS st ON st.user_id = `from`.id";
        $options['group'] = '`from`.id';

        $options['conditions'] = str_replace('AND id NOT IN', 'AND `from`.id NOT IN', $options['conditions']);

        $options = Account::addCondition($options, ['st.status = :status', 'status' => 'active']);

        return $options;
    }

    /**
     * @param Event $event
     * @param array $fields
     * @return array
     */
    public function setAutoscuolaCustomFieldsAdminAccount(Event $event, array $fields)
    {
        /* Selettore per la sede di insegnamento */
        $options = [
            'conditions' => ['typology = :typology', 'typology' => 0],
            'order'      => 'name ASC',
        ];

        $offices = [null => 'Scegli una sede'];
        foreach (Location::all($options) as $office) {
            $offices[$office->id] = $office->name;
        }

        $fields[] = ['custom_place', 'class' => \Application\Admin\Form\SelectField::class, 'items' => $offices];

        $fields[] = [
            'places',
            'class'         => \Application\Autoscuola\Form\ChecklistLuoghiInsegnanti::class,
            'attribute_key' => 'user_id',
            'foreign_key'   => 'location_id',
            'foreign'       => \Application\Autoscuola\Model\Location::class,
            'through'       => \Application\Autoscuola\Model\LocationUser::class,
        ];

        $fields[] = [
            'locations',
            'class'         => \Application\Autoscuola\Form\ChecklistLuoghiIstruttori::class,
            'attribute_key' => 'user_id',
            'foreign_key'   => 'location_id',
            'foreign'       => \Application\Autoscuola\Model\Location::class,
            'through'       => \Application\Autoscuola\Model\LocationUser::class,
        ];

        $fields[] = [
            'custom_birth_date',
            'class' => \Application\Admin\Form\DateField::class,
        ];
        $fields[] = ['custom_birth_place'];
        $fields[] = [
            'custom_birth_province',
            'class' => \Application\Core\Form\ProvinceField::class,
        ];

        return $fields;
    }

    /**
     * @param Event $event
     */
    public function displayAutoscuolaCustomFieldsAdminAccount(Event $event)
    {
        $view = $event->getSubject();
        $lang = $this->container->get('language');
        $user = $event->getSubject()->model;

        $code = '<fieldset class="panel">';
        $code .= '<h3>Autoscuola</h3>';

        if ($user->role_id === Autoscuola::ROLE_STUDENT_T || $user->role_id === Autoscuola::ROLE_STUDENT_P || $user->role_id === Autoscuola::ROLE_ADMIN || $user->role_id === Role::ADMIN) {
            $code .= '<dl class="extend"><dt><label for="custom_place">' . $lang->custom_place . '</label></dt><dd>' . $view->field_custom_place . '</dd></dl>';
        }

        if ($user->role_id === Autoscuola::ROLE_TEACHER || $user->role_id === Autoscuola::ROLE_TEACHER_T) {
            $code .= '<dl class="extend"><dt><label for="places">' . $lang->custom_place . '</label></dt><dd>' . $view->field_places . '</dd></dl>';
        }

        if ($user->role_id === Autoscuola::ROLE_TEACHER || $user->role_id === Autoscuola::ROLE_TEACHER_P) {
            $code .= '<dl class="extend"><dt><label for="locations">' . $lang->custom_zones . '</label></dt><dd>' . $view->field_locations . '</dd></dl>';
        }

        $code .= '</fieldset>';

        echo $code;
    }

    /**
     * @param Event $event
     */
    public function displayBirthCustomFieldsAdminAccount(Event $event)
    {
        $view = $event->getSubject();
        $lang = $this->container->get('language');

        $code = '<dl class="extend"><dt><label for="custom-birth-date">' . $lang->birth_date . '</label></dt><dd>' . $view->field_custom_birth_date . '</dd></dl>';
        $code .= '<dl class="extend"><dt><label for="custom-birth-place">' . $lang->birth_place . '</label></dt><dd>' . $view->field_custom_birth_place . '</dd></dl>';
        $code .= '<dl class="extend"><dt><label for="custom-birth-province">' . $lang->birth_province . '</label></dt><dd>' . $view->field_custom_birth_province . '</dd></dl>';

        echo $code;
    }

    /**
     * @param Event $event
     * @return RedirectResponse
     */
    public function redirectAnon(Event $event)
    {
        if (!$this->container->get('current_user')->isLogged()) {
            $path = $this->container->getParameter('site_module')->path;

            return new RedirectResponse('/user/login/?redirect=' . absolute_url($path));
        }

        // Redirect per non mostrare l'index del calendario che non serve
        return new RedirectResponse('/areaprivata/');
    }

    /**
     * @param Event $event
     */
    public function accessLimit(Event $event)
    {
        $user = $this->container->get('current_user');

        if (!$user->isLogged()) {
            $path = isset($event['node']) ? $event['node']->permalink() : absolute_url($this->container->getParameter('site_module')->path);

            return new RedirectResponse('/user/login/?redirect=' . $path);
        }

        switch ($user->role_id) {
            case Role::ADMIN:
            case Autoscuola::ROLE_ADMIN:
            case Autoscuola::ROLE_TEACHER:
            case Autoscuola::ROLE_TEACHER_P:
            case Autoscuola::ROLE_TEACHER_T:
            case Autoscuola::ROLE_STUDENT_P:
            case Autoscuola::ROLE_STUDENT_T:
                return false;
                break;

            default:
                throw new HttpUnauthorizedException();
                break;
        }
    }

    /**
     * Gestisce l'aspetto e il comportamento del calendario
     *
     * Modalità di visualizzazione degli eventi:
     * [multi]    : tanti calendari quanti sono quelli da caricare, in ogni calendario i suoi eventi
     * [single]    : un solo calendario, tutti gli eventi dei calendari caricati
     *
     * @param Event $event
     * @param       $handler
     * @return string
     */
    public function filterViewHandlers(Event $event, $handler)
    {
        $role = $this->container->get('current_user')->role_id;

        if ($role === Role::ADMIN || $role === Autoscuola::ROLE_ADMIN) {
            return ResourceDay::getClassPath();
        } else {
            return $handler;
        }
    }

    /**
     * @param array $calendars
     * @param array $options
     * @return array
     */
    protected function getCalendarsSelectOption(array $calendars = [], array $options = [])
    {
        if (empty($calendars)) {
            $calendars = Calendar::all($options);
        }

        $select_options = [];

        /** @var Calendar $calendar */
        foreach ($calendars as $calendar) {
            $select_options[$calendar->id] = $calendar->title;
        }

        return $select_options;
    }
}
