<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author Daniele Termini
 * @package Application\Autoscuola
 */

namespace Application\Autoscuola\Installer;

use Application\Autoscuola\Kernel;
use Application\Core\Model\ModuleSite;

/**
 * Class Installer
 */
class Installer extends \Application\Core\Installer\Installer
{
    /**
     * Crea la struttura in database
     */
    public function install()
    {
        /** @var \Monolog\Logger $logger */
        $logger = $this->container->get('logger');

        $logger->info('[Autoscuola][Installer] Installazione autoscuola');

        $schema = $this->getSchema();

        // Tabella prospezioni
        $prospect_table = $schema->createTable('pongho_autoscuola_prospect');
        $prospect_table->addColumn('id', 'integer', ['unsigned' => true, 'autoincrement' => true]);
        $prospect_table->addColumn('place', 'string', ['length' => 10, 'default' => '']);
        $prospect_table->addColumn('surname', 'string', ['length' => 255, 'default' => '']);
        $prospect_table->addColumn('name', 'string', ['length' => 255, 'default' => '']);
        $prospect_table->addColumn('birth_date', 'datetime');
        $prospect_table->addColumn('address', 'text', ['length' => 65535, 'default' => '']);
        $prospect_table->addColumn('client', 'boolean', ['default' => false]);

        $prospect_table->setPrimaryKey(['id']);


        // Tabella luoghi
        $locations_table = $schema->createTable('pongho_autoscuola_locations');
        $locations_table->addColumn('id', 'integer', ['unsigned' => true, 'autoincrement' => true]);
        $locations_table->addColumn('name', 'string', ['length' => 255, 'default' => '']);
        $locations_table->addColumn('abbreviation', 'string', ['length' => 10, 'default' => '']);
        $locations_table->addColumn('typology', 'string', ['length' => 20, 'default' => '']);

        $locations_table->setPrimaryKey(['id']);


        // Tabella mappa di compatibilità dei luoghi
        $locations_map_table = $schema->createTable('pongho_autoscuola_locations_map');
        $locations_map_table->addColumn('location1_id', 'integer', ['unsigned' => true]);
        $locations_map_table->addColumn('location2_id', 'integer', ['unsigned' => true]);
        $locations_map_table->addColumn('value', 'integer', ['default' => 0]);
        $locations_map_table->addColumn('typology', 'string', ['length' => 10, 'default' => '']);

        $locations_map_table->setPrimaryKey(['location1_id', 'location2_id', 'typology']);
        $locations_map_table->addForeignKeyConstraint($locations_table, ['location1_id'], ['id']);
        $locations_map_table->addForeignKeyConstraint($locations_table, ['location2_id'], ['id']);


        // Tabella di compatibilità con gli utenti
        $locations_users_table = $schema->createTable('pongho_autoscuola_locations_users');
        $locations_users_table->addColumn('location_id', 'integer', ['unsigned' => true]);
        $locations_users_table->addColumn('user_id', 'integer', ['unsigned' => true]);
        $locations_users_table->addColumn('typology', 'string', ['length' => 10, 'default' => '']);

        $locations_users_table->setPrimaryKey(['location_id', 'user_id', 'typology']);
        $locations_users_table->addForeignKeyConstraint($locations_table, ['location_id'], ['id']);
        $locations_users_table->addForeignKeyConstraint($schema->getTable('pongho_users'), ['user_id'], ['id']);


        // Tabella degli studenti
        $students_table = $schema->createTable('pongho_autoscuola_students');
        $students_table->addColumn('id', 'integer', ['unsigned' => true, 'autoincrement' => true]);
        $students_table->addColumn('user_id', 'integer', ['unsigned' => true]);
        $students_table->addColumn('subscription_date', 'datetime');
        $students_table->addColumn('location_id', 'integer', ['unsigned' => true, 'notnull' => false]);
        $students_table->addColumn('typology', 'string', ['length' => 50, 'default' => '']);
        $students_table->addColumn('license_type', 'string', ['length' => 10, 'default' => '']);
        $students_table->addColumn('license_date', 'datetime', ['notnull' => false]);
        $students_table->addColumn('temp_license_date', 'datetime', ['notnull' => false]);
        $students_table->addColumn('status', 'string', ['length' => 10, 'default' => '']);
        $students_table->addColumn('request_period_1', 'string', ['length' => 50, 'default' => '']);
        $students_table->addColumn('request_confirm_1', 'integer', ['unsigned' => true, 'notnull' => false]);
        $students_table->addColumn('request_date_1', 'datetime', ['notnull' => false]);
        $students_table->addColumn('request_result_1', 'string', ['length' => 50, 'default' => '']);
        $students_table->addColumn('request_period_2', 'string', ['length' => 50, 'default' => '']);
        $students_table->addColumn('request_confirm_2', 'integer', ['unsigned' => true, 'notnull' => false]);
        $students_table->addColumn('request_date_2', 'datetime', ['notnull' => false]);
        $students_table->addColumn('request_result_2', 'string', ['length' => 50, 'default' => '']);
        $students_table->addColumn('created_at', 'datetime');
        $students_table->addColumn('updated_at', 'datetime');

        $students_table->setPrimaryKey(['id']);
        $students_table->addForeignKeyConstraint($schema->getTable('pongho_users'), ['user_id'], ['id']);
        $students_table->addForeignKeyConstraint($locations_table, ['location_id'], ['id']);
        $students_table->addForeignKeyConstraint($schema->getTable('pongho_users'), ['request_confirm_1'], ['id']);
        $students_table->addForeignKeyConstraint($schema->getTable('pongho_users'), ['request_confirm_2'], ['id']);


        // Tabella dei commenti degli studenti
        $students_comments_table = $schema->createTable('pongho_autoscuola_students_comments');
        $students_comments_table->addColumn('id', 'integer', ['unsigned' => true, 'autoincrement' => true]);
        $students_comments_table->addColumn('student_id', 'integer', ['unsigned' => true]);
        $students_comments_table->addColumn('author_id', 'integer', ['unsigned' => true]);
        $students_comments_table->addColumn('content', 'text', ['length' => 65535, 'default' => '']);
        $students_comments_table->addColumn('created_at', 'datetime');

        $students_comments_table->setPrimaryKey(['id']);
        $students_comments_table->addForeignKeyConstraint($students_table, ['student_id'], ['id']);
        $students_comments_table->addForeignKeyConstraint($schema->getTable('pongho_users'), ['author_id'], ['id']);

        $this->updateSchema();

        $this->fixRegisteredUpdates();

        $module = $this->getOrCreateModule(Kernel::FRONT_CONTROLLER);

        $options = [
            'conditions' => [
                'module_id = :module AND site_id = :site',
                'module' => $module->id,
                'site'   => $this->container->get('site')->id,
            ],
        ];

        if (ModuleSite::first($options) === null) {
            ModuleSite::create([
                'module_id'  => $module->id,
                'site_id'    => $this->container->get('site')->id,
                'is_enabled' => true,
            ]);
        }
    }

    /**
     * Rimuove la struttura dal database
     */
    public function uninstall()
    {
        /** @var \Monolog\Logger $logger */
        $logger = $this->container->get('logger');

        $logger->info('[Autoscuola][Installer] Disinstallazione autoscuola');

        $schema = $this->getSchema();

        if ($schema->hasTable('pongho_autoscuola_students_comments')) {
            $schema->dropTable('pongho_autoscuola_students_comments');
        }

        if ($schema->hasTable('pongho_autoscuola_students')) {
            $schema->dropTable('pongho_autoscuola_students');
        }

        if ($schema->hasTable('pongho_autoscuola_locations_map')) {
            $schema->dropTable('pongho_autoscuola_locations_map');
        }

        if ($schema->hasTable('pongho_autoscuola_locations_users')) {
            $schema->dropTable('pongho_autoscuola_locations_users');
        }

        if ($schema->hasTable('pongho_autoscuola_locations')) {
            $schema->dropTable('pongho_autoscuola_locations');
        }

        if ($schema->hasTable('pongho_autoscuola_prospect')) {
            $schema->dropTable('pongho_autoscuola_prospect');
        }

        $this->updateSchema();

        if ($module = $this->getOrCreateModule(Kernel::FRONT_CONTROLLER, true)) {
            $options = [
                'conditions' => [
                    'module_id = :module',
                    'module' => $module->id,
                ],
            ];

            foreach (ModuleSite::all($options) as $module_site) {
                $module_site->delete();
            }

            $module->delete();
        }
    }
}
