<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Autoscuola
 */

namespace Application\Autoscuola\Handler\View;

use Application\Autoscuola\Model\Location;
use Application\Calendar\Handler\View\Base;
use Application\Calendar\Handler\ViewInterface;

/**
 * GuideAdmin
 */
class GuideAdmin extends Base implements ViewInterface
{
    public function getView()
    {
        $path = $this->site_module->path;
        $calendars = $this->data_handler->getCalendars();

        $calendar_ids = [];
        $resources = [];
        foreach ($calendars as $calendar) {
            $calendar_ids[] = $calendar->id;
            $resources[] = [
                'id'   => $calendar->id,
                'name' => shorter_string($calendar->title, 15),
            ];
        }

        $events = [
            'url'  => url("{$path}getevents/"),
            'data' => [
                'calendars' => $calendar_ids,
                'handler'   => $this->data_handler->getClassPath(),
            ],
            'type' => 'POST',
        ];
        $calendar_options['events'] = $events;

        $json = json_encode(array_merge([
            'resources'    => $resources,
            'events'       => $events,
            'selectable'   => true,
            'selectHelper' => true,
            'handler'      => $this->data_handler->getClassPath(),
        ], $this->calendar_options));

        $places = [];
        foreach (Location::all() as $location) {
            $places[$location->id] = $location->abbreviation;
        }

        $places = json_encode($places);

        $check_compatibility_url = url('/areaprivata/checkcompatibility/');
        $calendar_icon = themes_url('/images/calendar.png');

        $regexp = '/([^>\r\n]?)(\r\n|\n\r|\r|\n)/g';
        $code = <<<JS
var o = $json, places = $places;

function nl2br (str) {
    return (str + '').replace($regexp, '$1<br>$2');
}

function myEventCustomFieldsCallback(calEvent, popup) {
	$('[name="custom_start_place"]', popup).val(calEvent.custom_start_place);
	$('[name="custom_not_registered"]', popup).val(calEvent.custom_not_registered);
}

function myEventSaveCustomFieldsCallback(calEvent, popup, data) {
	calEvent.custom_start_place = $('[name="custom_start_place"]', popup).val();
	calEvent.custom_not_registered = $('[name="custom_not_registered"]', popup).is(':checked');

	return data;
}

function getMinutesFromTime(time) {
	var timesplit = time.split(':');

	return parseInt(timesplit[0], 10) * 60 + parseInt(timesplit[1], 10);
}

function checkValidTimeRange(start_hour, end_hour) {
	var start = getMinutesFromTime(start_hour),
		end = getMinutesFromTime(end_hour);

	if ( end <= start) return false;

	return true;
}

function myBeforeEventSaveCallback(cal, calEvent, data) {

	// Gestione delle ore degli eventi senza il giorno
	var day = new XDate(data['start_date_at']).toString('yyyy-MM-dd'),
		start_hour, end_hour;

		start_hour = (data['custom_start_hour_at'] === undefined ? new XDate(data['start_date_at']).toString('HH:mm') : data['custom_start_hour_at']);
		end_hour = (data['custom_end_hour_at'] === undefined ? new XDate(data['end_date_at']).toString('HH:mm') : data['custom_end_hour_at']);

	if ( checkValidTimeRange(start_hour, end_hour) ) {
		data['start_date_at'] = new XDate(day+' '+start_hour).toString('yyyy-MM-dd HH:mm');
		data['end_date_at'] = new XDate(day+' '+end_hour).toString('yyyy-MM-dd HH:mm');
	}

	data['custom_start_place'] = calEvent.custom_start_place;
	data['custom_not_registered'] = calEvent.custom_not_registered;

	if (data.custom_start_place && data.action !== 'delete') {
		$.post('$check_compatibility_url', data, function(d) {
			if (d.error) {
				alert(d.message);
			} else {
				if ( !d.compatible ) {
					alert('Tempo di percorrenza tra le due sedi troppo lungo, ti consiglio di cambiare orario');
				}
			}
		});
	}

	return data;
}

function myEventRenderCallback(calEvent, element) {
	var head = element.find('.fc-event-head'),
		label, title, desc = '',
		p = calEvent.participants,
		place = calEvent.custom_start_place,
		nl2br_description = nl2br(calEvent.description),
		hasDescCode = calEvent.description ? '<span class="fc-event-has-description">!</span>' : '';

	if (p === undefined) p = [];

	// Potrebbe essere una guida con un utente non registrato
	if (calEvent.custom_not_registered) {
		title = '<span class="fc-custom-event-type">' + (places[place] || '') + '</span> ' + nl2br_description;
	} else if (calEvent.categoryName && calEvent.categoryName !== 'GUIDE PRE ESAME' && calEvent.categoryName !== 'LEZIONI INDIVIDUALI') {
		if (calEvent.description) {
			title = calEvent.categoryName + ': ' + calEvent.description.substr(0, 20) + '...';
		} else {
			title = calEvent.categoryName;
		}
		desc = nl2br_description;
	} else {
		if (p.length > 1) {
			title = '<div><span class="fc-custom-event-type">' + (places[place] || '') + '</span> <span class="fc-custom-event-user">Più partecipanti...</span></div>';
			desc = '<ul class="fc-custom-event-user-list">';
			$.each(calEvent.participants, function(index, participant) {
				desc += '<li>' + participant.label + '</li>';
			});
			desc += '<ul>';
		} else {
			if (typeof calEvent.participants[0] !== 'undefined' ) {
				label = calEvent.participants[0].label;
			} else {
				label = nl2br_description || 'Nessun partecipante';
			}

			title = '<div><span class="fc-custom-event-type">' + (places[place] || '') + '</span> <span class="fc-custom-event-user">' + label + '</span></div>';
			desc = '<span>' + nl2br_description + '</span>';
		}
	}

	if (calEvent.description) {
	    title += hasDescCode;
	}

	element.find('.fc-event-title').html(desc);
	head.find('.fc-event-time').remove();
	head.append(title);
	head.attr('title', calEvent.description || '');
	head.css({
	    display: 'flex',
	    justifyContent: 'space-between'
	});

	//element.find('.fc-event-content').append(desc);
}

o.eventCustomFieldsCallback = myEventCustomFieldsCallback;
o.eventSaveCustomFieldsCallback = myEventSaveCustomFieldsCallback;
o.eventRenderCallback = myEventRenderCallback;
o.beforeEventSaveCallback = myBeforeEventSaveCallback;

$(document).ready(function() {
	var cal = $('#calendar'), title, dp;
	cal.ponghoFullCalendar(o);

	title = $('.fc-header-title', cal);
	dp = $('<input type="hidden" />');

	dp.appendTo(title).datepicker({
		showOn: 'button',
		beforeShow: function(input, inst) {
			$(input).val(new XDate(cal.fullCalendar('getDate')).toString('yyyy-MM-dd'));
		},
		buttonImage: '$calendar_icon',
		buttonImageOnly: true,
		buttonText: 'Clicca per scegliere una data',
		dateFormat: "yy-mm-dd",
		onSelect: function(dateText, inst) {
			cal.fullCalendar('gotoDate', new Date(dateText));
		}
	});

	title.on('click', function(event) {
		dp.datepicker('show');
	});
});

JS;

        $this->populateHead($code);

        return $calendars;
    }

    public static function getClassPath()
    {
        return \Application\Autoscuola\Handler\View\GuideAdmin::class;
    }
}
