<?php

namespace Application\Autoscuola\Handler\Data;

use Application\Autoscuola\Model\Location;
use Application\Autoscuola\Model\LocationMap;
use Application\Autoscuola\Model\LocationUser;
use Application\Autoscuola\Model\ModuleSite as Autoscuola;
use Application\Calendar\Handler\Data\Base;
use Application\Calendar\Model\Calendar;
use Application\Calendar\Model\CalendarCategory;
use Application\Calendar\Model\CalendarEvent;
use Application\Calendar\Model\CalendarEventUser;
use Application\Calendar\Model\CalendarUser;
use Application\Core\Model\Account;
use Application\Core\Model\Role;
use Pongho\DependencyInjection\Container;
use Pongho\Utilities\DateTime;

class GuideStudent extends Base
{
    protected $filters = [];

    protected $location_map = [];

    public static function getClassPath()
    {
        return \Application\Autoscuola\Handler\Data\GuideStudent::class;
    }

    public function getCalendars()
    {
        $roles = [Autoscuola::ROLE_TEACHER_P, Autoscuola::ROLE_TEACHER];

        $options = [
            'select'     => '`from`.*',
            'joins'      => 'INNER JOIN ' . Account::tableName() . ' AS a ON a.id = `from`.owner_id'
                          . ' INNER JOIN ' . LocationUser::tableName() . 'AS l ON l.user_id = `from`.owner_id',
            'conditions' => ['a.role_id IN (' . implode(',', $roles) . ') AND status = :status', 'status' => 'active'],
            'order'      => '`from`.position ASC',
        ];

        if ($this->filters) {
            foreach ($this->filters as $field => $value) {
                $options = Calendar::addCondition($options, ["$field = :$field", $field => $value]);
            }
        }

        $calendars = [];

        foreach (Calendar::all($options) as $calendar) {
            $id = 'calendar-' . $calendar->id;

            $calendar->is_editable = true;

            $event_options = [
                'conditions' => ['calendar_id = :calendar', 'calendar' => $calendar->id],
            ];

            $calendar->events = [];

            $calendars[$id] = $calendar;
        }

        return $calendars;
    }

    /**
     * {@inheritdoc}
     *
     * @param array  $options
     * @param string $start_date
     * @param string $end_date
     * @param array  $calendars
     *
     * @return array
     */
    public function getEvents(/* args */)
    {
        if (func_num_args() < 4) {
            throw new \InvalidArgumentException('This method requires 4 arguments: array $options, $start_date, $end_date, array $calendars');
        }

        $args = func_get_args();

        $options = $args[0];
        $start_date = $args[1];
        $end_date = $args[2];
        $calendars = $args[3];

        if (!is_array($options)) {
            throw new \InvalidArgumentException('The first argument must be an array');
        }

        $this->location_map = [];
        foreach (LocationMap::all(['conditions' => ['typology = ?', 'location']]) as $row) {
            $this->location_map[$row->location1_id][$row->location2_id] = $row->value;
        }

        $events = [];
        $busy = [];

        foreach (CalendarEvent::all($options) as $event) {
            $busy[$event->calendar_id][] = $event;
            $events[] = $this->makeEvent($event, true);
        }

        // Si lavora dalle 08:00 alle 20:00
        $start = new DateTime(date('Y-m-d 08:00:00', $start_date));
        $end = new DateTime(date('Y-m-d 23:00:00', $end_date - 1));

        // Gli allievi possono prenotare per il giorno successivo solo fino alle 18 del giorno corrente
        // in modo da lasciare del tempo utile agli istruttori per organizzarsi
        $now = new DateTime('now');
        $expire = new DateTime('today 18:00:00');
        $expire = $now < $expire ? $expire : new DateTime('tomorrow 18:00:00');
        if ($start < $expire) {
            foreach ($events as &$event) {
                $event['isLocked'] = true;
            }

            return $events;
        }

        // Gli eventi durano 30 minuti
        $fake_event_duration = new \DateInterval('PT30M');
        $day_interval = new \DateInterval('P1D');

        // Le guide sono prenotabili solo nelle prossime 3 settimane
        $end_limit = new \DateTime('today +21 days');

        foreach ($calendars as $calendar_id) {
            // Parto dal primo giorno
            $today = clone $start;

            // Si proseguo fino a raggiungere la data finale
            while ($today < $end) {
                // Salto le domeniche
                if (intval($today->format('w')) === 0) {
                    $today = $today->add($day_interval);

                    continue;
                }

                // Controllo il limite temporale per la prenotazione
                if ($today > $end_limit) {
                    break;
                }

                // Primo evento
                $fake_event_start = new DateTime($today->format('Y-m-d 08:00:00'));
                $day_end = new DateTime($today->format('Y-m-d 23:00:00'));

                while ($fake_event_start < $day_end) {
                    $fake_event_end = clone $fake_event_start;
                    $fake_event_end->add($fake_event_duration);

                    $compatible = true;
                    $free = true;
                    if (array_key_exists($calendar_id, $busy)) {
                        foreach ($busy[$calendar_id] as $event) {
                            if ($event->start_date_at < $fake_event_end && $event->end_date_at > $fake_event_start) {
                                $free = false;

                                break;
                            }

                            // Controllo la località di partenza in base alla mappa delle località
                            // Devo confrontare con l'evento precedente e successivo
                            if ($compatible && ($event->end_date_at == $fake_event_start || $event->start_date_at == $fake_event_end)) {
                                $compatible = $this->checkEventCompatibility($event->custom_start_place, $this->filters['location_id'], $fake_event_start);
                            }
                        }
                    }

                    if ($free) {
                        $events[] = [
                            'editable'           => $compatible,
                            'calendar'           => (int) $calendar_id,
                            'category'           => null,
                            'title'              => 'Guida',
                            'start'              => $fake_event_start->format('Y-m-d H:i'),
                            'end'                => $fake_event_end->format('Y-m-d H:i'),
                            'place'              => '',
                            'description'        => '',
                            'allDay'             => false,
                            'status'             => '',
                            'color'              => $compatible ? Autoscuola::$colors['free']['bg'] : Autoscuola::$colors['nocomp']['bg'],
                            'textColor'          => $compatible ? Autoscuola::$colors['free']['text'] : Autoscuola::$colors['free']['text'],
                            'participants'       => [],
                            'resourceId'         => (int) $calendar_id,

                            // Campi custom gestione guide studente
                            'isLocked'           => false,
                            'custom_start_place' => '',
                            'handler'            => static::getClassPath(),
                            'isCompatible'       => $compatible,
                            'isBooked'           => false,
                            'isMyBooking'        => false,
                        ];
                    }

                    $fake_event_start = $fake_event_end;
                }

                $today = $fake_event_start->add($day_interval);
            }
        }

        return $events;
    }

    protected function makeEvent(CalendarEvent $event, $is_editable = true)
    {
        $event_participants = $event->getParticipants();
        $is_editable = false;
        $is_locked = false;

        // è prenotato da qualcun altro?
        $is_booked = (($event->author_id !== $this->user->id) && (count($event_participants)) || $event->custom_not_registered);

        $participants = [];
        $in_participants = false;
        $event_booked_date = new DateTime();
        foreach ($event_participants as $participant) {
            $p = [];
            foreach ($participant->attributes as $field => $value) {
                $p[$field] = ($value instanceof DateTime ? $value->format('Y-m-d') : $value);
            }

            // Campi supplementari
            $p['label'] = $participant->user->name(true);

            $participants[] = $p;

            // Controllo se l'utente corrente è tra i partecipanti
            if ($is_editable === false && $participant->user_id === $this->user->id) {
                $event_booked_date = $participant->created_at;
                $in_participants = true;
                $is_editable = true;
                break;
            }
        }

        $is_my_booking = ($event->author_id === $this->user->id) || $in_participants;

        // Limito la modifica degli eventi in base all'ora
        $start_date = new DateTime('tomorrow');
        if (new DateTime() > new DateTime('today 18:00:00')) {
            $start_date->add(new \DateInterval('P1D'));
        }

        $compatible = $this->checkEventCompatibility($event->custom_start_place, $this->filters['location_id'], $event->start_date_at);

        // Test per rendering
        if ($event->category_id !== null) {
            // stato: non prenotabile
            // effetto: non clickabile (non puoi modificare questa prenotazione)
            $is_locked = true;
            $color = Autoscuola::$colors['busy']['bg'];
            $text_color = Autoscuola::$colors['busy']['text'];
        } else {
            if ($is_booked && !$is_my_booking) {
                // stato: occupato da altri
                // effetto: non clickabile ( la guida è già stata prenotata )
                $is_locked = false;
                //$color = $event->calendar->getColor('bg');
                //$text_color = $event->calendar->getColor('text');
                $color = Autoscuola::$colors['booked']['bg'];
                $text_color = Autoscuola::$colors['booked']['text'];
            } else {
                if ($is_my_booking) {
                    // stato: prenotato da me
                    // posso cancellarlo solo entro 5 minuti dall'ora di prenotazione
                    if ($event_booked_date > new DateTime('now -5 minutes')) {
                        // effetto: cancellazione ( vuoi eliminare la prenotazione? )
                        $is_locked = false;
                        $color = $event->calendar->getColor('bg');
                        $text_color = $event->calendar->getColor('text');
                    } else {
                        // effetto: non clickabile ( 24h dall'inizio )
                        $is_locked = true;
                        $color = Autoscuola::$colors['locked']['bg'];
                        $text_color = Autoscuola::$colors['locked']['text'];
                    }
                } else {
                    $is_locked = false;
                    if ($compatible) {
                        // stato: libero
                        // effetto: prenotabile
                        $color = Autoscuola::$colors['free']['bg'];
                        $text_color = Autoscuola::$colors['free']['text'];
                    } else {
                        // stato: non compatibile
                        // effetto: non clickabile ( non puoi prenotare questa guida perché non è compatibile con il filtro impostato )
                        $color = Autoscuola::$colors['nocomp']['bg'];
                        $text_color = Autoscuola::$colors['nocomp']['text'];
                    }
                }
            }
        }

        return [
            'id'                 => $event->id,
            'author'             => $event->author_id,
            'editable'           => $is_locked,
            'calendar'           => $event->calendar_id,
            'category'           => $event->category_id,
            'title'              => $event->title,
            'start'              => $event->start_date_at->format('Y-m-d H:i'),
            'end'                => $event->end_date_at->format('Y-m-d H:i'),
            'place'              => $event->place,
            'description'        => $event->content,
            'allDay'             => $event->is_all_day,
            'status'             => $event->status,
            'color'              => $color,
            'textColor'          => $text_color,
            'participants'       => $participants,
            'resourceId'         => $event->calendar->id,

            // Campi custom gestione guide studente
            'custom_start_place' => $event->custom_start_place,
            'handler'            => static::getClassPath(),
            'isLocked'           => $is_locked,
            'isCompatible'       => $compatible,
            'isBooked'           => $is_booked,
            'isMyBooking'        => $is_my_booking,
        ];
    }

    protected function checkEventCompatibility($event_start_place_id, $start_place_id, DateTime $booking_time)
    {
        return LocationMap::checkLocationCompatibility($event_start_place_id, $start_place_id, $booking_time, $this->location_map);
    }

    public static function parseFields($fields)
    {
        /**
         * Eseguo un controllo in modo che non ci sia già un evento inserito che si sovrappone a questo
         * per evitare inserimenti concorrenti (es due allievi prenotano lo stesso evento contemporaneamente)
         */

        $calendar = $fields['calendar_id'];
        $start = $fields['start_date_at'];
        $end = $fields['end_date_at'];
        $author = $fields['author_id'];

        $options = [
            'conditions' => [
                'calendar_id = :calendar AND author_id != :author AND ( (start_date_at <= :start AND end_date_at > :start) OR (start_date_at < :end AND end_date_at >= :end) )',
                'author'   => $author,
                'calendar' => $calendar,
                'start'    => $start,
                'end'      => $end,
            ],
            'order'      => 'created_at ASC',
        ];

        // Prendo solo il primo evento
        if (($event = CalendarEvent::first($options)) !== null) {
            if ($event->hasParticipants() && !$event->inParticipants($author)) {
                throw new \Exception('La guida è già stata prenotata, verrà aggiornato il calendario.');
            } elseif (!$event->hasParticipants() || $event->inParticipants($author)) {
                switch ($fields['action']) {
                    case 'add':
                        $fields['action'] = 'edit';
                        $fields['author_id'] = $event->author_id;
                        break;

                    case 'delete':
                        //if ( $event->getParticipant($event->author_id)->created_at < new DateTime('now -5 minutes') )

                        $fields['action'] = 'edit';
                        $fields['author_id'] = $event->author_id;
                        $fields['participants'] = [];
                        break;
                }
            }
        }

        return $fields;
    }

    public function setFilters(array $filters)
    {
        $this->filters = $filters;
    }

    public function getFilters()
    {
        return $this->filters;
    }

    /**
     * Carica il template del popup, utilizzato dalla chiamata AJAX in AjaxController
     *
     * @static
     * @access public
     *
     * @param array                                 $data i dati provenienti dal POST, gli eventi sono già gestiti dal parsePost()
     * @param \Pongho\DependencyInjection\Container $container
     */
    public static function getPopupTemplate(array $data, Container $container)
    {
        $site_module = $container->get('controller')->getSiteModule();
        $path = $site_module->path;

        if ($data['action'] === 'add') {
            $event = new CalendarEvent();
            $event->calendar_id = $data['resource_id'];
            $event->title = 'Occupato';
            $event->start_date_at = new DateTime($data['start']);
            $event->end_date_at = new DateTime($data['end']);
        } else {
            $event = CalendarEvent::find($data['event_id']);
        }

        $container->get('theme_view')
            ->setTemplate('calendar/popups/guide_admin.php')
            ->assignVars([
                'form_action'               => url("{$path}event/"),
                'autocomplete_url'          => url("{$path}autocomplete/"),
                'autocomplete_filter_roles' => json_encode([Autoscuola::ROLE_STUDENT_P]),
                'categories_url'            => url("{$path}categories/"),
                'category_select'           => Calendar::getCategoriesSelectOptions($event->calendar_id),
                'event'                     => $event,
                'places'                    => Location::getStartPlacesSelectOptions(),
            ]);
    }
}
