<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Autoscuola
 */

namespace Application\Autoscuola\Handler\Data;

use Application\Autoscuola\Model\Location;
use Application\Autoscuola\Model\ModuleSite as Autoscuola;
use Application\Autoscuola\Model\Student;
use Application\Calendar\Handler\Data\Base;
use Application\Calendar\Model\Calendar;
use Application\Calendar\Model\CalendarEvent;
use Application\Calendar\Model\CalendarEventUser;
use Application\Core\Model\Account;
use Pongho\DependencyInjection\Container;
use Pongho\Http\Exception\HttpNotFoundException;
use Pongho\Template\Theme;
use Pongho\Utilities\DateTime;

require_once __DIR__ . '/../../Resources/functions.php';

/**
 * GuideAdmin
 */
class GuideAdmin extends Base
{
    /**
     * {@inheritdoc}
     */
    public static function getClassPath()
    {
        return \Application\Autoscuola\Handler\Data\GuideAdmin::class;
    }

    /**
     * {@inheritdoc}
     */
    public function getCalendars()
    {
        $roles = [Autoscuola::ROLE_TEACHER_P, Autoscuola::ROLE_TEACHER];

        $options = [
            'select'     => '`from`.*',
            'joins'      => 'INNER JOIN ' . Account::tableName() . ' AS a ON a.id = `from`.owner_id',
            'conditions' => [
                'a.role_id IN (' . implode(',', $roles) . ') AND `from`.status = :status',
                'status' => 'active',
            ],
            'order'      => '`from`.position ASC',
        ];

        $calendars = [];

        /** @var Calendar $calendar */
        foreach (Calendar::all($options) as $calendar) {
            $calendar->is_editable = true;
            $calendar->events = [];

            $calendars['calendar-' . $calendar->id] = $calendar;
        }

        return $calendars;
    }

    /**
     * {@inheritdoc}
     *
     * @param array  $options
     * @param string $start_date
     * @param string $end_date
     * @param array  $calendars
     *
     * @return array
     */
    public function getEvents(/* args */)
    {
        if (func_num_args() < 4) {
            throw new \InvalidArgumentException('This method requires 4 arguments: array $options, $start_date, $end_date, array $calendars');
        }

        $args = func_get_args();

        $options = $args[0];
        $start_date = $args[1];
        $end_date = $args[2];
        $calendars = $args[3];

        if (!is_array($options)) {
            throw new \InvalidArgumentException('The first argument must be an array');
        }

        $events = [];
        $busy = [];

        // Nascondo dal calendario le guide degli studenti archiviati
        // Inoltre questa modifica mi impedisce di poter aggiungere guide per uno studente che non è in fase di pratica perché devo filtrare sulla tipologia
        // visto il caso di un utente (7079) che sta facendo più patenti contemporaneamente (A in pratica e B in teoria)
        $options['select'] = '`from`.*';
        $options['joins'] = 'LEFT JOIN ' . CalendarEventUser::tableName() . ' AS eu ON eu.calendar_event_id = `from`.id'
            . ' LEFT JOIN ' . Student::tableName() . ' AS s ON s.user_id = eu.user_id';
        $options['group'] = '`from`.id';

        $options = CalendarEvent::addCondition(
            $options,
            ["((eu.user_id IS NOT NULL AND s.status = 'active') OR (eu.user_id IS NULL))"]
        );

        /** @var CalendarEvent $event */
        foreach (CalendarEvent::all($options) as $event) {
            $busy[$event->calendar_id][] = $event;
            $events[] = $this->makeEvent($event, true);
        }

        // Si lavora dalle 08:00 alle 20:00
        $start = new DateTime(date('Y-m-d 08:00:00', $start_date));
        $end = new DateTime(date('Y-m-d 23:00:00', $end_date - 1));

        // Posso prenotare solo da domani
        if ($start < new DateTime('tomorrow')) {
            return $events;
        }

        // Gli eventi durano 30 minuti
        $fake_event_duration = new \DateInterval('PT60M');
        $day_interval = new \DateInterval('P1D');

        // Le guide sono prenotabili solo nelle prossime 3 settimane
        $end_limit = new \DateTime('today +21 days');

        foreach ($calendars as $calendar_id) {
            // Parto dal primo giorno
            $today = clone $start;

            // Si proseguo fino a raggiungere la data finale
            while ($today < $end) {
                // Salto le domeniche
                if (intval($today->format('w')) === 0) {
                    $today = $today->add($day_interval);

                    continue;
                }

                // Controllo il limite temporale per la prenotazione
                if ($today > $end_limit) {
                    break;
                }

                // Primo evento
                $fake_event_start = new DateTime($today->format('Y-m-d 08:00:00'));
                $day_end = new DateTime($today->format('Y-m-d 23:00:00'));

                while ($fake_event_start < $day_end) {
                    $fake_event_end = clone $fake_event_start;
                    $fake_event_end->add($fake_event_duration);

                    $free = true;
                    if (array_key_exists($calendar_id, $busy)) {
                        foreach ($busy[$calendar_id] as $event) {
                            if ($event->start_date_at < $fake_event_end && $event->end_date_at > $fake_event_start) {
                                $free = false;

                                break;
                            }
                        }
                    }

                    if ($free) {
                        $events[] = [
                            'editable'           => true,
                            'calendar'           => (int) $calendar_id,
                            'category'           => null,
                            'title'              => 'Guida',
                            'start'              => $fake_event_start->format('Y-m-d H:i'),
                            'end'                => $fake_event_end->format('Y-m-d H:i'),
                            'place'              => '',
                            'description'        => '',
                            'allDay'             => false,
                            'status'             => '',
                            'color'              => Autoscuola::$colors['free']['bg'],
                            'textColor'          => Autoscuola::$colors['free']['text'],
                            'participants'       => [],
                            'custom_start_place' => '',
                            'resourceId'         => (int) $calendar_id,
                        ];
                    }

                    $fake_event_start = $fake_event_end;
                }

                $today = $fake_event_start->add($day_interval);
            }
        }

        return $events;
    }

    /**
     * {@inheritdoc}
     */
    protected function makeEvent(CalendarEvent $event, $is_editable = true)
    {
        $participants = [];
        foreach ($event->getParticipants() as $participant) {
            $p = [];
            foreach ($participant->attributes as $field => $value) {
                $p[$field] = ($value instanceof DateTime ? $value->format('Y-m-d') : $value);
            }

            // Campi supplementari
            $p['label'] = $participant->user->name(true);

            $participants[] = $p;
        }

        $category = $event->category;
        $color = $event->calendar->getColor('bg');
        $textColor = $event->calendar->getColor('text');

        if ($category) {
            switch ($category->title) {
                case 'OCCUPATO':
                    $color = Autoscuola::$colors['busy_admin']['bg'];
                    $textColor = Autoscuola::$colors['busy_admin']['text'];

                    break;

                case 'GUIDE PRE ESAME':
                    $color = '#FBC02D';
                    $textColor = '#444';

                    break;

                case 'LEZIONI INDIVIDUALI':
                    $color = '#9C27B0';
                    $textColor = '#fff';

                    break;
            }
        }

        return [
            'id'                    => $event->id,
            'author'                => $event->author_id,
            'editable'              => true,
            'calendar'              => $event->calendar_id,
            'category'              => $event->category_id,
            'categoryName'          => $category ? $category->title : '',
            'title'                 => $event->title,
            'start'                 => $event->start_date_at->format('Y-m-d H:i'),
            'end'                   => $event->end_date_at->format('Y-m-d H:i'),
            'place'                 => $event->place,
            'description'           => $event->content,
            'allDay'                => $event->is_all_day,
            'status'                => $event->status,
            'color'                 => $color,
            'textColor'             => $textColor,
            'participants'          => $participants,
            'custom_start_place'    => $event->custom_start_place,
            'custom_not_registered' => $event->custom_not_registered,
            'resourceId'            => $event->calendar->id,
            'categories'            => Calendar::getCategoriesSelectOptions($event->calendar->id),
        ];
    }

    /**
     * {@inheritdoc}
     */
    public static function getPopupTemplate(array $data, Container $container)
    {
        $site_module = $container->get('controller')->getSiteModule();
        $path = $site_module->path;
        $view = $container->get('theme_view');

        if ($data['action'] === 'add') {
            $event = new CalendarEvent();
            $event->calendar_id = $data['resource_id'];
            $event->title = 'Occupato';
            $event->start_date_at = new DateTime($data['start']);

            // Forzo la durata del nuovo evento ad almeno un’ora
            // prendendo il valore più grande tra (end) e (start + 1 ora)
            $event->end_date_at = max(
                new DateTime($data['end']),
                (new DateTime($data['start']))->add(new \DateInterval('PT60M'))
            );
        } else {
            $event = CalendarEvent::find($data['event_id']);

            // Prendo il primo partecipante
            if ($participants = $event->getParticipants()) {
                $participant = reset($participants);

                static::addStudentVarsToView($participant->user_id, $view);
            }
        }

        // Select per la selezione delle ore
        $from = $site_module->getOption('min_calendar_day_time');
        //$from_minutes = hours2minutes($from);
        $from_hours = explode(':', $from);
        $from_hours = intval($from_hours[0]);

        $to = $site_module->getOption('max_calendar_day_time');
        //$to_minutes = hours2minutes($to);
        $to_hours = explode(':', $to);
        $to_hours = intval($to_hours[0]);

        $select_options = [];
        for ($i = $from_hours; $i < $to_hours; $i++) {
            $hour = str_pad($i, 2, '0', STR_PAD_LEFT);

            $full = $hour . ':00';
            $select_options[$full] = $full;

            $half = $hour . ':30';
            $select_options[$half] = $half;
        }

        $hours_select_from = $select_options;

        array_shift($select_options);
        $select_options[$to] = $to;

        $hours_select_to = $select_options;

        $view
            ->setTemplate('calendar/popups/guide_admin.php')
            ->assignVars([
                'form_action'               => url("{$path}event/"),
                'autocomplete_url'          => url("{$path}autocomplete/"),
                'autocomplete_filter_roles' => json_encode([
                    Autoscuola::ROLE_STUDENT_T,
                    Autoscuola::ROLE_STUDENT_P,
                ]),
                'categories_url'            => url("{$path}categories/"),
                'category_select'           => Calendar::getCategoriesSelectOptions($event->calendar_id),
                'event'                     => $event,
                'places'                    => Location::getStartPlacesSelectOptions(),
                'hours_select_from'         => $hours_select_from,
                'hours_select_to'           => $hours_select_to,
            ]);
    }

    /**
     * @param int   $user_id
     * @param Theme $view
     * @throws HttpNotFoundException
     */
    protected static function addStudentVarsToView($user_id, Theme $view)
    {
        // Ricavo tutti gli studenti della sessione corrente associati all'utente per avere uno storico parziale
        $options = [
            'select'     => 's.*',
            'joins'      => 'INNER JOIN ' . Student::tableName() . ' AS s ON s.subscription_date = `from`.subscription_date AND s.user_id = `from`.user_id',
            'conditions' => [
                '`from`.user_id = :user AND `from`.status = :status',
                'user'   => $user_id,
                'status' => 'active',
            ],
        ];

        $students = Student::all($options);

        // L'autoscuola e gli insegnanti/istruttori possono vedere lo storico dello studente solo se � attivo
        // Gli amministratori (es Meta Line, Alberto Buran) possono vedere sempre lo storico dello studente
        if (!$students) {
            throw new \RuntimeException('Could not find any students to build the history lessons.');
        }

        $current_student = end($students);

        $view
            ->assignVars([
                'student_history_url'  => absolute_url('/private-area/classroom/history/' . $current_student->user_id . '/'),
                'student_registry_url' => absolute_url('/private-area/registry/' . $current_student->user_id . '/'),
            ]);
    }
}
