<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Autoscuola
 */

namespace Application\Autoscuola\Handler\Data;

use Application\Autoscuola\Model\ModuleSite as Autoscuola;
use Application\Calendar\Handler\Data\Base;
use Application\Calendar\Model\Calendar;
use Application\Calendar\Model\CalendarEvent;
use Pongho\DependencyInjection\Container;
use Pongho\Utilities\DateTime;

/**
 * EsamiTeoria
 */
class EsamiTeoria extends Base
{
    /**
     * {@inheritdoc}
     */
    public static function getClassPath()
    {
        return \Application\Autoscuola\Handler\Data\EsamiTeoria::class;
    }

    /**
     * {@inheritdoc}
     */
    public function getCalendars()
    {
        $options = [
            'conditions' => ['id = :calendar', 'calendar' => Autoscuola::CALENDAR_TEORIA],
        ];

        $calendars = [];

        foreach (Calendar::all($options) as $calendar) {
            if ($this->user->getAccount()->getRoleId() === Autoscuola::ROLE_ADMIN) {
                $calendar->is_editable = true;
            }

            $calendar->events = [];

            $calendars['calendar-' . $calendar->id] = $calendar;
        }

        return $calendars;
    }

    /**
     * {@inheritdoc}
     */
    protected function makeEvent(CalendarEvent $event, $is_editable = true)
    {
        $participants = [];
        foreach ($event->getParticipants() as $participant) {
            $p = [];
            foreach ($participant->attributes as $field => $value) {
                $p[$field] = ($value instanceof DateTime ? $value->format('Y-m-d') : $value);
            }

            // Campi supplementari
            $p['label'] = $participant->user->name(true);

            $participants[] = $p;
        }

        return [
            'id'           => $event->id,
            'author'       => $event->author_id,
            'editable'     => ($event->status === ''),
            'calendar'     => $event->calendar_id,
            'category'     => $event->category_id,
            'title'        => $event->title,
            'start'        => $event->start_date_at->format('Y-m-d H:i'),
            'end'          => $event->end_date_at->format('Y-m-d H:i'),
            'place'        => $event->place,
            'description'  => $event->content,
            'allDay'       => $event->is_all_day,
            'status'       => $event->status,
            'color'        => $event->status === '' ? $event->calendar->getColor('bg') : Autoscuola::$colors['confirmed_t']['bg'],
            'textColor'    => $event->status === '' ? $event->calendar->getColor('text') : Autoscuola::$colors['confirmed_t']['text'],
            'participants' => $participants,
        ];
    }

    /**
     * {@inheritdoc}
     */
    public static function getPopupTemplate(array $data, Container $container)
    {
        /** @var \Application\Core\Utilities\ControllerHelper $helper */
        $helper = $container->get('controller_helper');
        $path = $helper->getModuleSite()->path;

        if ($data['action'] === 'add') {
            $event = new CalendarEvent();
            $event->calendar_id = Autoscuola::CALENDAR_TEORIA;
            $event->title = 'Esame teoria';
        } else {
            $event = CalendarEvent::find($data['event_id']);
        }

        $helper->getTheme()
            ->setTemplate('calendar/popups/esami_teoria.php')
            ->assignVars([
                'form_action'               => url("{$path}event/"),
                'autocomplete_url'          => url("{$path}autocomplete/"),
                'autocomplete_filter_roles' => json_encode([Autoscuola::ROLE_STUDENT_T]),
                'categories_url'            => url("{$path}categories/"),
                'event'                     => $event,
                'participants'              => self::getParticipants($event, $container),
                'deregister_url'            => url("/areaprivata/deregister/[user_id]/?t=teoria"),
            ]);
    }

    /**
     * @param CalendarEvent $event
     * @param Container     $container
     * @return \Application\Calendar\Model\CalendarEventUser[]
     */
    private static function getParticipants(CalendarEvent $event, Container $container)
    {
        $participants = [];
        foreach ($event->getParticipants() as $participant) {
            $participants[$participant->user_id] = $participant;
        }

        if ($participants) {
            $user_ids = implode(', ', array_keys($participants));

            $sql = <<<SQL
SELECT
       s.user_id,
       s.license_type,
       l.abbreviation AS location,
       CASE WHEN s.typology = 'pratica' THEN s.temp_license_date
            WHEN s.medical_date THEN s.medical_date
            ELSE NOW() END AS startDate,
       s.quiz_last_errors_avg,
       s.quiz_total,
       s.quiz_last_passed_percent
  FROM pongho_autoscuola_students AS s
 INNER JOIN (SELECT MAX(id) AS student_id, user_id FROM pongho_autoscuola_students WHERE user_id IN ({$user_ids}) GROUP BY user_id) AS tmp ON tmp.student_id = s.id
 INNER JOIN pongho_autoscuola_locations AS l ON l.id = s.location_id
SQL;

            /** @var \Doctrine\DBAL\Connection $connection */
            $connection = $container->get('connection');

            foreach ($connection->fetchAllAssociative($sql) as $row) {
                $participants[$row['user_id']]->license_type = $row['license_type'];
                $participants[$row['user_id']]->location = $row['location'];
                $participants[$row['user_id']]->startDate = new DateTime($row['startDate']);
                $participants[$row['user_id']]->quiz_last_errors_avg = $row['quiz_last_errors_avg'];
                $participants[$row['user_id']]->quiz_total = $row['quiz_total'];
                $participants[$row['user_id']]->quiz_last_passed_percent = $row['quiz_last_passed_percent'];
            }
        }

        return $participants;
    }
}
