<?php

/**
 * Questo file è parte di Pongho.
 *
 * @author    Daniele Termini
 * @copyright Web Agency Meta Line S.r.l.
 * @package   Application\Autoscuola
 */

namespace Application\Autoscuola\Handler\Data;

use Application\Autoscuola\Model\ModuleSite as Autoscuola;
use Application\Calendar\Handler\Data\Base;
use Application\Calendar\Model\Calendar;
use Application\Calendar\Model\CalendarEvent;
use Application\Cms\Model\Node;
use Pongho\DependencyInjection\Container;
use Pongho\Utilities\DateTime;

/**
 * EsamiPratica
 */
class EsamiPratica extends Base
{
    /**
     * {@inheritdoc}
     */
    public static function getClassPath()
    {
        return \Application\Autoscuola\Handler\Data\EsamiPratica::class;
    }

    /**
     * {@inheritdoc}
     */
    public function getCalendars()
    {
        $options = [
            'conditions' => ['id = :calendar', 'calendar' => Autoscuola::CALENDAR_PRATICA],
        ];

        $calendars = [];

        foreach (Calendar::all($options) as $calendar) {
            if ($this->user->getAccount()->getRoleId() === Autoscuola::ROLE_ADMIN) {
                $calendar->is_editable = true;
            }

            $calendar->events = [];

            $calendars['calendar-' . $calendar->id] = $calendar;
        }

        return $calendars;
    }

    /**
     * {@inheritdoc}
     */
    protected function makeEvent(CalendarEvent $event, $is_editable = true)
    {
        $participants = [];
        foreach ($event->getParticipants() as $participant) {
            $p = [];
            foreach ($participant->attributes as $field => $value) {
                $p[$field] = ($value instanceof DateTime ? $value->format('Y-m-d') : $value);
            }

            // Campi supplementari
            $p['label'] = $participant->user->name(true);

            $participants[] = $p;
        }

        return [
            'id'           => $event->id,
            'author'       => $event->author_id,
            'editable'     => ($event->status === ''),
            'calendar'     => $event->calendar_id,
            'category'     => $event->category_id,
            'title'        => $event->title,
            'start'        => $event->start_date_at->format('Y-m-d H:i'),
            'end'          => $event->end_date_at->format('Y-m-d H:i'),
            'place'        => $event->place,
            'description'  => $event->content,
            'allDay'       => $event->is_all_day,
            'status'       => $event->status,
            'color'        => $event->status === '' ? $event->calendar->getColor('bg') : Autoscuola::$colors['confirmed_p']['bg'],
            'textColor'    => $event->status === '' ? $event->calendar->getColor('text') : Autoscuola::$colors['confirmed_p']['text'],
            'participants' => $participants,
        ];
    }

    /**
     * {@inheritdoc}
     */
    public static function getPopupTemplate(array $data, Container $container)
    {
        /** @var \Application\Core\Utilities\ControllerHelper $helper */
        $helper = $container->get('controller_helper');
        $path = $helper->getModuleSite()->path;

        if ($data['action'] === 'add') {
            $event = new CalendarEvent();
            $event->calendar_id = Autoscuola::CALENDAR_PRATICA;
            $event->title = 'Esame guida';
            $event->custom_ampm = null;
        } else {
            $event = CalendarEvent::find($data['event_id']);

            $hours = (int) (new DateTime($data['start']))->format('H');
            $event->custom_ampm = $hours < 12 ? 'am' : 'pm';
        }

        $ampm_select = [
            'am' => [
                'Mattina',
                'data-start' => '08:00',
                'data-end'   => '12:00',
                'data-limit' => 12,
            ],
            'pm' => [
                'Pomeriggio',
                'data-start' => '14:00',
                'data-end'   => '18:00',
                'data-limit' => 6,
            ],
        ];

        $options = [
            'conditions' => ['module_site_id = :modulesite', 'modulesite' => Autoscuola::EXAMINER_SITE_MODULE],
            'order'      => 'title ASC',
        ];

        $examiners = [];
        foreach (Node::all($options) as $node) {
            $examiners[$node->id] = $node->title;
        }

        $helper->getTheme()
            ->setTemplate('calendar/popups/esami_pratica.php')
            ->assignVars([
                'form_action'      => url("{$path}event/"),
                'autocomplete_url' => url("{$path}autocomplete/"),
                'categories_url'   => url("{$path}categories/"),
                'ampm_select'      => $ampm_select,
                'event'            => $event,
                'participants'     => self::getParticipants($event, $container),
                'deregister_url'   => url("/areaprivata/deregister/[user_id]/?t=pratica"),
                'examiners_select' => $examiners,
            ]);
    }

    /**
     * @param CalendarEvent $event
     * @param Container     $container
     * @return \Application\Calendar\Model\CalendarEventUser[]
     */
    private static function getParticipants(CalendarEvent $event, Container $container)
    {
        $participants = [];
        foreach ($event->getParticipants() as $participant) {
            $participants[$participant->user_id] = $participant;
        }

        if ($participants) {
            $user_ids = implode(', ', array_keys($participants));

            $sql = <<<SQL
SELECT
       s.user_id,
       s.license_type,
       CASE WHEN s.typology = 'pratica' THEN s.temp_license_date
            WHEN s.medical_date THEN s.medical_date
            ELSE NOW() END AS startDate,
       l.abbreviation AS location
  FROM pongho_autoscuola_students AS s
 INNER JOIN (SELECT MAX(id) AS student_id, user_id FROM pongho_autoscuola_students WHERE user_id IN ({$user_ids}) GROUP BY user_id) AS tmp ON tmp.student_id = s.id
 INNER JOIN pongho_autoscuola_locations AS l ON l.id = s.location_id
SQL;

            /** @var \Doctrine\DBAL\Connection $connection */
            $connection = $container->get('connection');

            foreach ($connection->fetchAllAssociative($sql) as $row) {
                $participants[$row['user_id']]->license_type = $row['license_type'];
                $participants[$row['user_id']]->location = $row['location'];
                $participants[$row['user_id']]->startDate = new DateTime($row['startDate']);
            }
        }

        return $participants;
    }
}
