<?php
/**
 * Questo file è parte di Pongho.
 *
 * @author  Daniele Termini
 * @package Application\Admin
 */

namespace Application\Admin\Utilities;

use Application\Core\Utilities\ControllerHelper;
use Pongho\Core\TemplateResponse;
use Pongho\Template\CompositeView;
use Pongho\Template\View;

/**
 * Class AdminControllerHelper
 */
class AdminControllerHelper extends ControllerHelper
{
    /**
     * @var CompositeView
     */
    protected $composite_view;

    /**
     * @var View
     */
    protected $header_view;

    /**
     * @var View
     */
    protected $body_view;

    /**
     * @var View
     */
    protected $footer_view;

    /**
     * @var \Application\Core\Model\Site
     */
    protected $site;

    /**
     * @var string
     */
    protected $module_site_class = 'Application\\Core\\Model\\ModuleSite';

    /**
     * {@inheritdoc}
     */
    public function displayMessage($message, $is_error = false, array $parameters = array())
    {
        if ($this->container->getService('request')->isAjax()) {
            return $this->displayJsonMessage($message, $is_error, $parameters);
        }

        /** @var \Application\Core\I18n\Translator\Translator $translator */
        $translator = $this->container->getService('translator');

        $default = array(
            'title' => $is_error ? $translator->trans('Attention') : $translator->trans('Message'),
        );

        $parameters['message'] = $message;
        $parameters['error'] = $is_error;

        $parameters = array_merge($default, $parameters);

        $this->getBodyView()
            ->setTemplatePath(__DIR__ . '/../Resources/views/message.php')
            ->assignVars($parameters);

        return new TemplateResponse($this->getView(), $this->getResponseHeaders());
    }

    /**
     * {@inheritdoc}
     *
     * @return CompositeView
     */
    public function getView()
    {
        if ($this->composite_view === null) {
            $this->composite_view = new CompositeView();
            $this->composite_view
                ->attachView($this->getHeaderView())
                ->attachView($this->getBodyView())
                ->attachView($this->getFooterView());
        }

        return $this->composite_view;
    }

    /**
     * @return \Pongho\Template\View
     */
    public function getHeaderView()
    {
        if ($this->header_view === null) {
            $this->header_view = new View(__DIR__ . '/../Resources/views/header.php');
            $this->header_view->assignVars($this->getViewGlobalVariables());
        }

        return $this->header_view;
    }

    /**
     * @return \Pongho\Template\View
     */
    public function getBodyView()
    {
        if ($this->body_view === null) {
            $this->body_view = new View(__DIR__ . '/../Resources/views/default.php');
            $this->body_view->assignVars($this->getViewGlobalVariables());
        }

        return $this->body_view;
    }

    /**
     * @return \Pongho\Template\View
     */
    public function getFooterView()
    {
        if ($this->footer_view === null) {
            $this->footer_view = new View(__DIR__ . '/../Resources/views/footer.php');
            $this->footer_view->assignVars($this->getViewGlobalVariables());
        }

        return $this->footer_view;
    }

    /**
     * Restituiesce le variabili globali per la vista
     */
    public function getViewGlobalVariables()
    {
        return array(
            'head'       => $this->getHead(),
            'html'       => $this->container->getService('template_html'),
            'translator' => $this->container->getService('translator'),
            'lang'       => $this->container->getService('language'),
            'language'   => $this->getLanguage(),
            'site'       => $this->getSite(),
            'user'       => $this->getUser(),
        );
    }

    /**
     * {@inheritdoc}
     */
    public function getSite()
    {
        if ($this->site === null) {
            if ($this->container->getService('request')->query->has('site') || $this->container->hasParameter('site_id')) {
                $this->site = $this->container->getService('site');
            } else {
                $this->site = $this->container->getService('main_site');
            }
        }

        return $this->site;
    }

    /**
     * @return int
     */
    public function getModuleId()
    {
        return $this->container->getParameter('module_id');
    }

    /**
     * Imposta una classe per il ModuleSite
     *
     * @param string $module_site_class
     */
    public function setModuleSiteClass($module_site_class)
    {
        $this->module_site_class = $module_site_class;
    }

    /**
     * @return string|\Application\Core\Model\ModuleSite
     */
    public function getModuleSiteClass()
    {
        return $this->module_site_class;
    }

    /**
     * Restituisce il modello ModuleSite corrente.
     *
     * @param boolean $throw_exception
     * @return \Application\Core\Model\ModuleSite
     * @throws \UnexpectedValueException Quando il modulo amministrativo non è associato ad alcun modulo.
     */
    public function getModuleSite($throw_exception = true)
    {
        if ($this->module_site === null) {
            $module_id = $this->getModuleId();

            if ($module_id === null) {
                if ($throw_exception) {
                    throw new \UnexpectedValueException('L\'item corrente del menu ha un "module_id" non valido.');
                }

                return null;
            } else {
                $site = $this->getSite();
                $language = $this->getLanguage();
                $class = $this->getModuleSiteClass();

                $this->module_site = $class::findById($module_id, $site->id, $language->language_id);
            }
        }

        return $this->module_site;
    }

    /**
     * Restituisce la lingua per l'interfaccia
     * @return int
     */
    public function getContentLanguageId()
    {
        /** @var \Application\Admin\Utilities\ContentLanguage $content_language */
        $content_language = $this->container->getService('content_language');

        return $content_language->getId();
    }
}
