<?php
/**
 * Questo file è parte di Pongho.
 *
 * @author Daniele Termini
 * @package Application\Admin
 */

namespace Application\Admin\Tests\Utilities;


use Application\Admin\Utilities\FilterConverter;

class FilterConverterTest extends \PHPUnit_Framework_TestCase
{
    /**
     * @var FilterConverter
     */
    protected $filter_composer;

    /**
     * {@inheritdoc}
     */
    public function setup()
    {
        $this->filter_composer = new FilterConverter();
    }

    /**
     * @return array
     */
    public function conditionsProvider()
    {
        return array(
            '= 0 (numero)'                                  => array(
                array('EQ' => 0),
                array('`col` = :col', 'col' => 0)
            ),
            '= 1 (numero)'                                  => array(
                array('EQ' => 1),
                array('`col` = :col', 'col' => 1)
            ),
            '= 0 (stringa)'                                 => array(
                array('EQ' => '0'),
                array('`col` = :col', 'col' => 0)
            ),
            '= 1 (stringa)'                                 => array(
                array('EQ' => '1'),
                array('`col` = :col', 'col' => 1)
            ),
            '= true'                                        => array(
                array('EQ' => true),
                array('`col` = :col', 'col' => true)
            ),
            '= false'                                       => array(
                array('EQ' => false),
                array('`col` = :col', 'col' => false)
            ),
            '= stringa vuota / null'                        => array(
                array('EQ' => ''),
                array('(`col` = :col OR `col` IS NULL)', 'col' => '')
            ),
            '= campo in join'                               => array(
                array('EQ' => 'bar'),
                array('foo.`col` = :col', 'col' => 'bar'),
                'foo.`col`'
            ),
            'LIKE'                                          => array(
                array('LIKE' => 'val'),
                array('pongho_like(`col`, :col)', 'col' => '%val%')
            ),
            'START'                                         => array(
                array('START' => 'val'),
                array('pongho_like(`col`, :col)', 'col' => 'val%')
            ),
            'END'                                           => array(
                array('END' => 'val'),
                array('pongho_like(`col`, :col)', 'col' => '%val')
            ),
            'IN'                                            => array(
                array('IN' => array(1, 2, 3)),
                array("`col` IN ('1', '2', '3')")
            ),
            'NOTIN'                                         => array(
                array('NOTIN' => array(1, 2, 3)),
                array("`col` NOT IN ('1', '2', '3')")
            ),
            '<'                                             => array(
                array('LT' => 0),
                array('`col` < :col', 'col' => 0)
            ),
            '<='                                            => array(
                array('LTE' => 0),
                array('`col` <= :col', 'col' => 0)
            ),
            '>'                                             => array(
                array('GT' => 0),
                array('`col` > :col', 'col' => 0)
            ),
            '>='                                            => array(
                array('GTE' => 0),
                array('`col` >= :col', 'col' => 0)
            ),
            '<>'                                            => array(
                array('NOT' => 0),
                array('`col` <> :col', 'col' => 0)
            ),
            'NULL'                                          => array(
                array('NULL' => 0),
                array('`col` IS NULL')
            ),
            'NOTNULL'                                       => array(
                array('NOTNULL' => 0),
                array('`col` IS NOT NULL')
            ),
            'BETWEEN'                                       => array(
                array('BETWEEN' => json_encode(array(1, 10))),
                array('`col` BETWEEN :col_l AND :col_r', 'col_l' => 1, 'col_r' => 10)
            ),
            'DATE = mezzanotte'                             => array(
                array('DATEEQ' => '2014-01-31 00:00:00'),
                array('`col` = :col', 'col' => '2014-01-31 00:00:00')
            ),
            'DATE = ora esatta'                             => array(
                array('DATEEQ' => '2014-01-31 12:34:56'),
                array('`col` = :col', 'col' => '2014-01-31 12:34:56')
            ),
            'DATE = tutto il giorno'                        => array(
                array('DATEEQ' => '2014-01-31'),
                array(
                    '`col` BETWEEN :col_l AND :col_r',
                    'col_l' => '2014-01-31 00:00:00',
                    'col_r' => '2014-01-31 23:59:59'
                )
            ),
            'DATEBETWEEN intervallo in giorni'              => array(
                array('DATEBETWEEN' => array('2014-01-31', '2014-02-28')),
                array(
                    '`col` BETWEEN :col_l AND :col_r',
                    'col_l' => '2014-01-31 00:00:00',
                    'col_r' => '2014-02-28 23:59:59'
                )
            ),
            'DATEBETWEEN intervallo esatto mezzanotte'      => array(
                array(
                    'DATEBETWEEN' => array(
                        '2014-01-31 00:00:00',
                        '2014-02-28 00:00:00'
                    )
                ),
                array(
                    '`col` BETWEEN :col_l AND :col_r',
                    'col_l' => '2014-01-31 00:00:00',
                    'col_r' => '2014-02-28 00:00:00'
                )
            ),
            'DATEBETWEEN intervallo esatto'                 => array(
                array(
                    'DATEBETWEEN' => array(
                        '2014-01-31 12:34:56',
                        '2014-02-28 12:34:56'
                    )
                ),
                array(
                    '`col` BETWEEN :col_l AND :col_r',
                    'col_l' => '2014-01-31 12:34:56',
                    'col_r' => '2014-02-28 12:34:56'
                )
            ),
            'DATEBETWEEN intervallo misto senza ora fine'   => array(
                array(
                    'DATEBETWEEN' => array(
                        '2014-01-31 12:34:56',
                        '2014-02-28'
                    )
                ),
                array(
                    '`col` BETWEEN :col_l AND :col_r',
                    'col_l' => '2014-01-31 12:34:56',
                    'col_r' => '2014-02-28 23:59:59'
                )
            ),
            'DATEBETWEEN intervallo misto senza ora inizio' => array(
                array(
                    'DATEBETWEEN' => array(
                        '2014-01-31',
                        '2014-02-28 12:34:56'
                    )
                ),
                array(
                    '`col` BETWEEN :col_l AND :col_r',
                    'col_l' => '2014-01-31 00:00:00',
                    'col_r' => '2014-02-28 12:34:56'
                )
            ),
        );
    }

    /**
     * @dataProvider conditionsProvider
     * @param        $data
     * @param array  $expected_condition
     * @param string $column_alias
     */
    public function testPrepareCondition($data, array $expected_condition, $column_alias = null)
    {
        $condition = $this->filter_composer->toCondition('col', $data, $column_alias);

        $this->assertEquals($expected_condition, $condition);
    }
}
